<?php

namespace WPZEO\Modules;

use WPZEO\Core\Settings;

if (! defined('ABSPATH')) {
	exit;
}

class SettingsApi
{
	/**
	 * @var Settings
	 */
	private $settings;

	/**
	 * @param Settings $settings
	 */
	public function __construct(Settings $settings)
	{
		$this->settings = $settings;
		add_action('rest_api_init', [$this, 'register_rest_routes']);
	}

	public function register_rest_routes()
	{
		register_rest_route(
			'wpzeo/v1',
			'/modules',
			[
				[
					'methods'             => 'GET',
					'callback'            => [$this, 'get_modules'],
					'permission_callback' => [$this, 'can_manage_settings'],
				],
				[
					'methods'             => 'POST',
					'callback'            => [$this, 'update_modules'],
					'permission_callback' => [$this, 'can_manage_settings'],
				],
			]
		);

		register_rest_route(
			'wpzeo/v1',
			'/general-settings',
			[
				[
					'methods'             => 'GET',
					'callback'            => [$this, 'get_general_settings'],
					'permission_callback' => [$this, 'can_manage_settings'],
				],
				[
					'methods'             => 'POST',
					'callback'            => [$this, 'update_general_settings'],
					'permission_callback' => [$this, 'can_manage_settings'],
				],
			]
		);

		register_rest_route(
			'wpzeo/v1',
			'/capabilities',
			[
				[
					'methods'             => 'GET',
					'callback'            => [$this, 'get_capabilities'],
					'permission_callback' => [$this, 'can_manage_settings'],
				],
			]
		);
	}

	/**
	 * @return bool
	 */
	public function can_manage_settings()
	{
		return current_user_can('manage_options');
	}

	/**
	 * @return \WP_REST_Response
	 */
	public function get_modules()
	{
		return new \WP_REST_Response(
			[
				'modules' => $this->settings->get_modules(),
			],
			200
		);
	}

	/**
	 * Payload keys:
	 * meta, sitemap, robots, social, redirects
	 *
	 * @param \WP_REST_Request $request
	 * @return \WP_REST_Response
	 */
	public function update_modules($request)
	{
		$current = $this->settings->get_modules();
		$input   = $request->get_json_params();
		$input   = is_array($input) ? $input : [];

		$payload = [
			'meta'      => array_key_exists('meta', $input) ? (int) ! empty($input['meta']) : (int) ! empty($current['meta']),
			'sitemap'   => array_key_exists('sitemap', $input) ? (int) ! empty($input['sitemap']) : (int) ! empty($current['sitemap']),
			'robots'    => array_key_exists('robots', $input) ? (int) ! empty($input['robots']) : (int) ! empty($current['robots']),
			'social'    => array_key_exists('social', $input) ? (int) ! empty($input['social']) : (int) ! empty($current['social']),
			'redirects' => array_key_exists('redirects', $input) ? (int) ! empty($input['redirects']) : (int) ! empty($current['redirects']),
		];

		update_option(Settings::OPTION_MODULES, $payload, false);

		return new \WP_REST_Response(
			[
				'modules' => $this->settings->get_modules(),
			],
			200
		);
	}

	/**
	 * @return \WP_REST_Response
	 */
	public function get_general_settings()
	{
		return new \WP_REST_Response(
			[
				'settings' => $this->settings->get_general(),
			],
			200
		);
	}

	/**
	 * Payload keys:
	 * default_robots, noindex_search,
	 * title_template_post_types, description_template_post_types,
	 * title_template_taxonomies, description_template_taxonomies,
	 * title_template_author, description_template_author
	 *
	 * @param \WP_REST_Request $request
	 * @return \WP_REST_Response
	 */
	public function update_general_settings($request)
	{
		$current = $this->settings->get_general();
		$input   = $request->get_json_params();
		$input   = is_array($input) ? $input : [];

		$default_robots = array_key_exists('default_robots', $input)
			? sanitize_text_field((string) $input['default_robots'])
			: (string) $current['default_robots'];

		if (! in_array($default_robots, ['index,follow', 'noindex,follow', 'index,nofollow', 'noindex,nofollow'], true)) {
			$default_robots = 'index,follow';
		}

		$noindex_search = array_key_exists('noindex_search', $input)
			? (int) ! empty($input['noindex_search'])
			: (int) ! empty($current['noindex_search']);

		$public_post_types = get_post_types(['public' => true], 'names');
		$public_taxonomies = get_taxonomies(['public' => true], 'names');

		$title_template_post_types = array_key_exists('title_template_post_types', $input)
			? $this->sanitize_template_map($input['title_template_post_types'], $public_post_types)
			: (isset($current['title_template_post_types']) && is_array($current['title_template_post_types']) ? $this->sanitize_template_map($current['title_template_post_types'], $public_post_types) : []);
		$description_template_post_types = array_key_exists('description_template_post_types', $input)
			? $this->sanitize_template_map($input['description_template_post_types'], $public_post_types)
			: (isset($current['description_template_post_types']) && is_array($current['description_template_post_types']) ? $this->sanitize_template_map($current['description_template_post_types'], $public_post_types) : []);
		$title_template_taxonomies = array_key_exists('title_template_taxonomies', $input)
			? $this->sanitize_template_map($input['title_template_taxonomies'], $public_taxonomies)
			: (isset($current['title_template_taxonomies']) && is_array($current['title_template_taxonomies']) ? $this->sanitize_template_map($current['title_template_taxonomies'], $public_taxonomies) : []);
		$description_template_taxonomies = array_key_exists('description_template_taxonomies', $input)
			? $this->sanitize_template_map($input['description_template_taxonomies'], $public_taxonomies)
			: (isset($current['description_template_taxonomies']) && is_array($current['description_template_taxonomies']) ? $this->sanitize_template_map($current['description_template_taxonomies'], $public_taxonomies) : []);

		$title_template_author = array_key_exists('title_template_author', $input)
			? sanitize_text_field((string) $input['title_template_author'])
			: (isset($current['title_template_author']) ? sanitize_text_field((string) $current['title_template_author']) : '');
		$description_template_author = array_key_exists('description_template_author', $input)
			? sanitize_text_field((string) $input['description_template_author'])
			: (isset($current['description_template_author']) ? sanitize_text_field((string) $current['description_template_author']) : '');

		$payload = [
			'default_robots' => $default_robots,
			'noindex_search' => $noindex_search,
			'title_template_post_types' => $title_template_post_types,
			'description_template_post_types' => $description_template_post_types,
			'title_template_taxonomies' => $title_template_taxonomies,
			'description_template_taxonomies' => $description_template_taxonomies,
			'title_template_author' => $title_template_author,
			'description_template_author' => $description_template_author,
		];

		update_option(Settings::OPTION_GENERAL, $payload, false);

		return new \WP_REST_Response(
			[
				'settings' => $this->settings->get_general(),
			],
			200
		);
	}

	/**
	 * @param mixed $templates
	 * @param array<string, string> $allowed_keys
	 * @return array<string, string>
	 */
	private function sanitize_template_map($templates, $allowed_keys)
	{
		$templates = is_array($templates) ? $templates : [];
		$sanitized = [];

		foreach ($templates as $key => $template) {
			$key = sanitize_key((string) $key);
			if (! isset($allowed_keys[$key])) {
				continue;
			}
			$value = sanitize_text_field((string) $template);
			if ('' === trim($value)) {
				continue;
			}
			$sanitized[$key] = $value;
		}

		return $sanitized;
	}

	/**
	 * @return \WP_REST_Response
	 */
	public function get_capabilities()
	{
		$base = rest_url('wpzeo/v1');
		return new \WP_REST_Response(
			[
				'plugin'       => 'wpZeo - Zero SEO',
				'version'      => defined('WPZEO_VERSION') ? (string) WPZEO_VERSION : '',
				'namespace'    => 'wpzeo/v1',
				'base_url'     => $base,
				'auth'         => [
					'required_capability' => 'manage_options',
				],
				'endpoints'    => [
					['path' => '/modules', 'methods' => ['GET', 'POST']],
					['path' => '/general-settings', 'methods' => ['GET', 'POST']],
					['path' => '/meta/{post|term|author}/{id}', 'methods' => ['GET', 'POST']],
					['path' => '/sitemap-settings', 'methods' => ['GET', 'POST']],
					['path' => '/social-settings', 'methods' => ['GET', 'POST']],
					['path' => '/robots-settings', 'methods' => ['GET', 'POST']],
					['path' => '/redirects', 'methods' => ['GET', 'POST']],
					['path' => '/redirects-export', 'methods' => ['GET']],
					['path' => '/redirects-import', 'methods' => ['POST']],
					['path' => '/migration-status', 'methods' => ['GET']],
					['path' => '/migration-mode', 'methods' => ['GET', 'POST']],
					['path' => '/migration-report', 'methods' => ['GET', 'DELETE']],
					['path' => '/migration-import', 'methods' => ['POST']],
					['path' => '/capabilities', 'methods' => ['GET']],
				],
				'notes'        => [
					'WordPress core content endpoints remain under /wp-json/wp/v2/*.',
				],
			],
			200
		);
	}
}
