<?php

namespace WPZEO\Modules;

use WPZEO\Core\Settings;

if (! defined('ABSPATH')) {
	exit;
}

class Redirects
{
	/**
	 * @var Settings
	 */
	private $settings;

	/**
	 * @param Settings $settings
	 */
	public function __construct(Settings $settings)
	{
		$this->settings = $settings;
		add_action('rest_api_init', [$this, 'register_rest_routes']);
		add_action('template_redirect', [$this, 'maybe_redirect'], 0);
	}

	public function register_rest_routes()
	{
		register_rest_route(
			'wpzeo/v1',
			'/redirects',
			[
				[
					'methods'             => 'GET',
					'callback'            => [$this, 'get_redirects'],
					'permission_callback' => [$this, 'can_manage_settings'],
				],
				[
					'methods'             => 'POST',
					'callback'            => [$this, 'update_redirects'],
					'permission_callback' => [$this, 'can_manage_settings'],
				],
			]
			);

		register_rest_route(
			'wpzeo/v1',
			'/redirects-export',
			[
				[
					'methods'             => 'GET',
					'callback'            => [$this, 'export_redirects_csv'],
					'permission_callback' => [$this, 'can_manage_settings'],
				],
			]
		);

		register_rest_route(
			'wpzeo/v1',
			'/redirects-import',
			[
				[
					'methods'             => 'POST',
					'callback'            => [$this, 'import_redirects_csv'],
					'permission_callback' => [$this, 'can_manage_settings'],
				],
			]
		);
	}

	/**
	 * @return bool
	 */
	public function can_manage_settings()
	{
		return current_user_can('manage_options');
	}

	/**
	 * @return \WP_REST_Response
	 */
	public function get_redirects()
	{
		return new \WP_REST_Response(
			[
				'rules' => $this->settings->get_redirect_rules(),
			],
			200
		);
	}

	/**
	 * Payload:
	 * {
	 *   "rules": [
	 *     {"source":"/old","target":"https://example.com/new","code":"301","type":"exact"}
	 *   ],
	 *   "reset_stats": false
	 * }
	 *
	 * @param \WP_REST_Request $request
	 * @return \WP_REST_Response
	 */
	public function update_redirects($request)
	{
		$rules = $request->get_param('rules');
		if (! is_array($rules)) {
			return new \WP_REST_Response(
				[
					'error'   => 'invalid_payload',
					'message' => 'Expected "rules" array.',
				],
				400
			);
		}

		$current_stats = $this->settings->get_redirect_stats();
		$reset_stats = (bool) $request->get_param('reset_stats');
		$prepared = $this->prepare_rules_for_save($rules, $current_stats, $reset_stats);
		$sanitized = $prepared['rules'];
		$rejected  = $prepared['rejected'];
		$next_stats = $prepared['stats'];

		update_option(Settings::OPTION_REDIRECTS, ['rules' => $sanitized], false);
		update_option(Settings::OPTION_REDIRECT_STATS, $next_stats, false);
		$fresh_rules = $this->settings->get_redirect_rules();

		return new \WP_REST_Response(
			[
				'rules'          => $fresh_rules,
				'count'          => count($fresh_rules),
				'rejected'       => $rejected,
				'rejected_count' => count($rejected),
			],
				200
		);
	}

	/**
	 * @return \WP_REST_Response
	 */
	public function export_redirects_csv()
	{
		$rules = $this->settings->get_redirect_rules();
			$lines = ['source,target,code,type,hits,last_hit'];
			foreach ($rules as $rule) {
				$fields = [
					(string) $rule['source'],
					(string) $rule['target'],
					(string) $rule['code'],
					(string) (isset($rule['type']) ? $rule['type'] : 'exact'),
					(string) absint(isset($rule['hits']) ? $rule['hits'] : 0),
					(string) (isset($rule['last_hit']) ? $rule['last_hit'] : ''),
				];
			$escaped = array_map([$this, 'escape_csv_field'], $fields);
			$lines[] = implode(',', $escaped);
		}

		return new \WP_REST_Response(
			[
				'csv' => implode("\n", $lines) . "\n",
				'count' => count($rules),
			],
			200
		);
	}

	/**
	 * Payload:
	 * {
	 *   "csv": "source,target,code,type\n/old,/new,301,exact",
	 *   "replace": true,
	 *   "reset_stats": false
	 * }
	 *
	 * @param \WP_REST_Request $request
	 * @return \WP_REST_Response
	 */
	public function import_redirects_csv($request)
	{
		$csv = (string) $request->get_param('csv');
		if ('' === trim($csv)) {
			return new \WP_REST_Response(
				[
					'error'   => 'invalid_payload',
					'message' => 'Expected non-empty "csv" string.',
				],
				400
			);
		}

		$replace = $request->has_param('replace') ? (bool) $request->get_param('replace') : true;
		$reset_stats = (bool) $request->get_param('reset_stats');
		$rows = preg_split('/\r\n|\r|\n/', $csv);
		$rows = is_array($rows) ? $rows : [];

		$parsed_rules = [];
		$rejected_rows = [];
		foreach ($rows as $index => $row) {
			$row = trim((string) $row);
			if ('' === $row) {
				continue;
			}

			$columns = str_getcsv($row);
				if (! is_array($columns) || count($columns) < 2) {
					$rejected_rows[] = [
						'line'   => $index + 1,
						'row'    => $row,
						'reason' => 'Expected source,target[,code[,type]] columns.',
					];
					continue;
				}

				$source = trim((string) $columns[0]);
				$target = trim((string) $columns[1]);
				$code   = isset($columns[2]) ? trim((string) $columns[2]) : '301';
				$type   = isset($columns[3]) ? trim((string) $columns[3]) : 'exact';

			if (1 === $index && strtolower($source) === 'source' && strtolower($target) === 'target') {
				continue;
			}
			if (0 === $index && strtolower($source) === 'source' && strtolower($target) === 'target') {
				continue;
			}

				$parsed_rules[] = [
					'source' => $source,
					'target' => $target,
					'code'   => $code,
					'type'   => $type,
				];
			}

		$current_stats = $this->settings->get_redirect_stats();
		$base_rules = [];
		if (! $replace) {
			$base_rules = $this->settings->get_redirect_rules();
		}
		$merge_rules = array_merge($base_rules, $parsed_rules);
		$prepared = $this->prepare_rules_for_save($merge_rules, $current_stats, $reset_stats);
		$sanitized = $prepared['rules'];
		$rejected  = array_merge($prepared['rejected'], $rejected_rows);
		$next_stats = $prepared['stats'];

		update_option(Settings::OPTION_REDIRECTS, ['rules' => $sanitized], false);
		update_option(Settings::OPTION_REDIRECT_STATS, $next_stats, false);
		$fresh_rules = $this->settings->get_redirect_rules();

		return new \WP_REST_Response(
			[
				'rules'          => $fresh_rules,
				'count'          => count($fresh_rules),
				'rejected'       => $rejected,
				'rejected_count' => count($rejected),
			],
			200
		);
	}

	public function maybe_redirect()
	{
		if (is_admin() || wp_doing_ajax() || wp_doing_cron()) {
			return;
		}

		$request_uri = isset($_SERVER['REQUEST_URI']) ? (string) wp_unslash($_SERVER['REQUEST_URI']) : '';
		if ('' === $request_uri) {
			return;
		}

		$path = wp_parse_url($request_uri, PHP_URL_PATH);
		$path = is_string($path) ? '/' . ltrim($path, '/') : '';
		if ('' === $path) {
			return;
		}

		$rules = $this->settings->get_redirect_rules();
		foreach ($rules as $rule) {
			$type   = isset($rule['type']) ? (string) $rule['type'] : 'exact';
			$source = (string) $rule['source'];
			$target = (string) $rule['target'];
			$code   = (int) $rule['code'];

			if ('' === $target) {
				continue;
			}

			if ('regex' === $type) {
				$resolved_target = $this->resolve_regex_target($source, $target, $path);
				if ('' === $resolved_target) {
					continue;
				}
				$this->record_redirect_hit($source, $target, (string) $code, 'regex');
				wp_safe_redirect($resolved_target, $code);
				exit;
			}

			$normalized_source = '/' . ltrim($source, '/');
			$path_base   = untrailingslashit($path);
			$source_base = untrailingslashit($normalized_source);
			if ($path !== $normalized_source && $path_base !== $source_base) {
				continue;
			}

			$this->record_redirect_hit($normalized_source, $target, (string) $code, 'exact');
			wp_safe_redirect($target, $code);
			exit;
		}
	}

	/**
	 * @param string $source
	 * @param string $target
	 * @param string $code
	 * @param string $type
	 */
	private function record_redirect_hit($source, $target, $code, $type)
	{
		$stats = $this->settings->get_redirect_stats();
		$key = $this->get_rule_key($source, $target, $code, $type);
		$entry = isset($stats[$key]) && is_array($stats[$key]) ? $stats[$key] : [];
		$stats[$key] = [
			'hits' => absint(isset($entry['hits']) ? $entry['hits'] : 0) + 1,
			'last_hit' => gmdate('c'),
		];
		update_option(Settings::OPTION_REDIRECT_STATS, $stats, false);
	}

	/**
	 * @param string $source
	 * @param string $target
	 * @param string $code
	 * @param string $type
	 * @return string
	 */
	private function get_rule_key($source, $target, $code, $type)
	{
		return sha1($source . '|' . $target . '|' . $code . '|' . $type);
	}

	/**
	 * @param array<int, array<string, mixed>> $rules
	 * @param array<string, array<string, mixed>> $current_stats
	 * @param bool $reset_stats
	 * @return array<string, mixed>
	 */
	private function prepare_rules_for_save($rules, $current_stats, $reset_stats)
	{
		$sanitized = [];
		$rejected  = [];
		$next_stats = [];
		$seen = [];

		foreach ($rules as $rule) {
			if (! is_array($rule)) {
				$rejected[] = [
					'rule'   => null,
					'reason' => 'Rule must be an object.',
				];
				continue;
			}

			$source = isset($rule['source']) ? trim((string) $rule['source']) : '';
			$target = isset($rule['target']) ? trim((string) $rule['target']) : '';
			$code   = isset($rule['code']) ? (string) $rule['code'] : '301';
			$type   = isset($rule['type']) ? sanitize_key((string) $rule['type']) : 'exact';

			if ('' === $source || '' === $target) {
				$rejected[] = [
					'rule'   => $rule,
					'reason' => 'Source and target are required.',
				];
				continue;
			}

			if (! in_array($type, ['exact', 'regex'], true)) {
				$type = 'exact';
			}

			if ('regex' !== $type) {
				if (0 !== strpos($source, '/')) {
					$source = '/' . $source;
				}
			} elseif (! $this->is_valid_regex_pattern($source)) {
				$rejected[] = [
					'rule'   => $rule,
					'reason' => 'Regex source must be a valid PCRE pattern with delimiters.',
				];
				continue;
			}

			if (0 === strpos($target, '/')) {
				$target = home_url($target);
			} else {
				$parsed = wp_parse_url($target);
				if (
					! is_array($parsed) ||
					empty($parsed['scheme']) ||
					empty($parsed['host']) ||
					! in_array(strtolower((string) $parsed['scheme']), ['http', 'https'], true)
				) {
					$rejected[] = [
						'rule'   => $rule,
						'reason' => 'Target must be a relative path or absolute http/https URL.',
					];
					continue;
				}

				$target = esc_url_raw($target);
			}

			if ('' === $target) {
				$rejected[] = [
					'rule'   => $rule,
					'reason' => 'Target URL is invalid.',
				];
				continue;
			}

			if (! in_array($code, ['301', '302'], true)) {
				$code = '301';
			}

			$rule_key = $this->get_rule_key($source, $target, $code, $type);
			if (isset($seen[$rule_key])) {
				continue;
			}
			$seen[$rule_key] = true;

			$sanitized[] = [
				'source' => $source,
				'target' => $target,
				'code'   => $code,
				'type'   => $type,
			];

			$existing = isset($current_stats[$rule_key]) && is_array($current_stats[$rule_key]) ? $current_stats[$rule_key] : [];
			$next_stats[$rule_key] = [
				'hits' => $reset_stats ? 0 : absint(isset($existing['hits']) ? $existing['hits'] : 0),
				'last_hit' => $reset_stats ? '' : (string) (isset($existing['last_hit']) ? $existing['last_hit'] : ''),
			];
		}

		return [
			'rules'    => $sanitized,
			'rejected' => $rejected,
			'stats'    => $next_stats,
		];
	}

	/**
	 * @param string $pattern
	 * @return bool
	 */
	private function is_valid_regex_pattern($pattern)
	{
		$result = @preg_match($pattern, '');
		return false !== $result;
	}

	/**
	 * @param string $pattern
	 * @param string $target
	 * @param string $path
	 * @return string
	 */
	private function resolve_regex_target($pattern, $target, $path)
	{
		if (! $this->is_valid_regex_pattern($pattern)) {
			return '';
		}

		$replaced = @preg_replace($pattern, $target, $path, 1);
		if (! is_string($replaced) || $replaced === $path) {
			return '';
		}

		if (0 === strpos($replaced, '/')) {
			return (string) home_url($replaced);
		}

		$url = esc_url_raw($replaced);
		return is_string($url) ? $url : '';
	}

	/**
	 * @param string $value
	 * @return string
	 */
	private function escape_csv_field($value)
	{
		$value = str_replace('"', '""', (string) $value);
		if (false !== strpos($value, ',') || false !== strpos($value, '"') || false !== strpos($value, "\n")) {
			return '"' . $value . '"';
		}
		return $value;
	}
}
