<?php

namespace WPZEO\Modules;

use WPZEO\Core\Settings;

if (! defined('ABSPATH')) {
	exit;
}

class Migration
{
	/**
	 * @var Settings
	 */
	private $settings;

	/**
	 * @param Settings $settings
	 */
	public function __construct(Settings $settings)
	{
		$this->settings = $settings;
		add_action('rest_api_init', [$this, 'register_rest_routes']);
	}

	public function register_rest_routes()
	{
		register_rest_route(
			'wpzeo/v1',
			'/migration-status',
			[
				[
					'methods'             => 'GET',
					'callback'            => [$this, 'get_migration_status'],
					'permission_callback' => [$this, 'can_manage_settings'],
				],
			]
		);

		register_rest_route(
			'wpzeo/v1',
			'/migration-mode',
			[
				[
					'methods'             => 'GET',
					'callback'            => [$this, 'get_migration_mode'],
					'permission_callback' => [$this, 'can_manage_settings'],
				],
				[
					'methods'             => 'POST',
					'callback'            => [$this, 'update_migration_mode'],
					'permission_callback' => [$this, 'can_manage_settings'],
				],
			]
		);

		register_rest_route(
			'wpzeo/v1',
			'/migration-report',
			[
				[
					'methods'             => 'GET',
					'callback'            => [$this, 'get_last_migration_report'],
					'permission_callback' => [$this, 'can_manage_settings'],
				],
				[
					'methods'             => 'DELETE',
					'callback'            => [$this, 'clear_last_migration_report'],
					'permission_callback' => [$this, 'can_manage_settings'],
				],
			]
		);

		register_rest_route(
			'wpzeo/v1',
			'/migration-import',
			[
				[
					'methods'             => 'POST',
					'callback'            => [$this, 'run_migration_import'],
					'permission_callback' => [$this, 'can_manage_settings'],
				],
			]
		);
	}

	/**
	 * @return bool
	 */
	public function can_manage_settings()
	{
		return current_user_can('manage_options');
	}

	/**
	 * @return \WP_REST_Response
	 */
	public function get_migration_status()
	{
		$detected = self::detect_supported_plugins();
		$legacy   = $this->get_legacy_data_status();
		$config   = $this->settings->get_migration();
		$effective = $this->is_effective_migration_mode($config, $detected);
		$last_report = get_option(Settings::OPTION_LAST_MIGRATION_REPORT, []);

		return new \WP_REST_Response(
			[
				'mode' => [
					'enabled'   => ! empty($config['enabled']),
					'auto_enable' => ! empty($config['auto_enable']),
					'effective' => $effective,
				],
				'detected_plugins' => array_values($detected),
				'legacy_data'      => $legacy,
				'recommended'      => $this->has_detection_or_data($detected, $legacy),
				'api'              => [
					'status' => 'ok',
					'namespace' => 'wpzeo/v1',
				],
				'last_report'      => is_array($last_report) ? $last_report : [],
			],
			200
		);
	}

	/**
	 * @return \WP_REST_Response
	 */
	public function get_migration_mode()
	{
		$config = $this->settings->get_migration();
		$detected = self::detect_supported_plugins();
		$effective = $this->is_effective_migration_mode($config, $detected);

		return new \WP_REST_Response(
			[
				'enabled' => ! empty($config['enabled']),
				'auto_enable' => ! empty($config['auto_enable']),
				'effective' => $effective,
			],
			200
		);
	}

	/**
	 * Payload keys: enabled, auto_enable
	 *
	 * @param \WP_REST_Request $request
	 * @return \WP_REST_Response
	 */
	public function update_migration_mode($request)
	{
		$current = $this->settings->get_migration();
		$input   = $request->get_json_params();
		$input   = is_array($input) ? $input : [];

		$payload = [
			'enabled'     => array_key_exists('enabled', $input) ? (int) ! empty($input['enabled']) : (int) ! empty($current['enabled']),
			'auto_enable' => array_key_exists('auto_enable', $input) ? (int) ! empty($input['auto_enable']) : (int) ! empty($current['auto_enable']),
		];
		update_option(
			Settings::OPTION_MIGRATION,
			array_merge(is_array($current) ? $current : [], $payload),
			false
		);

		return $this->get_migration_mode();
	}

	/**
	 * @return \WP_REST_Response
	 */
	public function get_last_migration_report()
	{
		$report = get_option(Settings::OPTION_LAST_MIGRATION_REPORT, []);
		return new \WP_REST_Response(
			[
				'last_report' => is_array($report) ? $report : [],
			],
			200
		);
	}

	/**
	 * @return \WP_REST_Response
	 */
	public function clear_last_migration_report()
	{
		update_option(Settings::OPTION_LAST_MIGRATION_REPORT, [], false);
		return new \WP_REST_Response(
			[
				'last_report' => [],
				'cleared' => true,
			],
			200
		);
	}

	/**
	 * Payload:
	 * source: auto|yoast|rank_math
	 * include_posts: bool (default true)
	 * include_terms: bool (default true)
	 * include_authors: bool (default true)
	 * include_redirects: bool (default true)
	 * overwrite: bool (default false)
	 * dry_run: bool (default false)
	 * limit: int (default 2000, max 10000)
	 *
	 * @param \WP_REST_Request $request
	 * @return \WP_REST_Response
	 */
	public function run_migration_import($request)
	{
		$source = sanitize_key((string) $request->get_param('source'));
		if ('' === $source) {
			$source = 'auto';
		}
		if (! in_array($source, ['auto', 'yoast', 'rank_math'], true)) {
			return new \WP_REST_Response(
				[
					'error' => 'invalid_source',
					'message' => 'source must be one of: auto, yoast, rank_math',
				],
				400
			);
		}

		$include_posts   = $request->has_param('include_posts') ? (bool) $request->get_param('include_posts') : true;
		$include_terms   = $request->has_param('include_terms') ? (bool) $request->get_param('include_terms') : true;
		$include_authors = $request->has_param('include_authors') ? (bool) $request->get_param('include_authors') : true;
		$include_redirects = $request->has_param('include_redirects') ? (bool) $request->get_param('include_redirects') : true;
		$overwrite       = (bool) $request->get_param('overwrite');
		$redirect_conflict = sanitize_key((string) $request->get_param('redirect_conflict'));
		$dry_run         = (bool) $request->get_param('dry_run');
		$limit           = absint($request->get_param('limit'));
		if ($limit <= 0) {
			$limit = 2000;
		}
		if ($limit > 10000) {
			$limit = 10000;
		}
		if (! in_array($redirect_conflict, ['skip', 'overwrite', 'append'], true)) {
			$redirect_conflict = $overwrite ? 'overwrite' : 'skip';
		}

		$resolved_source = $this->resolve_source($source);
		if ('' === $resolved_source) {
			return new \WP_REST_Response(
				[
					'error' => 'no_source_data',
					'message' => 'No supported source data found for migration.',
				],
				404
			);
		}

		$result = [
			'source' => $resolved_source,
			'dry_run' => $dry_run,
			'overwrite' => $overwrite,
			'redirect_conflict' => $redirect_conflict,
			'run_at' => gmdate('c'),
			'counts' => [
				'posts' => ['scanned' => 0, 'changed' => 0],
				'terms' => ['scanned' => 0, 'changed' => 0],
				'authors' => ['scanned' => 0, 'changed' => 0],
				'redirects' => ['scanned' => 0, 'changed' => 0, 'rejected' => 0],
			],
		];

		if ($include_posts) {
			$result['counts']['posts'] = $this->migrate_posts($resolved_source, $overwrite, $dry_run, $limit);
		}
		if ($include_terms) {
			$result['counts']['terms'] = $this->migrate_terms($resolved_source, $overwrite, $dry_run, $limit);
		}
		if ($include_authors) {
			$result['counts']['authors'] = $this->migrate_authors($resolved_source, $overwrite, $dry_run, $limit);
		}
		if ($include_redirects) {
			$result['counts']['redirects'] = $this->migrate_redirects($resolved_source, $redirect_conflict, $dry_run, $limit);
		}

		update_option(Settings::OPTION_LAST_MIGRATION_REPORT, $result, false);

		return new \WP_REST_Response($result, 200);
	}

	/**
	 * @return array<string, array<string, mixed>>
	 */
	public static function detect_supported_plugins()
	{
		$plugins = [
			'yoast' => [
				'slug' => 'wordpress-seo/wp-seo.php',
				'name' => 'Yoast SEO',
			],
			'rank_math' => [
				'slug' => 'seo-by-rank-math/rank-math.php',
				'name' => 'Rank Math SEO',
			],
		];

		if (! function_exists('get_plugins')) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		$all_plugins = function_exists('get_plugins') ? get_plugins() : [];
		$active_plugins = (array) get_option('active_plugins', []);
		$network_active = (array) get_site_option('active_sitewide_plugins', []);

		$result = [];
		foreach ($plugins as $id => $plugin) {
			$slug = $plugin['slug'];
			$installed = isset($all_plugins[$slug]);
			$active = in_array($slug, $active_plugins, true) || isset($network_active[$slug]);
			$version = $installed && isset($all_plugins[$slug]['Version']) ? (string) $all_plugins[$slug]['Version'] : '';

			$result[$id] = [
				'id'        => $id,
				'name'      => $plugin['name'],
				'slug'      => $slug,
				'installed' => $installed,
				'active'    => $active,
				'version'   => $version,
			];
		}

		return $result;
	}

	/**
	 * @return array<string, bool>
	 */
	private function get_legacy_data_status()
	{
		global $wpdb;

		$yoast_posts = (bool) $wpdb->get_var(
			"SELECT post_id FROM {$wpdb->postmeta} WHERE meta_key LIKE '_yoast_wpseo_%' LIMIT 1"
		);
		$yoast_tax = (array) get_option('wpseo_taxonomy_meta', []);
		$yoast_users = (bool) $wpdb->get_var(
			"SELECT user_id FROM {$wpdb->usermeta} WHERE meta_key IN ('wpseo_title','wpseo_desc','wpseo_metadesc','wpseo_noindex_author') LIMIT 1"
		);

		$rank_posts = (bool) $wpdb->get_var(
			"SELECT post_id FROM {$wpdb->postmeta} WHERE meta_key LIKE 'rank_math_%' LIMIT 1"
		);
		$rank_terms = (bool) $wpdb->get_var(
			"SELECT term_id FROM {$wpdb->termmeta} WHERE meta_key LIKE 'rank_math_%' LIMIT 1"
		);
		$rank_users = (bool) $wpdb->get_var(
			"SELECT user_id FROM {$wpdb->usermeta} WHERE meta_key LIKE 'rank_math_%' LIMIT 1"
		);
		$yoast_redirects = (array) get_option('wpseo-premium-redirects-base', []);
		$rank_redirects = false;
		$rank_table = $wpdb->prefix . 'rank_math_redirections';
		if ($this->table_exists($rank_table)) {
			$rank_redirects = (bool) $wpdb->get_var("SELECT id FROM {$rank_table} LIMIT 1");
		}

		return [
			'yoast'     => $yoast_posts || ! empty($yoast_tax) || $yoast_users || ! empty($yoast_redirects),
			'rank_math' => $rank_posts || $rank_terms || $rank_users || $rank_redirects,
		];
	}

	/**
	 * @param array<string, mixed> $config
	 * @param array<string, array<string, mixed>> $detected
	 * @return bool
	 */
	private function is_effective_migration_mode($config, $detected)
	{
		if (! empty($config['enabled'])) {
			return true;
		}
		if (empty($config['auto_enable'])) {
			return false;
		}

		foreach ($detected as $plugin) {
			if (! empty($plugin['installed'])) {
				return true;
			}
		}

		return false;
	}

	/**
	 * @param array<string, array<string, mixed>> $detected
	 * @param array<string, bool> $legacy
	 * @return bool
	 */
	private function has_detection_or_data($detected, $legacy)
	{
		foreach ($detected as $plugin) {
			if (! empty($plugin['installed']) || ! empty($plugin['active'])) {
				return true;
			}
		}

		return ! empty($legacy['yoast']) || ! empty($legacy['rank_math']);
	}

	/**
	 * @param string $requested
	 * @return string
	 */
	private function resolve_source($requested)
	{
		if (in_array($requested, ['yoast', 'rank_math'], true)) {
			return $requested;
		}

		$legacy = $this->get_legacy_data_status();
		if (! empty($legacy['yoast'])) {
			return 'yoast';
		}
		if (! empty($legacy['rank_math'])) {
			return 'rank_math';
		}

		$detected = self::detect_supported_plugins();
		if (! empty($detected['yoast']['installed'])) {
			return 'yoast';
		}
		if (! empty($detected['rank_math']['installed'])) {
			return 'rank_math';
		}

		return '';
	}

	/**
	 * @param string $source
	 * @param bool $overwrite
	 * @param bool $dry_run
	 * @param int $limit
	 * @return array<string, int>
	 */
	private function migrate_posts($source, $overwrite, $dry_run, $limit)
	{
		$post_types = get_post_types(['public' => true], 'names');
		$post_ids = get_posts([
			'post_type'      => array_values($post_types),
			'post_status'    => 'any',
			'posts_per_page' => $limit,
			'orderby'        => 'ID',
			'order'          => 'ASC',
			'fields'         => 'ids',
			'no_found_rows'  => true,
		]);

		$changed = 0;
		foreach ((array) $post_ids as $post_id) {
			$payload = ('yoast' === $source)
				? $this->build_post_payload_from_yoast((int) $post_id)
				: $this->build_post_payload_from_rank_math((int) $post_id);

			if (empty($payload)) {
				continue;
			}

			if ($this->apply_payload_to_post((int) $post_id, $payload, $overwrite, $dry_run)) {
				$changed++;
			}
		}

		return [
			'scanned' => count((array) $post_ids),
			'changed' => $changed,
		];
	}

	/**
	 * @param string $source
	 * @param bool $overwrite
	 * @param bool $dry_run
	 * @param int $limit
	 * @return array<string, int>
	 */
	private function migrate_terms($source, $overwrite, $dry_run, $limit)
	{
		$term_ids = get_terms([
			'taxonomy'   => get_taxonomies([], 'names'),
			'hide_empty' => false,
			'fields'     => 'ids',
			'number'     => $limit,
		]);
		if (is_wp_error($term_ids)) {
			$term_ids = [];
		}

		$yoast_tax_meta = (array) get_option('wpseo_taxonomy_meta', []);
		$changed = 0;
		foreach ((array) $term_ids as $term_id) {
			$payload = ('yoast' === $source)
				? $this->build_term_payload_from_yoast((int) $term_id, $yoast_tax_meta)
				: $this->build_term_payload_from_rank_math((int) $term_id);

			if (empty($payload)) {
				continue;
			}

			if ($this->apply_payload_to_term((int) $term_id, $payload, $overwrite, $dry_run)) {
				$changed++;
			}
		}

		return [
			'scanned' => count((array) $term_ids),
			'changed' => $changed,
		];
	}

	/**
	 * @param string $source
	 * @param bool $overwrite
	 * @param bool $dry_run
	 * @param int $limit
	 * @return array<string, int>
	 */
	private function migrate_authors($source, $overwrite, $dry_run, $limit)
	{
		$users = get_users([
			'number' => $limit,
			'fields' => ['ID'],
		]);

		$changed = 0;
		foreach ($users as $user) {
			$user_id = isset($user->ID) ? (int) $user->ID : 0;
			if ($user_id <= 0) {
				continue;
			}

			$payload = ('yoast' === $source)
				? $this->build_author_payload_from_yoast($user_id)
				: $this->build_author_payload_from_rank_math($user_id);

			if (empty($payload)) {
				continue;
			}

			if ($this->apply_payload_to_author($user_id, $payload, $overwrite, $dry_run)) {
				$changed++;
			}
		}

		return [
			'scanned' => count($users),
			'changed' => $changed,
		];
	}

	/**
	 * @param string $source
	 * @param bool $overwrite
	 * @param bool $dry_run
	 * @param int $limit
	 * @return array<string, int>
	 */
	private function migrate_redirects($source, $redirect_conflict, $dry_run, $limit)
	{
		$rules_data = ('yoast' === $source)
			? $this->get_yoast_redirect_rules($limit)
			: $this->get_rank_math_redirect_rules($limit);

		$incoming_rules = $rules_data['rules'];
		$rejected = $rules_data['rejected'];
		$scanned = (int) $rules_data['scanned'];
		$changed = 0;

		$current_rules = $this->settings->get_redirect_rules();
		$rules_to_save = [];
		$identity_index = [];
		$full_seen = [];
		foreach ($current_rules as $rule) {
			if (! is_array($rule)) {
				continue;
			}
			$source_key = isset($rule['source']) ? (string) $rule['source'] : '';
			$type_key = isset($rule['type']) ? (string) $rule['type'] : 'exact';
			$normalized = [
				'source' => $source_key,
				'target' => isset($rule['target']) ? (string) $rule['target'] : '',
				'code'   => isset($rule['code']) ? (string) $rule['code'] : '301',
				'type'   => $type_key,
			];
			$rules_to_save[] = $normalized;
			$idx = count($rules_to_save) - 1;
			$identity_key = $this->build_redirect_identity_key($source_key, $type_key);
			if (! isset($identity_index[$identity_key])) {
				$identity_index[$identity_key] = $idx;
			}
			$full_seen[$this->build_redirect_full_key($normalized)] = true;
		}

		foreach ($incoming_rules as $rule) {
			$identity_key = $this->build_redirect_identity_key($rule['source'], $rule['type']);
			$full_key = $this->build_redirect_full_key($rule);
			$has_identity = isset($identity_index[$identity_key]);
			$has_full = isset($full_seen[$full_key]);

			if ('append' === $redirect_conflict) {
				if ($has_full) {
					continue;
				}
				$rules_to_save[] = $rule;
				$idx = count($rules_to_save) - 1;
				if (! isset($identity_index[$identity_key])) {
					$identity_index[$identity_key] = $idx;
				}
				$full_seen[$full_key] = true;
				$changed++;
				continue;
			}

			if ($has_identity) {
				if ('skip' === $redirect_conflict) {
					continue;
				}
				$idx = $identity_index[$identity_key];
				$current_full = $this->build_redirect_full_key($rules_to_save[$idx]);
				if ($current_full === $full_key) {
					continue;
				}
				unset($full_seen[$current_full]);
				$rules_to_save[$idx] = $rule;
				$full_seen[$full_key] = true;
				$changed++;
				continue;
			}

			$rules_to_save[] = $rule;
			$idx = count($rules_to_save) - 1;
			$identity_index[$identity_key] = $idx;
			$full_seen[$full_key] = true;
			$changed++;
		}

		if (! $dry_run && $changed > 0) {
			update_option(Settings::OPTION_REDIRECTS, ['rules' => array_values($rules_to_save)], false);
		}

		return [
			'scanned'  => $scanned,
			'changed'  => $changed,
			'rejected' => count($rejected),
		];
	}

	/**
	 * @param int $limit
	 * @return array<string, mixed>
	 */
	private function get_yoast_redirect_rules($limit)
	{
		$rows = (array) get_option('wpseo-premium-redirects-base', []);
		$rows = array_slice($rows, 0, $limit);

		$rules = [];
		$rejected = [];
		foreach ($rows as $row) {
			if (! is_array($row)) {
				$rejected[] = ['rule' => $row, 'reason' => 'Invalid row format'];
				continue;
			}

			$origin = isset($row['origin']) ? (string) $row['origin'] : '';
			$url    = isset($row['url']) ? (string) $row['url'] : '';
			$code   = isset($row['type']) ? (string) $row['type'] : '301';
			$type   = (isset($row['format']) && 'regex' === (string) $row['format']) ? 'regex' : 'exact';

			$rule = $this->normalize_redirect_rule($origin, $url, $code, $type);
			if (is_wp_error($rule)) {
				$rejected[] = ['rule' => $row, 'reason' => $rule->get_error_message()];
				continue;
			}
			$rules[] = $rule;
		}

		return [
			'rules'    => $rules,
			'rejected' => $rejected,
			'scanned'  => count($rows),
		];
	}

	/**
	 * @param int $limit
	 * @return array<string, mixed>
	 */
	private function get_rank_math_redirect_rules($limit)
	{
		global $wpdb;

		$table = $wpdb->prefix . 'rank_math_redirections';
		if (! $this->table_exists($table)) {
			return [
				'rules' => [],
				'rejected' => [],
				'scanned' => 0,
			];
		}

		$sql = $wpdb->prepare(
			"SELECT id, sources, url_to, header_code, status FROM {$table} ORDER BY id ASC LIMIT %d",
			$limit
		);
		$rows = (array) $wpdb->get_results($sql, ARRAY_A);

		$rules = [];
		$rejected = [];
		foreach ($rows as $row) {
			if (! is_array($row)) {
				continue;
			}

			$status = isset($row['status']) ? (string) $row['status'] : 'active';
			if ('active' !== $status) {
				continue;
			}

			$target = isset($row['url_to']) ? (string) $row['url_to'] : '';
			$code   = isset($row['header_code']) ? (string) $row['header_code'] : '301';
			$sources = maybe_unserialize(isset($row['sources']) ? $row['sources'] : []);
			if (! is_array($sources)) {
				$rejected[] = ['rule' => $row, 'reason' => 'Invalid sources payload'];
				continue;
			}

			foreach ($sources as $source_row) {
				if (! is_array($source_row)) {
					continue;
				}
				$pattern = isset($source_row['pattern']) ? (string) $source_row['pattern'] : '';
				$comparison = isset($source_row['comparison']) ? (string) $source_row['comparison'] : 'exact';
				$type = ('regex' === $comparison) ? 'regex' : 'exact';

				if ('exact' !== $comparison && 'regex' !== $comparison) {
					$rejected[] = ['rule' => $source_row, 'reason' => 'Unsupported comparison type'];
					continue;
				}

				if ('regex' === $type) {
					$pattern = $this->to_pcre_with_delimiters($pattern);
				}

				$rule = $this->normalize_redirect_rule($pattern, $target, $code, $type);
				if (is_wp_error($rule)) {
					$rejected[] = ['rule' => $source_row, 'reason' => $rule->get_error_message()];
					continue;
				}
				$rules[] = $rule;
			}
		}

		return [
			'rules'    => $rules,
			'rejected' => $rejected,
			'scanned'  => count($rows),
		];
	}

	/**
	 * @param string $source
	 * @param string $target
	 * @param string $code
	 * @param string $type
	 * @return array<string, string>|\WP_Error
	 */
	private function normalize_redirect_rule($source, $target, $code, $type)
	{
		$source = trim((string) $source);
		$target = trim((string) $target);
		$type   = ('regex' === $type) ? 'regex' : 'exact';
		$code   = in_array((string) $code, ['301', '302'], true) ? (string) $code : '301';

		if ('' === $source || '' === $target) {
			return new \WP_Error('invalid_rule', 'Source and target are required');
		}

		if ('regex' === $type) {
			if (false === @preg_match($source, '')) {
				return new \WP_Error('invalid_rule', 'Regex source must be a valid PCRE pattern with delimiters');
			}
		} else {
			$source = '/' . ltrim($source, '/');
		}

		if (0 === strpos($target, '/')) {
			$target = home_url($target);
		} else {
			$parsed = wp_parse_url($target);
			if (
				! is_array($parsed) ||
				empty($parsed['scheme']) ||
				empty($parsed['host']) ||
				! in_array(strtolower((string) $parsed['scheme']), ['http', 'https'], true)
			) {
				return new \WP_Error('invalid_rule', 'Target must be a relative path or absolute http/https URL');
			}
			$target = esc_url_raw($target);
		}

		if ('' === $target) {
			return new \WP_Error('invalid_rule', 'Target URL is invalid');
		}

		return [
			'source' => $source,
			'target' => $target,
			'code'   => $code,
			'type'   => $type,
		];
	}

	/**
	 * @param string $source
	 * @param string $type
	 * @return string
	 */
	private function build_redirect_identity_key($source, $type)
	{
		return strtolower(trim((string) $type)) . '|' . trim((string) $source);
	}

	/**
	 * @param array<string, string> $rule
	 * @return string
	 */
	private function build_redirect_full_key($rule)
	{
		return strtolower(trim((string) $rule['type']))
			. '|'
			. trim((string) $rule['source'])
			. '|'
			. trim((string) $rule['target'])
			. '|'
			. trim((string) $rule['code']);
	}

	/**
	 * @param string $pattern
	 * @return string
	 */
	private function to_pcre_with_delimiters($pattern)
	{
		$pattern = trim((string) $pattern);
		if ('' === $pattern) {
			return $pattern;
		}

		if (false !== @preg_match($pattern, '')) {
			return $pattern;
		}

		$delimiters = ['#', '~', '%', '!'];
		foreach ($delimiters as $delimiter) {
			$candidate = $delimiter . str_replace($delimiter, '\\' . $delimiter, $pattern) . $delimiter;
			if (false !== @preg_match($candidate, '')) {
				return $candidate;
			}
		}

		return $pattern;
	}

	/**
	 * @param string $table_name
	 * @return bool
	 */
	private function table_exists($table_name)
	{
		global $wpdb;

		$like = $wpdb->esc_like($table_name);
		$exists = $wpdb->get_var($wpdb->prepare('SHOW TABLES LIKE %s', $like));
		return is_string($exists) && $exists === $table_name;
	}

	/**
	 * @param int $post_id
	 * @return array<string, string>
	 */
	private function build_post_payload_from_yoast($post_id)
	{
		$payload = [
			'_wpzeo_title'               => (string) get_post_meta($post_id, '_yoast_wpseo_title', true),
			'_wpzeo_description'         => (string) get_post_meta($post_id, '_yoast_wpseo_metadesc', true),
			'_wpzeo_focus_keyword'       => (string) get_post_meta($post_id, '_yoast_wpseo_focuskw', true),
			'_wpzeo_canonical'           => (string) get_post_meta($post_id, '_yoast_wpseo_canonical', true),
			'_wpzeo_og_title'            => (string) get_post_meta($post_id, '_yoast_wpseo_opengraph-title', true),
			'_wpzeo_og_description'      => (string) get_post_meta($post_id, '_yoast_wpseo_opengraph-description', true),
			'_wpzeo_twitter_title'       => (string) get_post_meta($post_id, '_yoast_wpseo_twitter-title', true),
			'_wpzeo_twitter_description' => (string) get_post_meta($post_id, '_yoast_wpseo_twitter-description', true),
			'_wpzeo_og_image'            => (string) get_post_meta($post_id, '_yoast_wpseo_opengraph-image', true),
			'_wpzeo_twitter_image'       => (string) get_post_meta($post_id, '_yoast_wpseo_twitter-image', true),
		];

		$robots = $this->normalize_yoast_robots_for_post($post_id);
		if ('' !== $robots['main']) {
			$payload['_wpzeo_robots'] = $robots['main'];
		}
		if ('' !== $robots['advanced']) {
			$payload['_wpzeo_robots_advanced'] = $robots['advanced'];
		}

		return $this->trim_payload($payload);
	}

	/**
	 * @param int $post_id
	 * @return array<string, string>
	 */
	private function build_post_payload_from_rank_math($post_id)
	{
		$payload = [
			'_wpzeo_title'               => (string) get_post_meta($post_id, 'rank_math_title', true),
			'_wpzeo_description'         => (string) get_post_meta($post_id, 'rank_math_description', true),
			'_wpzeo_focus_keyword'       => (string) get_post_meta($post_id, 'rank_math_focus_keyword', true),
			'_wpzeo_canonical'           => (string) get_post_meta($post_id, 'rank_math_canonical_url', true),
			'_wpzeo_og_title'            => (string) get_post_meta($post_id, 'rank_math_facebook_title', true),
			'_wpzeo_og_description'      => (string) get_post_meta($post_id, 'rank_math_facebook_description', true),
			'_wpzeo_twitter_title'       => (string) get_post_meta($post_id, 'rank_math_twitter_title', true),
			'_wpzeo_twitter_description' => (string) get_post_meta($post_id, 'rank_math_twitter_description', true),
			'_wpzeo_og_image'            => (string) get_post_meta($post_id, 'rank_math_facebook_image', true),
			'_wpzeo_twitter_image'       => (string) get_post_meta($post_id, 'rank_math_twitter_image', true),
		];

		$robots_value = get_post_meta($post_id, 'rank_math_robots', true);
		$robots = $this->normalize_rank_math_robots($robots_value);
		if ('' !== $robots['main']) {
			$payload['_wpzeo_robots'] = $robots['main'];
		}
		if ('' !== $robots['advanced']) {
			$payload['_wpzeo_robots_advanced'] = $robots['advanced'];
		}

		return $this->trim_payload($payload);
	}

	/**
	 * @param int $term_id
	 * @param array<string, array<int, array<string, string>>> $yoast_tax_meta
	 * @return array<string, string>
	 */
	private function build_term_payload_from_yoast($term_id, $yoast_tax_meta)
	{
		$data = [];
		foreach ($yoast_tax_meta as $terms) {
			if (isset($terms[$term_id]) && is_array($terms[$term_id])) {
				$data = $terms[$term_id];
				break;
			}
		}

		if (empty($data)) {
			return [];
		}

		$description = isset($data['wpseo_metadesc']) && '' !== (string) $data['wpseo_metadesc']
			? (string) $data['wpseo_metadesc']
			: (isset($data['wpseo_desc']) ? (string) $data['wpseo_desc'] : '');

		$payload = [
			'_wpzeo_term_title'        => isset($data['wpseo_title']) ? (string) $data['wpseo_title'] : '',
			'_wpzeo_term_description'  => $description,
			'_wpzeo_term_focus_keyword' => isset($data['wpseo_focuskw']) ? (string) $data['wpseo_focuskw'] : '',
			'_wpzeo_term_canonical'    => isset($data['wpseo_canonical']) ? (string) $data['wpseo_canonical'] : '',
			'_wpzeo_term_og_image'     => isset($data['wpseo_opengraph-image']) ? (string) $data['wpseo_opengraph-image'] : '',
		];

		if (! empty($data['wpseo_noindex']) && in_array((string) $data['wpseo_noindex'], ['index', 'noindex'], true)) {
			$payload['_wpzeo_term_robots'] = (string) $data['wpseo_noindex'] . ',follow';
		}

		return $this->trim_payload($payload);
	}

	/**
	 * @param int $term_id
	 * @return array<string, string>
	 */
	private function build_term_payload_from_rank_math($term_id)
	{
		$payload = [
			'_wpzeo_term_title'         => (string) get_term_meta($term_id, 'rank_math_title', true),
			'_wpzeo_term_description'   => (string) get_term_meta($term_id, 'rank_math_description', true),
			'_wpzeo_term_focus_keyword' => (string) get_term_meta($term_id, 'rank_math_focus_keyword', true),
			'_wpzeo_term_canonical'     => (string) get_term_meta($term_id, 'rank_math_canonical_url', true),
			'_wpzeo_term_og_image'      => (string) get_term_meta($term_id, 'rank_math_facebook_image', true),
		];

		$robots_value = get_term_meta($term_id, 'rank_math_robots', true);
		$robots = $this->normalize_rank_math_robots($robots_value);
		if ('' !== $robots['main']) {
			$payload['_wpzeo_term_robots'] = $robots['main'];
		}

		return $this->trim_payload($payload);
	}

	/**
	 * @param int $user_id
	 * @return array<string, string>
	 */
	private function build_author_payload_from_yoast($user_id)
	{
		$title = (string) get_user_meta($user_id, 'wpseo_title', true);
		$desc  = (string) get_user_meta($user_id, 'wpseo_metadesc', true);
		if ('' === $desc) {
			$desc = (string) get_user_meta($user_id, 'wpseo_desc', true);
		}
		$noindex = (string) get_user_meta($user_id, 'wpseo_noindex_author', true);

		$payload = [
			'_wpzeo_author_title'       => $title,
			'_wpzeo_author_description' => $desc,
		];
		if (! empty($noindex)) {
			$payload['_wpzeo_author_robots'] = 'noindex,follow';
		}

		return $this->trim_payload($payload);
	}

	/**
	 * @param int $user_id
	 * @return array<string, string>
	 */
	private function build_author_payload_from_rank_math($user_id)
	{
		$payload = [
			'_wpzeo_author_title'         => (string) get_user_meta($user_id, 'rank_math_title', true),
			'_wpzeo_author_description'   => (string) get_user_meta($user_id, 'rank_math_description', true),
			'_wpzeo_author_focus_keyword' => (string) get_user_meta($user_id, 'rank_math_focus_keyword', true),
			'_wpzeo_author_canonical'     => (string) get_user_meta($user_id, 'rank_math_canonical_url', true),
			'_wpzeo_author_og_image'      => (string) get_user_meta($user_id, 'rank_math_facebook_image', true),
		];
		if ('' === $payload['_wpzeo_author_og_image']) {
			$payload['_wpzeo_author_og_image'] = (string) get_user_meta($user_id, 'rank_math_twitter_image', true);
		}

		$robots_value = get_user_meta($user_id, 'rank_math_robots', true);
		$robots = $this->normalize_rank_math_robots($robots_value);
		if ('' !== $robots['main']) {
			$payload['_wpzeo_author_robots'] = $robots['main'];
		}

		return $this->trim_payload($payload);
	}

	/**
	 * @param int $post_id
	 * @param array<string, string> $payload
	 * @param bool $overwrite
	 * @param bool $dry_run
	 * @return bool
	 */
	private function apply_payload_to_post($post_id, $payload, $overwrite, $dry_run)
	{
		return $this->apply_payload(
			$payload,
			static function ($key) use ($post_id) {
				return (string) get_post_meta($post_id, $key, true);
			},
			static function ($key, $value) use ($post_id) {
				update_post_meta($post_id, $key, $value);
			},
			$overwrite,
			$dry_run
		);
	}

	/**
	 * @param int $term_id
	 * @param array<string, string> $payload
	 * @param bool $overwrite
	 * @param bool $dry_run
	 * @return bool
	 */
	private function apply_payload_to_term($term_id, $payload, $overwrite, $dry_run)
	{
		return $this->apply_payload(
			$payload,
			static function ($key) use ($term_id) {
				return (string) get_term_meta($term_id, $key, true);
			},
			static function ($key, $value) use ($term_id) {
				update_term_meta($term_id, $key, $value);
			},
			$overwrite,
			$dry_run
		);
	}

	/**
	 * @param int $user_id
	 * @param array<string, string> $payload
	 * @param bool $overwrite
	 * @param bool $dry_run
	 * @return bool
	 */
	private function apply_payload_to_author($user_id, $payload, $overwrite, $dry_run)
	{
		return $this->apply_payload(
			$payload,
			static function ($key) use ($user_id) {
				return (string) get_user_meta($user_id, $key, true);
			},
			static function ($key, $value) use ($user_id) {
				update_user_meta($user_id, $key, $value);
			},
			$overwrite,
			$dry_run
		);
	}

	/**
	 * @param array<string, string> $payload
	 * @param callable $read_fn
	 * @param callable $write_fn
	 * @param bool $overwrite
	 * @param bool $dry_run
	 * @return bool
	 */
	private function apply_payload($payload, $read_fn, $write_fn, $overwrite, $dry_run)
	{
		$changed = false;
		foreach ($payload as $key => $value) {
			$value = trim((string) $value);
			if ('' === $value) {
				continue;
			}

			$current = trim((string) call_user_func($read_fn, $key));
			if ('' !== $current && ! $overwrite) {
				continue;
			}
			if ($current === $value) {
				continue;
			}

			$changed = true;
			if (! $dry_run) {
				call_user_func($write_fn, $key, $value);
			}
		}

		return $changed;
	}

	/**
	 * @param int $post_id
	 * @return array<string, string>
	 */
	private function normalize_yoast_robots_for_post($post_id)
	{
		$noindex = (int) get_post_meta($post_id, '_yoast_wpseo_meta-robots-noindex', true);
		$nofollow = (string) get_post_meta($post_id, '_yoast_wpseo_meta-robots-nofollow', true);
		$advanced = get_post_meta($post_id, '_yoast_wpseo_meta-robots-adv', true);

		$index_token = '';
		if (1 === $noindex) {
			$index_token = 'noindex';
		} elseif (2 === $noindex) {
			$index_token = 'index';
		}
		$follow_token = ! empty($nofollow) ? 'nofollow' : '';

		$main = '';
		if ('' !== $index_token || '' !== $follow_token) {
			$main = ('' !== $index_token ? $index_token : 'index') . ',' . ('' !== $follow_token ? $follow_token : 'follow');
		}

		$advanced_tokens = $this->normalize_robots_tokens($advanced);
		$advanced_tokens = array_values(array_diff($advanced_tokens, ['index', 'noindex', 'follow', 'nofollow']));

		return [
			'main' => $main,
			'advanced' => implode(',', $advanced_tokens),
		];
	}

	/**
	 * @param mixed $value
	 * @return array<string, string>
	 */
	private function normalize_rank_math_robots($value)
	{
		$tokens = $this->normalize_robots_tokens($value);
		$index = in_array('noindex', $tokens, true) ? 'noindex' : (in_array('index', $tokens, true) ? 'index' : '');
		$follow = in_array('nofollow', $tokens, true) ? 'nofollow' : (in_array('follow', $tokens, true) ? 'follow' : '');

		$main = '';
		if ('' !== $index || '' !== $follow) {
			$main = ('' !== $index ? $index : 'index') . ',' . ('' !== $follow ? $follow : 'follow');
		}

		$advanced = array_values(array_diff($tokens, ['index', 'noindex', 'follow', 'nofollow']));

		return [
			'main' => $main,
			'advanced' => implode(',', $advanced),
		];
	}

	/**
	 * @param mixed $value
	 * @return array<int, string>
	 */
	private function normalize_robots_tokens($value)
	{
		$items = [];
		if (is_array($value)) {
			$items = $value;
		} else {
			$string_value = trim((string) $value);
			if ('' === $string_value) {
				return [];
			}
			$maybe = maybe_unserialize($string_value);
			if (is_array($maybe)) {
				$items = $maybe;
			} else {
				$items = explode(',', $string_value);
			}
		}

		$tokens = [];
		foreach ($items as $item) {
			$parts = explode(',', (string) $item);
			foreach ($parts as $part) {
				$token = strtolower(trim((string) $part));
				if ('' === $token) {
					continue;
				}
				$tokens[] = $token;
			}
		}

		return array_values(array_unique($tokens));
	}

	/**
	 * @param array<string, string> $payload
	 * @return array<string, string>
	 */
	private function trim_payload($payload)
	{
		foreach ($payload as $key => $value) {
			$payload[$key] = trim((string) $value);
			if ('' === $payload[$key]) {
				unset($payload[$key]);
			}
		}
		return $payload;
	}
}
