(function () {
  "use strict";

  if (typeof wpzeoAdmin === "undefined") {
    return;
  }

  var textarea = document.getElementById("wpzeo_redirects_raw");
  var btnLoad = document.getElementById("wpzeo-load-redirects");
  var btnSave = document.getElementById("wpzeo-save-redirects");
  var btnExportCsv = document.getElementById("wpzeo-export-redirects-csv");
  var btnImportCsv = document.getElementById("wpzeo-import-redirects-csv");
  var fileImportCsv = document.getElementById("wpzeo-import-redirects-file");
  var statusEl = document.getElementById("wpzeo-redirects-status");

  if (!textarea || !btnLoad || !btnSave || !btnExportCsv || !btnImportCsv || !fileImportCsv || !statusEl) {
    return;
  }

  function setStatus(message, type) {
    statusEl.className = "notice inline";
    statusEl.style.display = "block";
    if (type === "success") {
      statusEl.classList.add("notice-success");
    } else if (type === "warning") {
      statusEl.classList.add("notice-warning");
    } else if (type === "error") {
      statusEl.classList.add("notice-error");
    } else {
      statusEl.classList.add("notice-info");
    }
    statusEl.innerHTML = "<p>" + message + "</p>";
  }

  function normalizeRule(rule) {
    var source = String(rule.source || "").trim();
    var target = String(rule.target || "").trim();
    var code = String(rule.code || "301").trim();
    var type = String(rule.type || "exact").trim().toLowerCase();

    if (source === "" || target === "") {
      return null;
    }

    if (type !== "regex") {
      type = "exact";
    }

    if (type === "exact" && source.charAt(0) !== "/") {
      source = "/" + source;
    }

    if (code !== "301" && code !== "302") {
      code = "301";
    }

    return { source: source, target: target, code: code, type: type };
  }

  function parseTextarea() {
    var lines = textarea.value.split(/\r?\n/);
    var rules = [];
    var invalid = [];

    for (var i = 0; i < lines.length; i++) {
      var line = lines[i].trim();
      if (!line) {
        continue;
      }

      var parts = line.split("|").map(function (part) {
        return part.trim();
      });

      if (parts.length < 2) {
        invalid.push("Line " + (i + 1) + ": expected source|target|code|type");
        continue;
      }

      var parsed = normalizeRule({
        source: parts[0],
        target: parts[1],
        code: parts[2] || "301",
        type: parts[3] || "exact",
      });

      if (!parsed) {
        invalid.push("Line " + (i + 1) + ": invalid source/target");
        continue;
      }

      rules.push(parsed);
    }

    return { rules: rules, invalid: invalid };
  }

  function rulesToTextarea(rules) {
    textarea.value = (rules || [])
      .map(function (rule) {
        return [rule.source, rule.target, rule.code, rule.type || "exact"].join("|");
      })
      .join("\n");
  }

  function request(method, body) {
    return requestToUrl(wpzeoAdmin.restBase, method, body);
  }

  function requestToUrl(url, method, body) {
    return fetch(url, {
      method: method,
      credentials: "same-origin",
      headers: {
        "Content-Type": "application/json",
        "X-WP-Nonce": wpzeoAdmin.nonce,
      },
      body: body ? JSON.stringify(body) : undefined,
    }).then(function (response) {
      return response.json().then(function (json) {
        return { ok: response.ok, status: response.status, json: json };
      });
    });
  }

  btnLoad.addEventListener("click", function () {
    setStatus(wpzeoAdmin.i18n.loading, "info");

    request("GET").then(function (result) {
      if (!result.ok) {
        var message = (result.json && (result.json.message || result.json.error)) || wpzeoAdmin.i18n.error;
        setStatus(message, "error");
        return;
      }

      var rules = result.json.rules || [];
      rulesToTextarea(rules);
      var totalHits = rules.reduce(function (sum, rule) {
        return sum + Number(rule.hits || 0);
      }, 0);
      setStatus(wpzeoAdmin.i18n.loaded + " (" + rules.length + " rules, " + totalHits + " hits)", "success");
    }).catch(function () {
      setStatus(wpzeoAdmin.i18n.error, "error");
    });
  });

  btnSave.addEventListener("click", function () {
    var parsed = parseTextarea();
    if (parsed.invalid.length > 0) {
      setStatus(parsed.invalid.join("<br>"), "warning");
      return;
    }

    setStatus(wpzeoAdmin.i18n.saving, "info");

    request("POST", { rules: parsed.rules }).then(function (result) {
      if (!result.ok) {
        var message = (result.json && (result.json.message || result.json.error)) || wpzeoAdmin.i18n.error;
        setStatus(message, "error");
        return;
      }

      var rejectedCount = Number(result.json.rejected_count || 0);
      var message = wpzeoAdmin.i18n.saved + " (" + (result.json.count || 0) + " rules)";
      if (rejectedCount > 0) {
        message += "<br>" + rejectedCount + " rule(s) rejected by API.";
        setStatus(message, "warning");
      } else {
        setStatus(message, "success");
      }

      rulesToTextarea(result.json.rules || []);
    }).catch(function () {
      setStatus(wpzeoAdmin.i18n.error, "error");
    });
  });

  btnExportCsv.addEventListener("click", function () {
    setStatus(wpzeoAdmin.i18n.exporting, "info");

    requestToUrl(wpzeoAdmin.restExport, "GET").then(function (result) {
      if (!result.ok) {
        var message = (result.json && (result.json.message || result.json.error)) || wpzeoAdmin.i18n.error;
        setStatus(message, "error");
        return;
      }

      var csv = String((result.json && result.json.csv) || "");
      var blob = new Blob([csv], { type: "text/csv;charset=utf-8" });
      var url = URL.createObjectURL(blob);
      var a = document.createElement("a");
      a.href = url;
      a.download = "wpzeo-redirects.csv";
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);
      URL.revokeObjectURL(url);

      setStatus(wpzeoAdmin.i18n.exported + " (" + Number(result.json.count || 0) + " rules)", "success");
    }).catch(function () {
      setStatus(wpzeoAdmin.i18n.error, "error");
    });
  });

  btnImportCsv.addEventListener("click", function () {
    fileImportCsv.click();
  });

  fileImportCsv.addEventListener("change", function () {
    var file = fileImportCsv.files && fileImportCsv.files[0];
    if (!file) {
      return;
    }

    var reader = new FileReader();
    reader.onload = function () {
      var csv = String(reader.result || "");
      if (!csv.trim()) {
        setStatus("CSV is empty.", "warning");
        return;
      }

      setStatus(wpzeoAdmin.i18n.importing, "info");
      requestToUrl(wpzeoAdmin.restImport, "POST", { csv: csv, replace: true, reset_stats: false }).then(function (result) {
        if (!result.ok) {
          var message = (result.json && (result.json.message || result.json.error)) || wpzeoAdmin.i18n.error;
          setStatus(message, "error");
          return;
        }

        rulesToTextarea(result.json.rules || []);
        var rejectedCount = Number(result.json.rejected_count || 0);
        var message = wpzeoAdmin.i18n.imported + " (" + Number(result.json.count || 0) + " rules)";
        if (rejectedCount > 0) {
          message += "<br>" + rejectedCount + " row(s) rejected by API.";
          setStatus(message, "warning");
        } else {
          setStatus(message, "success");
        }
      }).catch(function () {
        setStatus(wpzeoAdmin.i18n.error, "error");
      });
    };
    reader.onerror = function () {
      setStatus("Could not read CSV file.", "error");
    };
    reader.readAsText(file);
  });
})();
