(function () {
  "use strict";

  if (typeof wpzeoMigrationAdmin === "undefined") {
    return;
  }

  var btnRefresh = document.getElementById("wpzeo-migration-refresh-status");
  var btnDryRun = document.getElementById("wpzeo-migration-dry-run");
  var btnRun = document.getElementById("wpzeo-migration-run");
  var source = document.getElementById("wpzeo_migration_source");
  var redirectConflict = document.getElementById("wpzeo_migration_redirect_conflict");
  var includePosts = document.getElementById("wpzeo_migration_include_posts");
  var includeTerms = document.getElementById("wpzeo_migration_include_terms");
  var includeAuthors = document.getElementById("wpzeo_migration_include_authors");
  var includeRedirects = document.getElementById("wpzeo_migration_include_redirects");
  var overwrite = document.getElementById("wpzeo_migration_overwrite");
  var statusEl = document.getElementById("wpzeo-migration-status");
  var resultEl = document.getElementById("wpzeo-migration-result");

  if (
    !btnRefresh ||
    !btnDryRun ||
    !btnRun ||
    !source ||
    !redirectConflict ||
    !includePosts ||
    !includeTerms ||
    !includeAuthors ||
    !includeRedirects ||
    !overwrite ||
    !statusEl ||
    !resultEl
  ) {
    return;
  }

  function setStatus(message, type) {
    statusEl.className = "notice inline";
    statusEl.style.display = "block";
    if (type === "success") {
      statusEl.classList.add("notice-success");
    } else if (type === "warning") {
      statusEl.classList.add("notice-warning");
    } else if (type === "error") {
      statusEl.classList.add("notice-error");
    } else {
      statusEl.classList.add("notice-info");
    }
    statusEl.innerHTML = "<p>" + message + "</p>";
  }

  function showResult(json) {
    resultEl.style.display = "block";
    resultEl.textContent = JSON.stringify(json || {}, null, 2);
  }

  function setBusy(isBusy) {
    btnRefresh.disabled = isBusy;
    btnDryRun.disabled = isBusy;
    btnRun.disabled = isBusy;
  }

  function request(url, method, body) {
    return fetch(url, {
      method: method,
      credentials: "same-origin",
      headers: {
        "Content-Type": "application/json",
        "X-WP-Nonce": wpzeoMigrationAdmin.nonce,
      },
      body: body ? JSON.stringify(body) : undefined,
    }).then(function (response) {
      return response.json().then(function (json) {
        return { ok: response.ok, status: response.status, json: json };
      });
    });
  }

  function buildPayload(dryRun) {
    return {
      source: source.value,
      redirect_conflict: redirectConflict.value,
      include_posts: includePosts.checked ? 1 : 0,
      include_terms: includeTerms.checked ? 1 : 0,
      include_authors: includeAuthors.checked ? 1 : 0,
      include_redirects: includeRedirects.checked ? 1 : 0,
      overwrite: overwrite.checked ? 1 : 0,
      dry_run: dryRun ? 1 : 0,
    };
  }

  function loadStatus() {
    setBusy(true);
    setStatus(wpzeoMigrationAdmin.i18n.loadingStatus, "info");
    request(wpzeoMigrationAdmin.restStatus, "GET")
      .then(function (result) {
        if (!result.ok) {
          var message =
            (result.json && (result.json.message || result.json.error)) ||
            wpzeoMigrationAdmin.i18n.error;
          setStatus(message, "error");
          showResult(result.json);
          return;
        }
        showResult(result.json);
        setStatus(wpzeoMigrationAdmin.i18n.statusLoaded, "success");
      })
      .catch(function () {
        setStatus(wpzeoMigrationAdmin.i18n.error, "error");
      })
      .finally(function () {
        setBusy(false);
      });
  }

  function runImport(dryRun) {
    var payload = buildPayload(dryRun);
    setBusy(true);
    setStatus(
      dryRun ? wpzeoMigrationAdmin.i18n.runningDryRun : wpzeoMigrationAdmin.i18n.runningImport,
      "info"
    );
    request(wpzeoMigrationAdmin.restImport, "POST", payload)
      .then(function (result) {
        showResult(result.json);
        if (!result.ok) {
          var message =
            (result.json && (result.json.message || result.json.error)) ||
            wpzeoMigrationAdmin.i18n.error;
          setStatus(message, "error");
          return;
        }
        setStatus(dryRun ? wpzeoMigrationAdmin.i18n.dryRunDone : wpzeoMigrationAdmin.i18n.importDone, "success");
      })
      .catch(function () {
        setStatus(wpzeoMigrationAdmin.i18n.error, "error");
      })
      .finally(function () {
        setBusy(false);
      });
  }

  btnRefresh.addEventListener("click", loadStatus);
  btnDryRun.addEventListener("click", function () {
    runImport(true);
  });
  btnRun.addEventListener("click", function () {
    runImport(false);
  });
})();
