<?php

namespace WPZEO\Modules;

use WPZEO\Core\Settings;

if (! defined('ABSPATH')) {
	exit;
}

class Social
{
	/**
	 * @var Settings
	 */
	private $settings;
	/**
	 * @var bool
	 */
	private $defer_to_external_seo = false;

	/**
	 * @param Settings $settings
	 */
	public function __construct(Settings $settings)
	{
		$this->settings = $settings;
		$this->defer_to_external_seo = $this->should_defer_to_external_seo();
		add_action('rest_api_init', [$this, 'register_rest_routes']);
		add_action('wp_head', [$this, 'output_social_tags'], 2);
	}

	public function register_rest_routes()
	{
		register_rest_route(
			'wpzeo/v1',
			'/social-settings',
			[
				[
					'methods'             => 'GET',
					'callback'            => [$this, 'get_social_settings'],
					'permission_callback' => [$this, 'can_manage_settings'],
				],
				[
					'methods'             => 'POST',
					'callback'            => [$this, 'update_social_settings'],
					'permission_callback' => [$this, 'can_manage_settings'],
				],
			]
		);
	}

	/**
	 * @return bool
	 */
	public function can_manage_settings()
	{
		return current_user_can('manage_options');
	}

	/**
	 * @return \WP_REST_Response
	 */
	public function get_social_settings()
	{
		return new \WP_REST_Response(
			[
				'settings' => $this->settings->get_social(),
			],
			200
		);
	}

	/**
	 * Payload keys:
	 * default_image, twitter_card
	 *
	 * @param \WP_REST_Request $request
	 * @return \WP_REST_Response
	 */
	public function update_social_settings($request)
	{
		$current = $this->settings->get_social();
		$input   = $request->get_json_params();
		$input   = is_array($input) ? $input : [];

		$default_image = array_key_exists('default_image', $input) ? esc_url_raw((string) $input['default_image']) : (string) $current['default_image'];
		$twitter_card  = array_key_exists('twitter_card', $input) ? sanitize_text_field((string) $input['twitter_card']) : (string) $current['twitter_card'];

		if (! in_array($twitter_card, ['summary', 'summary_large_image'], true)) {
			$twitter_card = 'summary_large_image';
		}

		$payload = [
			'default_image' => $default_image,
			'twitter_card'  => $twitter_card,
		];

		update_option(Settings::OPTION_SOCIAL, $payload, false);

		return new \WP_REST_Response(
			[
				'settings' => $this->settings->get_social(),
			],
			200
		);
	}

	public function output_social_tags()
	{
		if ($this->defer_to_external_seo || is_admin() || ! is_main_query()) {
			return;
		}

		$title       = $this->get_title();
		$description = $this->get_description();
		$og_title    = $this->get_og_title($title);
		$og_description = $this->get_og_description($description);
		$url         = $this->get_url();
		$image       = $this->get_image();
		$twitter_title = $this->get_twitter_title($og_title);
		$twitter_description = $this->get_twitter_description($og_description);
		$twitter_image = $this->get_twitter_image($image);
		$social      = $this->settings->get_social();
		$card        = (string) $social['twitter_card'];

		if ('' !== $og_title) {
			echo '<meta property="og:title" content="' . esc_attr($og_title) . '">' . "\n";
		}
		if ('' !== $twitter_title) {
			echo '<meta name="twitter:title" content="' . esc_attr($twitter_title) . '">' . "\n";
		}

		if ('' !== $og_description) {
			echo '<meta property="og:description" content="' . esc_attr($og_description) . '">' . "\n";
		}
		if ('' !== $twitter_description) {
			echo '<meta name="twitter:description" content="' . esc_attr($twitter_description) . '">' . "\n";
		}

		if ('' !== $url) {
			echo '<meta property="og:url" content="' . esc_url($url) . '">' . "\n";
		}

		if ('' !== $image) {
			echo '<meta property="og:image" content="' . esc_url($image) . '">' . "\n";
		}
		if ('' !== $twitter_image) {
			echo '<meta name="twitter:image" content="' . esc_url($twitter_image) . '">' . "\n";
		}

		if ('' === $twitter_image && '' !== $image) {
			echo '<meta name="twitter:image" content="' . esc_url($image) . '">' . "\n";
		}

		echo '<meta property="og:type" content="' . esc_attr(is_singular() ? 'article' : 'website') . '">' . "\n";
		echo '<meta name="twitter:card" content="' . esc_attr($card) . '">' . "\n";
	}

	/**
	 * @return string
	 */
	private function get_title()
	{
		if (is_singular()) {
			$post_id = get_queried_object_id();
			$value   = trim((string) get_post_meta($post_id, '_wpzeo_title', true));
			if ('' !== $value) {
				return $value;
			}

			$post = get_post($post_id);
			if ($post instanceof \WP_Post) {
				$template = $this->get_post_type_template('title', (string) $post->post_type);
				$resolved = $this->resolve_template($template, $this->get_post_template_vars($post_id, $post));
				if ('' !== $resolved) {
					return $resolved;
				}
			}
			return wp_strip_all_tags((string) get_the_title($post_id), true);
		}

		if (is_category() || is_tag() || is_tax()) {
			$term = get_queried_object();
			if ($term instanceof \WP_Term) {
				$value = trim((string) get_term_meta($term->term_id, '_wpzeo_term_title', true));
				if ('' !== $value) {
					return $value;
				}

				$template = $this->get_taxonomy_template('title', (string) $term->taxonomy);
				$resolved = $this->resolve_template($template, $this->get_term_template_vars($term));
				if ('' !== $resolved) {
					return $resolved;
				}
				return wp_strip_all_tags((string) single_term_title('', false), true);
			}
		}

		if (is_author()) {
			$author_id = (int) get_query_var('author');
			$value     = trim((string) get_user_meta($author_id, '_wpzeo_author_title', true));
			if ('' !== $value) {
				return $value;
			}

			$template = $this->get_author_template('title');
			$resolved = $this->resolve_template($template, $this->get_author_template_vars($author_id));
			if ('' !== $resolved) {
				return $resolved;
			}

			$user = get_userdata($author_id);
			if ($user) {
				return wp_strip_all_tags((string) $user->display_name, true);
			}
		}

		return wp_strip_all_tags((string) wp_get_document_title(), true);
	}

	/**
	 * @return string
	 */
	private function get_description()
	{
		if (is_singular()) {
			$post_id = get_queried_object_id();
			$value   = trim((string) get_post_meta($post_id, '_wpzeo_description', true));
			if ('' !== $value) {
				return $value;
			}

			$post = get_post($post_id);
			if ($post instanceof \WP_Post) {
				$template = $this->get_post_type_template('description', (string) $post->post_type);
				$resolved = $this->resolve_template($template, $this->get_post_template_vars($post_id, $post));
				if ('' !== $resolved) {
					return $resolved;
				}
			}
			return wp_strip_all_tags((string) get_the_excerpt($post_id), true);
		}

		if (is_category() || is_tag() || is_tax()) {
			$term = get_queried_object();
			if ($term instanceof \WP_Term) {
				$value = trim((string) get_term_meta($term->term_id, '_wpzeo_term_description', true));
				if ('' !== $value) {
					return $value;
				}

				$template = $this->get_taxonomy_template('description', (string) $term->taxonomy);
				$resolved = $this->resolve_template($template, $this->get_term_template_vars($term));
				if ('' !== $resolved) {
					return $resolved;
				}
				return wp_strip_all_tags((string) term_description($term), true);
			}
		}

		if (is_author()) {
			$author_id = (int) get_query_var('author');
			$value     = trim((string) get_user_meta($author_id, '_wpzeo_author_description', true));
			if ('' !== $value) {
				return $value;
			}

			$template = $this->get_author_template('description');
			$resolved = $this->resolve_template($template, $this->get_author_template_vars($author_id));
			if ('' !== $resolved) {
				return $resolved;
			}
			return wp_strip_all_tags((string) get_the_author_meta('description', $author_id), true);
		}

		return wp_strip_all_tags((string) get_bloginfo('description'), true);
	}

	/**
	 * @return string
	 */
	private function get_url()
	{
		if (is_paged()) {
			return (string) get_pagenum_link(max(1, (int) get_query_var('paged')));
		}

		if (is_singular()) {
			return (string) get_permalink(get_queried_object_id());
		}

		if (is_category() || is_tag() || is_tax()) {
			$term = get_queried_object();
			if ($term instanceof \WP_Term) {
				$link = get_term_link($term);
				if (! is_wp_error($link)) {
					return (string) $link;
				}
			}
		}

		if (is_author()) {
			return (string) get_author_posts_url((int) get_query_var('author'));
		}

		$request = '';
		if (isset($GLOBALS['wp']) && isset($GLOBALS['wp']->request)) {
			$request = (string) $GLOBALS['wp']->request;
		}

		if ('' === $request) {
			return (string) home_url('/');
		}

		return (string) home_url('/' . ltrim($request, '/'));
	}

	/**
	 * @return string
	 */
	private function get_image()
	{
		if (is_singular()) {
			$post_id = get_queried_object_id();
			$custom  = trim((string) get_post_meta($post_id, '_wpzeo_og_image', true));
			if ('' !== $custom) {
				return $custom;
			}

			$thumb = get_the_post_thumbnail_url($post_id, 'full');
				if ($thumb) {
					return (string) $thumb;
				}
			}

			if (is_category() || is_tag() || is_tax()) {
				$term = get_queried_object();
				if ($term instanceof \WP_Term) {
					$custom = trim((string) get_term_meta($term->term_id, '_wpzeo_term_og_image', true));
					if ('' !== $custom) {
						return $custom;
					}
				}
			}

			if (is_author()) {
				$author_id = (int) get_query_var('author');
				$custom    = trim((string) get_user_meta($author_id, '_wpzeo_author_og_image', true));
				if ('' !== $custom) {
					return $custom;
				}
			}

			$social = $this->settings->get_social();
		return trim((string) $social['default_image']);
	}

	/**
	 * @param string $kind title|description
	 * @param string $post_type
	 * @return string
	 */
	private function get_post_type_template($kind, $post_type)
	{
		$general = $this->settings->get_general();
		$key = ('description' === $kind) ? 'description_template_post_types' : 'title_template_post_types';
		$templates = isset($general[$key]) && is_array($general[$key]) ? $general[$key] : [];
		return isset($templates[$post_type]) ? trim((string) $templates[$post_type]) : '';
	}

	/**
	 * @param string $kind title|description
	 * @param string $taxonomy
	 * @return string
	 */
	private function get_taxonomy_template($kind, $taxonomy)
	{
		$general = $this->settings->get_general();
		$key = ('description' === $kind) ? 'description_template_taxonomies' : 'title_template_taxonomies';
		$templates = isset($general[$key]) && is_array($general[$key]) ? $general[$key] : [];
		return isset($templates[$taxonomy]) ? trim((string) $templates[$taxonomy]) : '';
	}

	/**
	 * @param string $kind title|description
	 * @return string
	 */
	private function get_author_template($kind)
	{
		$general = $this->settings->get_general();
		$key = ('description' === $kind) ? 'description_template_author' : 'title_template_author';
		return isset($general[$key]) ? trim((string) $general[$key]) : '';
	}

	/**
	 * @param int $post_id
	 * @param \WP_Post $post
	 * @return array<string, string>
	 */
	private function get_post_template_vars($post_id, $post)
	{
		return [
			'%title%' => wp_strip_all_tags((string) get_the_title($post_id), true),
			'%sitename%' => wp_strip_all_tags((string) get_bloginfo('name'), true),
			'%excerpt%' => $this->get_post_excerpt_for_template($post_id, $post),
		];
	}

	/**
	 * @param \WP_Term $term
	 * @return array<string, string>
	 */
	private function get_term_template_vars($term)
	{
		return [
			'%term%' => wp_strip_all_tags((string) single_term_title('', false), true),
			'%term_description%' => wp_strip_all_tags((string) term_description($term), true),
			'%sitename%' => wp_strip_all_tags((string) get_bloginfo('name'), true),
		];
	}

	/**
	 * @param int $author_id
	 * @return array<string, string>
	 */
	private function get_author_template_vars($author_id)
	{
		$user = get_userdata($author_id);
		$name = $user ? (string) $user->display_name : '';
		$bio = $user ? (string) get_the_author_meta('description', $user->ID) : '';

		return [
			'%author%' => wp_strip_all_tags($name, true),
			'%author_bio%' => wp_strip_all_tags($bio, true),
			'%sitename%' => wp_strip_all_tags((string) get_bloginfo('name'), true),
		];
	}

	/**
	 * @param int $post_id
	 * @param \WP_Post $post
	 * @return string
	 */
	private function get_post_excerpt_for_template($post_id, $post)
	{
		$excerpt = trim(wp_strip_all_tags((string) get_the_excerpt($post_id), true));
		if ('' !== $excerpt) {
			return $excerpt;
		}

		$content = trim(wp_strip_all_tags((string) $post->post_content, true));
		if ('' === $content) {
			return '';
		}

		if (function_exists('mb_substr')) {
			return (string) mb_substr($content, 0, 160);
		}

		return (string) substr($content, 0, 160);
	}

	/**
	 * @param string $template
	 * @param array<string, string> $vars
	 * @return string
	 */
	private function resolve_template($template, $vars)
	{
		$template = trim((string) $template);
		if ('' === $template) {
			return '';
		}

		$resolved = str_ireplace(array_keys($vars), array_values($vars), $template);
		$resolved = wp_strip_all_tags((string) $resolved, true);
		$resolved = preg_replace('/\s+/', ' ', (string) $resolved);
		return is_string($resolved) ? trim($resolved) : '';
	}

	/**
	 * @param string $fallback
	 * @return string
	 */
	private function get_og_title($fallback)
	{
		if (is_singular()) {
			$post_id = get_queried_object_id();
			$custom  = trim((string) get_post_meta($post_id, '_wpzeo_og_title', true));
			if ('' !== $custom) {
				return $custom;
			}
		}

		return $fallback;
	}

	/**
	 * @param string $fallback
	 * @return string
	 */
	private function get_og_description($fallback)
	{
		if (is_singular()) {
			$post_id = get_queried_object_id();
			$custom  = trim((string) get_post_meta($post_id, '_wpzeo_og_description', true));
			if ('' !== $custom) {
				return $custom;
			}
		}

		return $fallback;
	}

	/**
	 * @param string $fallback
	 * @return string
	 */
	private function get_twitter_title($fallback)
	{
		if (is_singular()) {
			$post_id = get_queried_object_id();
			$custom  = trim((string) get_post_meta($post_id, '_wpzeo_twitter_title', true));
			if ('' !== $custom) {
				return $custom;
			}
		}

		return $fallback;
	}

	/**
	 * @param string $fallback
	 * @return string
	 */
	private function get_twitter_description($fallback)
	{
		if (is_singular()) {
			$post_id = get_queried_object_id();
			$custom  = trim((string) get_post_meta($post_id, '_wpzeo_twitter_description', true));
			if ('' !== $custom) {
				return $custom;
			}
		}

		return $fallback;
	}

	/**
	 * @param string $fallback
	 * @return string
	 */
	private function get_twitter_image($fallback)
	{
		if (is_singular()) {
			$post_id = get_queried_object_id();
			$custom  = trim((string) get_post_meta($post_id, '_wpzeo_twitter_image', true));
			if ('' !== $custom) {
				return $custom;
			}
		}

		return $fallback;
	}

	/**
	 * @return bool
	 */
	private function should_defer_to_external_seo()
	{
		$known_constants = [
			'WPSEO_VERSION',
			'RANK_MATH_VERSION',
			'AIOSEO_VERSION',
			'SEOPRESS_VERSION',
			'THE_SEO_FRAMEWORK_VERSION',
		];

		$detected = false;
		foreach ($known_constants as $constant_name) {
			if (defined($constant_name)) {
				$detected = true;
				break;
			}
		}

		return (bool) apply_filters('wpzeo/defer_to_external_seo', $detected, $detected);
	}
}
