<?php

namespace WPZEO\Modules;

use WPZEO\Core\Settings;

if (! defined('ABSPATH')) {
	exit;
}

class Sitemap
{
	/**
	 * @var Settings
	 */
	private $settings;

	/**
	 * @param Settings $settings
	 */
	public function __construct(Settings $settings)
	{
		$this->settings = $settings;

		add_action('init', [$this, 'register_rewrite']);
		add_action('rest_api_init', [$this, 'register_rest_routes']);
		add_filter('query_vars', [$this, 'register_query_vars']);
		add_action('template_redirect', [$this, 'maybe_render_sitemap']);
	}

	public function register_rest_routes()
	{
		register_rest_route(
			'wpzeo/v1',
			'/sitemap-settings',
			[
				[
					'methods'             => 'GET',
					'callback'            => [$this, 'get_sitemap_settings'],
					'permission_callback' => [$this, 'can_manage_settings'],
				],
				[
					'methods'             => 'POST',
					'callback'            => [$this, 'update_sitemap_settings'],
					'permission_callback' => [$this, 'can_manage_settings'],
				],
			]
		);
	}

	/**
	 * @return bool
	 */
	public function can_manage_settings()
	{
		return current_user_can('manage_options');
	}

	/**
	 * @return \WP_REST_Response
	 */
	public function get_sitemap_settings()
	{
		$post_types = [];
		foreach (get_post_types(['public' => true], 'objects') as $post_type) {
			$post_types[] = [
				'name'  => (string) $post_type->name,
				'label' => (string) $post_type->labels->singular_name,
			];
		}

		$taxonomies = [];
		foreach (get_taxonomies(['public' => true], 'objects') as $taxonomy) {
			$taxonomies[] = [
				'name'  => (string) $taxonomy->name,
				'label' => (string) $taxonomy->labels->singular_name,
			];
		}

		return new \WP_REST_Response(
			[
				'settings'             => $this->settings->get_sitemap(),
				'available_post_types' => $post_types,
				'available_taxonomies' => $taxonomies,
			],
			200
		);
	}

	/**
	 * Payload keys:
	 * enabled, include_authors, limit, include_post_types, include_taxonomies, exclude_post_ids, exclude_term_ids
	 *
	 * @param \WP_REST_Request $request
	 * @return \WP_REST_Response
	 */
	public function update_sitemap_settings($request)
	{
		$current = $this->settings->get_sitemap();
		$input   = $request->get_json_params();
		$input   = is_array($input) ? $input : [];

		$public_post_types = get_post_types(['public' => true], 'names');
		$public_taxonomies = get_taxonomies(['public' => true], 'names');

		$enabled = array_key_exists('enabled', $input) ? (int) ! empty($input['enabled']) : (int) ! empty($current['enabled']);
		$include_authors = array_key_exists('include_authors', $input) ? (int) ! empty($input['include_authors']) : (int) ! empty($current['include_authors']);
		$limit = array_key_exists('limit', $input) ? absint($input['limit']) : absint($current['limit']);
		if ($limit < 100) {
			$limit = 100;
		}
		if ($limit > 5000) {
			$limit = 5000;
		}

		$include_post_types = array_key_exists('include_post_types', $input) ? $input['include_post_types'] : $current['include_post_types'];
		$include_taxonomies = array_key_exists('include_taxonomies', $input) ? $input['include_taxonomies'] : $current['include_taxonomies'];
		$exclude_post_ids   = array_key_exists('exclude_post_ids', $input) ? $input['exclude_post_ids'] : $current['exclude_post_ids'];
		$exclude_term_ids   = array_key_exists('exclude_term_ids', $input) ? $input['exclude_term_ids'] : $current['exclude_term_ids'];

		$include_post_types = is_array($include_post_types) ? $include_post_types : [];
		$include_taxonomies = is_array($include_taxonomies) ? $include_taxonomies : [];
		$exclude_post_ids   = is_array($exclude_post_ids) ? $exclude_post_ids : [];
		$exclude_term_ids   = is_array($exclude_term_ids) ? $exclude_term_ids : [];

		$sanitized_post_types = [];
		foreach ($include_post_types as $post_type) {
			$post_type = sanitize_key((string) $post_type);
			if (isset($public_post_types[$post_type])) {
				$sanitized_post_types[] = $post_type;
			}
		}
		$sanitized_post_types = array_values(array_unique($sanitized_post_types));

		$sanitized_taxonomies = [];
		foreach ($include_taxonomies as $taxonomy) {
			$taxonomy = sanitize_key((string) $taxonomy);
			if (isset($public_taxonomies[$taxonomy])) {
				$sanitized_taxonomies[] = $taxonomy;
			}
		}
		$sanitized_taxonomies = array_values(array_unique($sanitized_taxonomies));

		$sanitized_exclude_post_ids = array_values(array_filter(array_map('absint', $exclude_post_ids), static function ($id) {
			return $id > 0;
		}));
		$sanitized_exclude_term_ids = array_values(array_filter(array_map('absint', $exclude_term_ids), static function ($id) {
			return $id > 0;
		}));

		$payload = [
			'enabled'            => $enabled,
			'include_authors'    => $include_authors,
			'limit'              => $limit,
			'include_post_types' => $sanitized_post_types,
			'include_taxonomies' => $sanitized_taxonomies,
			'exclude_post_ids'   => $sanitized_exclude_post_ids,
			'exclude_term_ids'   => $sanitized_exclude_term_ids,
		];

		update_option(Settings::OPTION_SITEMAP, $payload, false);

		return new \WP_REST_Response(
			[
				'settings' => $this->settings->get_sitemap(),
			],
			200
		);
	}

	public function register_rewrite()
	{
		add_rewrite_rule('^wpzeo-sitemap\.xml$', 'index.php?wpzeo_sitemap=1', 'top');
		add_rewrite_rule('^wpzeo-sitemap\.xml/?$', 'index.php?wpzeo_sitemap=1', 'top');
	}

	/**
	 * @param array<int, string> $vars
	 * @return array<int, string>
	 */
	public function register_query_vars($vars)
	{
		$vars[] = 'wpzeo_sitemap';
		return $vars;
	}

	public function maybe_render_sitemap()
	{
		$flag       = get_query_var('wpzeo_sitemap');
		$request    = isset($_SERVER['REQUEST_URI']) ? (string) wp_unslash($_SERVER['REQUEST_URI']) : '';
		$path       = wp_parse_url($request, PHP_URL_PATH);
		$is_direct  = in_array($path, ['/wpzeo-sitemap.xml', '/wpzeo-sitemap.xml/'], true);
		$sitemap    = $this->settings->get_sitemap();
		$enabled    = ! empty($sitemap['enabled']);
		$is_routed  = ('1' === (string) $flag);

		if (! $enabled || (! $is_routed && ! $is_direct)) {
			return;
		}

		$limit = (int) $sitemap['limit'];
		if ($limit < 100) {
			$limit = 100;
		}

		$urls = $this->collect_urls($limit, ! empty($sitemap['include_authors']));

		nocache_headers();
		header('Content-Type: application/xml; charset=' . get_bloginfo('charset'), true);
		echo $this->build_xml($urls);
		exit;
	}

	/**
	 * @param int $limit
	 * @param bool $include_authors
	 * @return array<int, array<string, string>>
	 */
	private function collect_urls($limit, $include_authors)
	{
		$urls = [];
		$sitemap_settings = $this->settings->get_sitemap();
		$include_post_types = isset($sitemap_settings['include_post_types']) && is_array($sitemap_settings['include_post_types']) ? $sitemap_settings['include_post_types'] : [];
		$include_taxonomies = isset($sitemap_settings['include_taxonomies']) && is_array($sitemap_settings['include_taxonomies']) ? $sitemap_settings['include_taxonomies'] : [];
		$exclude_post_ids = isset($sitemap_settings['exclude_post_ids']) && is_array($sitemap_settings['exclude_post_ids']) ? array_map('absint', $sitemap_settings['exclude_post_ids']) : [];
		$exclude_term_ids = isset($sitemap_settings['exclude_term_ids']) && is_array($sitemap_settings['exclude_term_ids']) ? array_map('absint', $sitemap_settings['exclude_term_ids']) : [];

		$urls[] = [
			'loc'     => home_url('/'),
			'lastmod' => gmdate('c'),
		];

		$public_post_types = get_post_types(['public' => true], 'names');
		$post_types = [];
		foreach ($include_post_types as $post_type) {
			$post_type = sanitize_key((string) $post_type);
			if (isset($public_post_types[$post_type])) {
				$post_types[] = $post_type;
			}
		}

		$post_types = array_values(array_unique($post_types));

		$posts = [];
			if (! empty($post_types)) {
				$posts = get_posts([
					'post_type'      => $post_types,
					'post_status'    => 'publish',
				'posts_per_page' => $limit,
				'orderby'        => 'modified',
				'order'          => 'DESC',
					'fields'         => 'ids',
					'no_found_rows'  => true,
					'post__not_in'   => $exclude_post_ids,
					'meta_query'     => [
						'relation' => 'OR',
						[
							'key'     => '_wpzeo_exclude_sitemap',
							'compare' => 'NOT EXISTS',
						],
						[
							'key'     => '_wpzeo_exclude_sitemap',
							'value'   => '1',
							'compare' => '!=',
						],
					],
				]);
			}

			foreach ($posts as $post_id) {
				if (in_array((int) $post_id, $exclude_post_ids, true)) {
					continue;
				}
				if ('1' === (string) get_post_meta((int) $post_id, '_wpzeo_exclude_sitemap', true)) {
					continue;
				}
				$link = get_permalink($post_id);
				if (! $link) {
					continue;
			}
			$urls[] = [
				'loc'     => $link,
				'lastmod' => get_post_modified_time('c', true, $post_id),
			];
		}

		$public_taxonomies = get_taxonomies(['public' => true], 'names');
		$taxonomies = [];
		foreach ($include_taxonomies as $taxonomy) {
			$taxonomy = sanitize_key((string) $taxonomy);
			if (isset($public_taxonomies[$taxonomy])) {
				$taxonomies[] = $taxonomy;
			}
		}
		$taxonomies = array_values(array_unique($taxonomies));

		foreach ($taxonomies as $taxonomy) {
			$terms = get_terms([
				'taxonomy'   => $taxonomy,
				'hide_empty' => true,
				'number'     => min(200, $limit),
				'exclude'    => $exclude_term_ids,
			]);

			if (is_wp_error($terms)) {
				continue;
			}

			foreach ($terms as $term) {
				if (in_array((int) $term->term_id, $exclude_term_ids, true)) {
					continue;
				}
				$link = get_term_link($term);
				if (is_wp_error($link)) {
					continue;
				}
				$urls[] = [
					'loc'     => (string) $link,
					'lastmod' => gmdate('c'),
				];
			}
		}

		if ($include_authors) {
			$authors = get_users([
				'who'    => 'authors',
				'number' => min(200, $limit),
				'fields' => ['ID'],
			]);

			foreach ($authors as $author) {
				$urls[] = [
					'loc'     => get_author_posts_url((int) $author->ID),
					'lastmod' => gmdate('c'),
				];
			}
		}

		$unique = [];
		foreach ($urls as $url_item) {
			if (empty($url_item['loc'])) {
				continue;
			}
			$key = (string) $url_item['loc'];
			$unique[$key] = $url_item;
		}

		return array_slice(array_values($unique), 0, $limit);
	}

	/**
	 * @param array<int, array<string, string>> $urls
	 * @return string
	 */
	private function build_xml($urls)
	{
		$xml  = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
		$xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";

		foreach ($urls as $url) {
			$loc     = isset($url['loc']) ? esc_url_raw((string) $url['loc']) : '';
			$lastmod = isset($url['lastmod']) ? sanitize_text_field((string) $url['lastmod']) : '';

			if ('' === $loc) {
				continue;
			}

			$xml .= "  <url>\n";
			$xml .= '    <loc>' . esc_html($loc) . "</loc>\n";
			if ('' !== $lastmod) {
				$xml .= '    <lastmod>' . esc_html($lastmod) . "</lastmod>\n";
			}
			$xml .= "  </url>\n";
		}

		$xml .= "</urlset>\n";
		return $xml;
	}
}
