<?php

namespace WPZEO\Modules;

use WPZEO\Core\Settings;

if (! defined('ABSPATH')) {
	exit;
}

class Robots
{
	/**
	 * @var Settings
	 */
	private $settings;

	/**
	 * @param Settings $settings
	 */
	public function __construct(Settings $settings)
	{
		$this->settings = $settings;
		add_action('rest_api_init', [$this, 'register_rest_routes']);
		add_filter('robots_txt', [$this, 'filter_robots_txt'], 10, 2);
	}

	public function register_rest_routes()
	{
		register_rest_route(
			'wpzeo/v1',
			'/robots-settings',
			[
				[
					'methods'             => 'GET',
					'callback'            => [$this, 'get_robots_settings'],
					'permission_callback' => [$this, 'can_manage_settings'],
				],
				[
					'methods'             => 'POST',
					'callback'            => [$this, 'update_robots_settings'],
					'permission_callback' => [$this, 'can_manage_settings'],
				],
			]
		);
	}

	/**
	 * @return bool
	 */
	public function can_manage_settings()
	{
		return current_user_can('manage_options');
	}

	/**
	 * @return \WP_REST_Response
	 */
	public function get_robots_settings()
	{
		return new \WP_REST_Response(
			[
				'settings' => $this->settings->get_robots(),
			],
			200
		);
	}

	/**
	 * Payload keys:
	 * custom_rules
	 *
	 * @param \WP_REST_Request $request
	 * @return \WP_REST_Response
	 */
	public function update_robots_settings($request)
	{
		$current = $this->settings->get_robots();
		$input   = $request->get_json_params();
		$input   = is_array($input) ? $input : [];

		$custom_rules = array_key_exists('custom_rules', $input)
			? sanitize_textarea_field((string) $input['custom_rules'])
			: (string) $current['custom_rules'];

		$payload = [
			'custom_rules' => $custom_rules,
		];

		update_option(Settings::OPTION_ROBOTS, $payload, false);

		return new \WP_REST_Response(
			[
				'settings' => $this->settings->get_robots(),
			],
			200
		);
	}

	/**
	 * @param string $output
	 * @param bool $public
	 * @return string
	 */
	public function filter_robots_txt($output, $public)
	{
		$settings = $this->settings->get_robots();
		$rules    = trim((string) $settings['custom_rules']);
		$sitemap  = "\nSitemap: " . home_url('/wpzeo-sitemap.xml/') . "\n";

		if ('' === $rules) {
			if (false !== strpos($output, '/wpzeo-sitemap.xml')) {
				return $output;
			}
			return rtrim($output) . $sitemap;
		}

		$append = "\n# wpZeo custom rules\n" . $rules . "\n";
		$final  = rtrim($output);
		if (false === strpos($final, '/wpzeo-sitemap.xml')) {
			$final .= $sitemap;
		}
		return $final . $append;
	}
}
