<?php

namespace WPZEO\Modules;

use WPZEO\Core\Settings;

if (! defined('ABSPATH')) {
	exit;
}

class Meta
{
	/**
	 * @var Settings
	 */
	private $settings;
	/**
	 * @var bool
	 */
	private $defer_to_external_seo = false;

	/**
	 * @param Settings $settings
	 */
	public function __construct(Settings $settings)
	{
		$this->settings = $settings;
		$this->defer_to_external_seo = $this->should_defer_to_external_seo();
		if (! $this->defer_to_external_seo) {
			remove_action('wp_head', 'rel_canonical');
		}
		add_action('rest_api_init', [$this, 'register_rest_routes']);

		add_action('add_meta_boxes', [$this, 'register_post_metabox']);
		add_action('save_post', [$this, 'save_post_meta']);

		add_action('init', [$this, 'register_term_hooks']);

		add_action('show_user_profile', [$this, 'render_author_fields']);
		add_action('edit_user_profile', [$this, 'render_author_fields']);
		add_action('personal_options_update', [$this, 'save_author_fields']);
		add_action('edit_user_profile_update', [$this, 'save_author_fields']);

		add_filter('document_title_parts', [$this, 'filter_document_title'], 20);
		add_filter('get_the_archive_description', [$this, 'filter_archive_description'], 20);
		add_filter('wp_robots', [$this, 'filter_wp_robots'], 20);
		add_action('wp_head', [$this, 'output_meta_tags'], 1);
	}

	public function register_rest_routes()
	{
		register_rest_route(
			'wpzeo/v1',
			'/meta/(?P<object_type>post|term|author)/(?P<object_id>[\d]+)',
			[
				[
					'methods'             => 'GET',
					'callback'            => [$this, 'rest_get_meta'],
					'permission_callback' => [$this, 'can_manage_settings'],
				],
				[
					'methods'             => 'POST',
					'callback'            => [$this, 'rest_update_meta'],
					'permission_callback' => [$this, 'can_manage_settings'],
				],
			]
		);
	}

	/**
	 * @return bool
	 */
	public function can_manage_settings()
	{
		return current_user_can('manage_options');
	}

	/**
	 * @param \WP_REST_Request $request
	 * @return \WP_REST_Response
	 */
	public function rest_get_meta($request)
	{
		$object_type = (string) $request->get_param('object_type');
		$object_id   = (int) $request->get_param('object_id');
		$include_defaults = (bool) $request->get_param('include_defaults');

		$stored = $this->get_object_meta($object_type, $object_id);
		if (is_wp_error($stored)) {
			return new \WP_REST_Response(
				[
					'error'   => $stored->get_error_code(),
					'message' => $stored->get_error_message(),
				],
				404
			);
		}

		$response = [
			'object_type' => $object_type,
			'object_id'   => $object_id,
			'meta'        => $stored,
		];

		if ($include_defaults) {
			$defaults = $this->get_object_default_meta($object_type, $object_id);
			$response['defaults'] = $defaults;
			$response['resolved'] = $this->merge_meta_with_defaults($stored, $defaults);
		}

		return new \WP_REST_Response(
			$response,
			200
		);
	}

	/**
	 * Accepted keys:
	 * title, description, focus_keyword, robots, canonical, og_image, og_title, og_description,
	 * twitter_title, twitter_description, twitter_image, robots_advanced, exclude_sitemap,
	 * content, content_post_id (term/author)
	 *
	 * @param \WP_REST_Request $request
	 * @return \WP_REST_Response
	 */
	public function rest_update_meta($request)
	{
		$object_type = (string) $request->get_param('object_type');
		$object_id   = (int) $request->get_param('object_id');

		$exists = $this->get_object_meta($object_type, $object_id);
		if (is_wp_error($exists)) {
			return new \WP_REST_Response(
				[
					'error'   => $exists->get_error_code(),
					'message' => $exists->get_error_message(),
				],
				404
			);
		}

		$title       = $request->get_param('title');
		$description = $request->get_param('description');
		$focus_keyword = $request->get_param('focus_keyword');
		$robots      = $request->get_param('robots');
		$canonical   = $request->get_param('canonical');
		$og_image    = $request->get_param('og_image');
		$og_title    = $request->get_param('og_title');
		$og_description = $request->get_param('og_description');
		$twitter_title = $request->get_param('twitter_title');
		$twitter_description = $request->get_param('twitter_description');
		$twitter_image = $request->get_param('twitter_image');
		$robots_advanced = $request->get_param('robots_advanced');
		$exclude_sitemap = $request->get_param('exclude_sitemap');
		$content = $request->get_param('content');
		$content_post_id = $request->get_param('content_post_id');

		$payload = [];
		if (null !== $title) {
			$payload['title'] = sanitize_text_field((string) $title);
		}
			if (null !== $description) {
				$payload['description'] = sanitize_textarea_field((string) $description);
			}
			if (null !== $focus_keyword) {
				$payload['focus_keyword'] = sanitize_text_field((string) $focus_keyword);
			}
			if (null !== $robots) {
				$payload['robots'] = sanitize_text_field((string) $robots);
			}
		if (null !== $canonical) {
			$payload['canonical'] = esc_url_raw((string) $canonical);
		}
			if (null !== $og_image) {
				$payload['og_image'] = esc_url_raw((string) $og_image);
			}
			if ('post' === $object_type && null !== $og_title) {
				$payload['og_title'] = sanitize_text_field((string) $og_title);
			}
			if ('post' === $object_type && null !== $og_description) {
				$payload['og_description'] = sanitize_textarea_field((string) $og_description);
			}
			if ('post' === $object_type && null !== $twitter_title) {
				$payload['twitter_title'] = sanitize_text_field((string) $twitter_title);
			}
			if ('post' === $object_type && null !== $twitter_description) {
				$payload['twitter_description'] = sanitize_textarea_field((string) $twitter_description);
			}
			if ('post' === $object_type && null !== $twitter_image) {
				$payload['twitter_image'] = esc_url_raw((string) $twitter_image);
			}
			if ('post' === $object_type && null !== $robots_advanced) {
				$payload['robots_advanced'] = $this->sanitize_advanced_robots((string) $robots_advanced);
			}
			if ('post' === $object_type && null !== $exclude_sitemap) {
				$payload['exclude_sitemap'] = ! empty($exclude_sitemap) ? '1' : '';
			}
			if (('term' === $object_type || 'author' === $object_type) && null !== $content) {
				$payload['content'] = wp_kses_post((string) $content);
			}
		if (('term' === $object_type || 'author' === $object_type) && null !== $content_post_id) {
			$payload['content_post_id'] = (string) absint($content_post_id);
		}

		if (isset($payload['robots']) && ! in_array($payload['robots'], ['', 'index,follow', 'noindex,follow', 'index,nofollow', 'noindex,nofollow'], true)) {
			$payload['robots'] = '';
		}

		$this->update_object_meta($object_type, $object_id, $payload);

		return new \WP_REST_Response(
			[
				'object_type' => $object_type,
				'object_id'   => $object_id,
				'meta'        => $this->get_object_meta($object_type, $object_id),
			],
			200
		);
	}

	/**
	 * @param string $object_type
	 * @param int $object_id
	 * @return array<string, string>|\WP_Error
	 */
	private function get_object_meta($object_type, $object_id)
	{
		if ($object_id <= 0) {
			return new \WP_Error('invalid_object', 'Invalid object id.');
		}

		if ('post' === $object_type) {
			$post = get_post($object_id);
			if (! $post) {
				return new \WP_Error('post_not_found', 'Post not found.');
			}

				return [
					'title'       => (string) get_post_meta($object_id, '_wpzeo_title', true),
					'description' => (string) get_post_meta($object_id, '_wpzeo_description', true),
					'focus_keyword' => (string) get_post_meta($object_id, '_wpzeo_focus_keyword', true),
					'robots'      => (string) get_post_meta($object_id, '_wpzeo_robots', true),
					'canonical'   => (string) get_post_meta($object_id, '_wpzeo_canonical', true),
					'og_image'    => (string) get_post_meta($object_id, '_wpzeo_og_image', true),
					'og_title'    => (string) get_post_meta($object_id, '_wpzeo_og_title', true),
					'og_description' => (string) get_post_meta($object_id, '_wpzeo_og_description', true),
					'twitter_title' => (string) get_post_meta($object_id, '_wpzeo_twitter_title', true),
					'twitter_description' => (string) get_post_meta($object_id, '_wpzeo_twitter_description', true),
					'twitter_image' => (string) get_post_meta($object_id, '_wpzeo_twitter_image', true),
					'robots_advanced' => (string) get_post_meta($object_id, '_wpzeo_robots_advanced', true),
					'exclude_sitemap' => (string) get_post_meta($object_id, '_wpzeo_exclude_sitemap', true),
				];
			}

			if ('term' === $object_type) {
			$term = get_term($object_id);
			if (! $term || is_wp_error($term)) {
				return new \WP_Error('term_not_found', 'Term not found.');
			}

				return [
					'title'       => (string) get_term_meta($object_id, '_wpzeo_term_title', true),
					'description' => (string) get_term_meta($object_id, '_wpzeo_term_description', true),
					'focus_keyword' => (string) get_term_meta($object_id, '_wpzeo_term_focus_keyword', true),
					'robots'      => (string) get_term_meta($object_id, '_wpzeo_term_robots', true),
					'canonical'   => (string) get_term_meta($object_id, '_wpzeo_term_canonical', true),
					'og_image'    => (string) get_term_meta($object_id, '_wpzeo_term_og_image', true),
					'content'     => (string) get_term_meta($object_id, '_wpzeo_term_content', true),
					'content_post_id' => (string) get_term_meta($object_id, '_wpzeo_term_content_post_id', true),
				];
			}

		if ('author' === $object_type) {
			$user = get_user_by('id', $object_id);
			if (! $user) {
				return new \WP_Error('author_not_found', 'Author not found.');
			}

				return [
					'title'       => (string) get_user_meta($object_id, '_wpzeo_author_title', true),
					'description' => (string) get_user_meta($object_id, '_wpzeo_author_description', true),
					'focus_keyword' => (string) get_user_meta($object_id, '_wpzeo_author_focus_keyword', true),
					'robots'      => (string) get_user_meta($object_id, '_wpzeo_author_robots', true),
					'canonical'   => (string) get_user_meta($object_id, '_wpzeo_author_canonical', true),
					'og_image'    => (string) get_user_meta($object_id, '_wpzeo_author_og_image', true),
					'content'     => (string) get_user_meta($object_id, '_wpzeo_author_content', true),
					'content_post_id' => (string) get_user_meta($object_id, '_wpzeo_author_content_post_id', true),
				];
			}

		return new \WP_Error('invalid_object_type', 'Unsupported object type.');
	}

	/**
	 * @param string $object_type
	 * @param int $object_id
	 * @param array<string, string> $payload
	 */
	private function update_object_meta($object_type, $object_id, $payload)
	{
		if ('post' === $object_type) {
			if (array_key_exists('title', $payload)) {
				$this->save_meta_value($object_id, '_wpzeo_title', $payload['title'], 'post');
			}
				if (array_key_exists('description', $payload)) {
					$this->save_meta_value($object_id, '_wpzeo_description', $payload['description'], 'post');
				}
				if (array_key_exists('focus_keyword', $payload)) {
					$this->save_meta_value($object_id, '_wpzeo_focus_keyword', $payload['focus_keyword'], 'post');
				}
				if (array_key_exists('robots', $payload)) {
					$this->save_meta_value($object_id, '_wpzeo_robots', $payload['robots'], 'post');
				}
			if (array_key_exists('canonical', $payload)) {
				$this->save_meta_value($object_id, '_wpzeo_canonical', $payload['canonical'], 'post');
			}
				if (array_key_exists('og_image', $payload)) {
					$this->save_meta_value($object_id, '_wpzeo_og_image', $payload['og_image'], 'post');
				}
				if (array_key_exists('og_title', $payload)) {
					$this->save_meta_value($object_id, '_wpzeo_og_title', $payload['og_title'], 'post');
				}
				if (array_key_exists('og_description', $payload)) {
					$this->save_meta_value($object_id, '_wpzeo_og_description', $payload['og_description'], 'post');
				}
				if (array_key_exists('twitter_title', $payload)) {
					$this->save_meta_value($object_id, '_wpzeo_twitter_title', $payload['twitter_title'], 'post');
				}
				if (array_key_exists('twitter_description', $payload)) {
					$this->save_meta_value($object_id, '_wpzeo_twitter_description', $payload['twitter_description'], 'post');
				}
				if (array_key_exists('twitter_image', $payload)) {
					$this->save_meta_value($object_id, '_wpzeo_twitter_image', $payload['twitter_image'], 'post');
				}
				if (array_key_exists('robots_advanced', $payload)) {
					$this->save_meta_value($object_id, '_wpzeo_robots_advanced', $payload['robots_advanced'], 'post');
				}
				if (array_key_exists('exclude_sitemap', $payload)) {
					$this->save_meta_value($object_id, '_wpzeo_exclude_sitemap', $payload['exclude_sitemap'], 'post');
				}
				return;
			}

			if ('term' === $object_type) {
			if (array_key_exists('title', $payload)) {
				$this->save_meta_value($object_id, '_wpzeo_term_title', $payload['title'], 'term');
			}
				if (array_key_exists('description', $payload)) {
					$this->save_meta_value($object_id, '_wpzeo_term_description', $payload['description'], 'term');
				}
				if (array_key_exists('focus_keyword', $payload)) {
					$this->save_meta_value($object_id, '_wpzeo_term_focus_keyword', $payload['focus_keyword'], 'term');
				}
				if (array_key_exists('robots', $payload)) {
					$this->save_meta_value($object_id, '_wpzeo_term_robots', $payload['robots'], 'term');
			}
				if (array_key_exists('canonical', $payload)) {
					$this->save_meta_value($object_id, '_wpzeo_term_canonical', $payload['canonical'], 'term');
				}
				if (array_key_exists('og_image', $payload)) {
					$this->save_meta_value($object_id, '_wpzeo_term_og_image', $payload['og_image'], 'term');
				}
				if (array_key_exists('content', $payload)) {
					$this->save_meta_value($object_id, '_wpzeo_term_content', $payload['content'], 'term');
				}
			if (array_key_exists('content_post_id', $payload)) {
				$this->save_meta_value($object_id, '_wpzeo_term_content_post_id', $payload['content_post_id'], 'term');
			}
			return;
		}

			if ('author' === $object_type) {
			if (array_key_exists('title', $payload)) {
				$this->save_meta_value($object_id, '_wpzeo_author_title', $payload['title'], 'user');
			}
				if (array_key_exists('description', $payload)) {
					$this->save_meta_value($object_id, '_wpzeo_author_description', $payload['description'], 'user');
				}
				if (array_key_exists('focus_keyword', $payload)) {
					$this->save_meta_value($object_id, '_wpzeo_author_focus_keyword', $payload['focus_keyword'], 'user');
				}
				if (array_key_exists('robots', $payload)) {
					$this->save_meta_value($object_id, '_wpzeo_author_robots', $payload['robots'], 'user');
			}
				if (array_key_exists('canonical', $payload)) {
					$this->save_meta_value($object_id, '_wpzeo_author_canonical', $payload['canonical'], 'user');
				}
				if (array_key_exists('og_image', $payload)) {
					$this->save_meta_value($object_id, '_wpzeo_author_og_image', $payload['og_image'], 'user');
				}
				if (array_key_exists('content', $payload)) {
					$this->save_meta_value($object_id, '_wpzeo_author_content', $payload['content'], 'user');
				}
			if (array_key_exists('content_post_id', $payload)) {
				$this->save_meta_value($object_id, '_wpzeo_author_content_post_id', $payload['content_post_id'], 'user');
			}
		}
	}

	public function register_post_metabox()
	{
		$post_types = get_post_types(['public' => true], 'names');
		foreach ($post_types as $post_type) {
			add_meta_box(
				'wpzeo_meta_box',
				'wpZeo SEO',
				[$this, 'render_post_metabox'],
				$post_type,
				'normal',
				'default'
			);
		}
	}

	/**
	 * @param \WP_Post $post
	 */
	public function render_post_metabox($post)
	{
		wp_nonce_field('wpzeo_post_meta_nonce', 'wpzeo_post_meta_nonce');

		$title       = trim((string) get_post_meta($post->ID, '_wpzeo_title', true));
		$description = trim((string) get_post_meta($post->ID, '_wpzeo_description', true));
		$focus_keyword = trim((string) get_post_meta($post->ID, '_wpzeo_focus_keyword', true));
		$robots      = trim((string) get_post_meta($post->ID, '_wpzeo_robots', true));
		$canonical   = trim((string) get_post_meta($post->ID, '_wpzeo_canonical', true));
		$og_image    = trim((string) get_post_meta($post->ID, '_wpzeo_og_image', true));
		$og_title    = trim((string) get_post_meta($post->ID, '_wpzeo_og_title', true));
		$og_description = trim((string) get_post_meta($post->ID, '_wpzeo_og_description', true));
		$twitter_title = trim((string) get_post_meta($post->ID, '_wpzeo_twitter_title', true));
		$twitter_description = trim((string) get_post_meta($post->ID, '_wpzeo_twitter_description', true));
		$twitter_image = trim((string) get_post_meta($post->ID, '_wpzeo_twitter_image', true));
		$robots_advanced = trim((string) get_post_meta($post->ID, '_wpzeo_robots_advanced', true));
		$exclude_sitemap = (string) get_post_meta($post->ID, '_wpzeo_exclude_sitemap', true);

		if ('' === $title) {
			$title = wp_strip_all_tags((string) get_the_title($post->ID), true);
		}
		if ('' === $description) {
			$description = $this->get_default_post_description($post);
		}
		if ('' === $robots) {
			$general = $this->settings->get_general();
			$robots = isset($general['default_robots']) ? (string) $general['default_robots'] : 'index,follow';
		}
		if ('' === $canonical && 'auto-draft' !== $post->post_status) {
			$permalink = get_permalink($post->ID);
			if (is_string($permalink) && '' !== $permalink) {
				$canonical = $permalink;
			}
		}
		if ('' === $og_image) {
			$thumb = get_the_post_thumbnail_url($post->ID, 'full');
			if (is_string($thumb) && '' !== $thumb) {
				$og_image = $thumb;
			}
		}
		?>
		<p><label for="wpzeo_title"><strong>SEO title</strong></label><br>
		<input id="wpzeo_title" class="widefat" type="text" name="wpzeo_title" value="<?php echo esc_attr((string) $title); ?>"></p>

		<p><label for="wpzeo_description"><strong>Meta description</strong></label><br>
		<textarea id="wpzeo_description" class="widefat" rows="3" name="wpzeo_description"><?php echo esc_textarea((string) $description); ?></textarea></p>

		<p><label for="wpzeo_focus_keyword"><strong>Focus keyword</strong></label><br>
		<input id="wpzeo_focus_keyword" class="widefat" type="text" name="wpzeo_focus_keyword" value="<?php echo esc_attr((string) $focus_keyword); ?>"></p>

		<p><label for="wpzeo_robots"><strong>Robots</strong></label><br>
		<select id="wpzeo_robots" name="wpzeo_robots">
			<option value="" <?php selected($robots, ''); ?>>Default</option>
			<option value="index,follow" <?php selected($robots, 'index,follow'); ?>>index,follow</option>
			<option value="noindex,follow" <?php selected($robots, 'noindex,follow'); ?>>noindex,follow</option>
			<option value="index,nofollow" <?php selected($robots, 'index,nofollow'); ?>>index,nofollow</option>
			<option value="noindex,nofollow" <?php selected($robots, 'noindex,nofollow'); ?>>noindex,nofollow</option>
		</select></p>

		<p><label for="wpzeo_canonical"><strong>Canonical URL</strong></label><br>
		<input id="wpzeo_canonical" class="widefat" type="url" name="wpzeo_canonical" value="<?php echo esc_attr((string) $canonical); ?>"></p>

		<p><label for="wpzeo_og_image"><strong>Social image URL (OG/Twitter)</strong></label><br>
		<input id="wpzeo_og_image" class="widefat" type="url" name="wpzeo_og_image" value="<?php echo esc_attr((string) $og_image); ?>"></p>

		<p><label for="wpzeo_og_title"><strong>Open Graph title override</strong></label><br>
		<input id="wpzeo_og_title" class="widefat" type="text" name="wpzeo_og_title" value="<?php echo esc_attr((string) $og_title); ?>"></p>

		<p><label for="wpzeo_og_description"><strong>Open Graph description override</strong></label><br>
		<textarea id="wpzeo_og_description" class="widefat" rows="2" name="wpzeo_og_description"><?php echo esc_textarea((string) $og_description); ?></textarea></p>

		<p><label for="wpzeo_twitter_title"><strong>Twitter/X title override</strong></label><br>
		<input id="wpzeo_twitter_title" class="widefat" type="text" name="wpzeo_twitter_title" value="<?php echo esc_attr((string) $twitter_title); ?>"></p>

		<p><label for="wpzeo_twitter_description"><strong>Twitter/X description override</strong></label><br>
		<textarea id="wpzeo_twitter_description" class="widefat" rows="2" name="wpzeo_twitter_description"><?php echo esc_textarea((string) $twitter_description); ?></textarea></p>

		<p><label for="wpzeo_twitter_image"><strong>Twitter/X image override</strong></label><br>
		<input id="wpzeo_twitter_image" class="widefat" type="url" name="wpzeo_twitter_image" value="<?php echo esc_attr((string) $twitter_image); ?>"></p>

		<p><label for="wpzeo_robots_advanced"><strong>Advanced robots directives</strong></label><br>
		<input id="wpzeo_robots_advanced" class="widefat" type="text" name="wpzeo_robots_advanced" value="<?php echo esc_attr((string) $robots_advanced); ?>">
		<small>Comma separated, e.g. <code>noarchive,nosnippet,max-image-preview:large</code></small></p>

		<p><label><input type="checkbox" name="wpzeo_exclude_sitemap" value="1" <?php checked($exclude_sitemap, '1'); ?>> Exclude this post from wpZeo sitemap</label></p>
		<?php
	}

	/**
	 * @param \WP_Post $post
	 * @return string
	 */
	private function get_default_post_description($post)
	{
		$source = trim((string) $post->post_excerpt);
		if ('' === $source) {
			$source = wp_strip_all_tags((string) $post->post_content, true);
		}
		if ('' === $source) {
			return '';
		}

		$source = preg_replace('/\s+/', ' ', $source);
		$source = is_string($source) ? trim($source) : '';
		if ('' === $source) {
			return '';
		}

		if (function_exists('mb_substr')) {
			return (string) mb_substr($source, 0, 160);
		}

		return (string) substr($source, 0, 160);
	}

	/**
	 * @param string $object_type
	 * @param int $object_id
	 * @return array<string, string>
	 */
	private function get_object_default_meta($object_type, $object_id)
	{
		$default_robots = $this->get_default_robots_value();

		if ('post' === $object_type) {
			$post = get_post($object_id);
			if (! $post instanceof \WP_Post) {
				return [];
			}

			$title = wp_strip_all_tags((string) get_the_title($object_id), true);
			$description = $this->get_default_post_description($post);
			$canonical = '';
			if ('auto-draft' !== $post->post_status) {
				$permalink = get_permalink($object_id);
				if (is_string($permalink) && '' !== $permalink) {
					$canonical = $permalink;
				}
			}
			$image = get_the_post_thumbnail_url($object_id, 'full');
			$image = is_string($image) && '' !== $image ? $image : $this->get_default_social_image();

			return [
				'title' => $title,
				'description' => $description,
				'focus_keyword' => '',
				'robots' => $default_robots,
				'canonical' => $canonical,
				'og_image' => is_string($image) ? $image : '',
				'og_title' => '',
				'og_description' => '',
				'twitter_title' => '',
				'twitter_description' => '',
				'twitter_image' => '',
				'robots_advanced' => '',
				'exclude_sitemap' => '',
			];
		}

		if ('term' === $object_type) {
			$term = get_term($object_id);
			if (is_wp_error($term) || ! ($term instanceof \WP_Term)) {
				return [];
			}

			$link = get_term_link($term);
			$canonical = (! is_wp_error($link) && is_string($link)) ? $link : '';

			return [
				'title' => wp_strip_all_tags((string) $term->name, true),
				'description' => wp_strip_all_tags((string) term_description($term), true),
				'focus_keyword' => '',
				'robots' => $default_robots,
				'canonical' => $canonical,
				'og_image' => $this->get_default_social_image(),
				'content' => '',
				'content_post_id' => '',
			];
		}

		if ('author' === $object_type) {
			$user = get_user_by('id', $object_id);
			if (! $user instanceof \WP_User) {
				return [];
			}

			$url = get_author_posts_url($object_id);

			return [
				'title' => wp_strip_all_tags((string) $user->display_name, true),
				'description' => wp_strip_all_tags((string) get_the_author_meta('description', $object_id), true),
				'focus_keyword' => '',
				'robots' => $default_robots,
				'canonical' => is_string($url) ? $url : '',
				'og_image' => $this->get_default_social_image(),
				'content' => '',
				'content_post_id' => '',
			];
		}

		return [];
	}

	/**
	 * @param array<string, string> $stored
	 * @param array<string, string> $defaults
	 * @return array<string, string>
	 */
	private function merge_meta_with_defaults($stored, $defaults)
	{
		$resolved = $defaults;
		foreach ($stored as $key => $value) {
			if (! is_string($key)) {
				continue;
			}

			$trimmed = trim((string) $value);
			if ('' !== $trimmed || ! array_key_exists($key, $resolved)) {
				$resolved[$key] = (string) $value;
			}
		}

		return $resolved;
	}

	/**
	 * @return string
	 */
	private function get_default_robots_value()
	{
		$general = $this->settings->get_general();
		$value = isset($general['default_robots']) ? (string) $general['default_robots'] : 'index,follow';
		if (! in_array($value, ['', 'index,follow', 'noindex,follow', 'index,nofollow', 'noindex,nofollow'], true)) {
			return 'index,follow';
		}
		return $value;
	}

	/**
	 * @return string
	 */
	private function get_default_social_image()
	{
		$social = $this->settings->get_social();
		return isset($social['default_image']) ? trim((string) $social['default_image']) : '';
	}

	/**
	 * @param int $post_id
	 */
	public function save_post_meta($post_id)
	{
		if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
			return;
		}

		if (! isset($_POST['wpzeo_post_meta_nonce']) || ! wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['wpzeo_post_meta_nonce'])), 'wpzeo_post_meta_nonce')) {
			return;
		}

		if (! current_user_can('edit_post', $post_id)) {
			return;
		}

		$this->save_meta_value($post_id, '_wpzeo_title', isset($_POST['wpzeo_title']) ? sanitize_text_field(wp_unslash($_POST['wpzeo_title'])) : '', 'post');
		$this->save_meta_value($post_id, '_wpzeo_description', isset($_POST['wpzeo_description']) ? sanitize_textarea_field(wp_unslash($_POST['wpzeo_description'])) : '', 'post');
		$this->save_meta_value($post_id, '_wpzeo_focus_keyword', isset($_POST['wpzeo_focus_keyword']) ? sanitize_text_field(wp_unslash($_POST['wpzeo_focus_keyword'])) : '', 'post');
		$this->save_meta_value($post_id, '_wpzeo_robots', isset($_POST['wpzeo_robots']) ? sanitize_text_field(wp_unslash($_POST['wpzeo_robots'])) : '', 'post');
		$this->save_meta_value($post_id, '_wpzeo_canonical', isset($_POST['wpzeo_canonical']) ? esc_url_raw(wp_unslash($_POST['wpzeo_canonical'])) : '', 'post');
		$this->save_meta_value($post_id, '_wpzeo_og_image', isset($_POST['wpzeo_og_image']) ? esc_url_raw(wp_unslash($_POST['wpzeo_og_image'])) : '', 'post');
		$this->save_meta_value($post_id, '_wpzeo_og_title', isset($_POST['wpzeo_og_title']) ? sanitize_text_field(wp_unslash($_POST['wpzeo_og_title'])) : '', 'post');
		$this->save_meta_value($post_id, '_wpzeo_og_description', isset($_POST['wpzeo_og_description']) ? sanitize_textarea_field(wp_unslash($_POST['wpzeo_og_description'])) : '', 'post');
		$this->save_meta_value($post_id, '_wpzeo_twitter_title', isset($_POST['wpzeo_twitter_title']) ? sanitize_text_field(wp_unslash($_POST['wpzeo_twitter_title'])) : '', 'post');
		$this->save_meta_value($post_id, '_wpzeo_twitter_description', isset($_POST['wpzeo_twitter_description']) ? sanitize_textarea_field(wp_unslash($_POST['wpzeo_twitter_description'])) : '', 'post');
		$this->save_meta_value($post_id, '_wpzeo_twitter_image', isset($_POST['wpzeo_twitter_image']) ? esc_url_raw(wp_unslash($_POST['wpzeo_twitter_image'])) : '', 'post');
		$this->save_meta_value($post_id, '_wpzeo_robots_advanced', isset($_POST['wpzeo_robots_advanced']) ? $this->sanitize_advanced_robots((string) wp_unslash($_POST['wpzeo_robots_advanced'])) : '', 'post');
		$this->save_meta_value($post_id, '_wpzeo_exclude_sitemap', isset($_POST['wpzeo_exclude_sitemap']) ? '1' : '', 'post');
	}

	public function register_term_hooks()
	{
		$taxonomies = get_taxonomies(['public' => true], 'names');

		foreach ($taxonomies as $taxonomy) {
			add_action($taxonomy . '_add_form_fields', [$this, 'render_term_add_fields']);
			add_action($taxonomy . '_edit_form_fields', [$this, 'render_term_edit_fields']);
			add_action('created_' . $taxonomy, [$this, 'save_term_fields']);
			add_action('edited_' . $taxonomy, [$this, 'save_term_fields']);
		}
	}

	/**
	 * @param string $taxonomy
	 */
	public function render_term_add_fields($taxonomy)
	{
		wp_nonce_field('wpzeo_term_meta_nonce', 'wpzeo_term_meta_nonce');
		?>
		<div class="form-field">
			<label for="wpzeo_term_title">SEO title</label>
			<input type="text" name="wpzeo_term_title" id="wpzeo_term_title" value="">
		</div>
			<div class="form-field">
				<label for="wpzeo_term_description">Meta description</label>
				<textarea name="wpzeo_term_description" id="wpzeo_term_description" rows="3"></textarea>
			</div>
			<div class="form-field">
				<label for="wpzeo_term_focus_keyword">Focus keyword</label>
				<input type="text" name="wpzeo_term_focus_keyword" id="wpzeo_term_focus_keyword" value="">
			</div>
			<div class="form-field">
			<label for="wpzeo_term_robots">Robots</label>
			<select name="wpzeo_term_robots" id="wpzeo_term_robots">
				<option value="">Default</option>
				<option value="index,follow">index,follow</option>
				<option value="noindex,follow">noindex,follow</option>
				<option value="index,nofollow">index,nofollow</option>
				<option value="noindex,nofollow">noindex,nofollow</option>
			</select>
		</div>
			<div class="form-field">
				<label for="wpzeo_term_canonical">Canonical URL</label>
				<input type="url" name="wpzeo_term_canonical" id="wpzeo_term_canonical" value="">
			</div>
			<div class="form-field">
				<label for="wpzeo_term_og_image">Social image URL (OG/Twitter)</label>
				<input type="url" name="wpzeo_term_og_image" id="wpzeo_term_og_image" value="">
			</div>
			<div class="form-field term-description-wrap">
				<label for="wpzeo_term_content_editor">Archive content (classic editor)</label>
				<?php
				wp_editor(
					'',
					'wpzeo_term_content_editor',
					[
						'textarea_name' => 'wpzeo_term_content',
						'textarea_rows' => 8,
						'media_buttons' => true,
						'teeny'         => false,
					]
				);
				?>
				<p>This content is rendered directly on the taxonomy archive page.</p>
			</div>
			<div class="form-field">
				<label for="wpzeo_term_content_post_id">Archive content source (Post/Page ID)</label>
				<input type="number" min="1" name="wpzeo_term_content_post_id" id="wpzeo_term_content_post_id" value="">
				<p>Edit that post/page in the normal editor. Its content will be shown on this taxonomy archive page.</p>
			</div>
		<?php
	}

	/**
	 * @param \WP_Term $term
	 */
	public function render_term_edit_fields($term)
	{
		wp_nonce_field('wpzeo_term_meta_nonce', 'wpzeo_term_meta_nonce');
		$title       = get_term_meta($term->term_id, '_wpzeo_term_title', true);
		$description = get_term_meta($term->term_id, '_wpzeo_term_description', true);
		$focus_keyword = get_term_meta($term->term_id, '_wpzeo_term_focus_keyword', true);
		$term_content = get_term_meta($term->term_id, '_wpzeo_term_content', true);
			$robots      = get_term_meta($term->term_id, '_wpzeo_term_robots', true);
			$canonical   = get_term_meta($term->term_id, '_wpzeo_term_canonical', true);
			$og_image    = get_term_meta($term->term_id, '_wpzeo_term_og_image', true);
			$content_post_id = get_term_meta($term->term_id, '_wpzeo_term_content_post_id', true);
		?>
		<tr class="form-field">
			<th scope="row"><label for="wpzeo_term_title">SEO title</label></th>
			<td><input type="text" name="wpzeo_term_title" id="wpzeo_term_title" value="<?php echo esc_attr((string) $title); ?>"></td>
		</tr>
			<tr class="form-field">
				<th scope="row"><label for="wpzeo_term_description">Meta description</label></th>
				<td><textarea name="wpzeo_term_description" id="wpzeo_term_description" rows="3"><?php echo esc_textarea((string) $description); ?></textarea></td>
			</tr>
			<tr class="form-field">
				<th scope="row"><label for="wpzeo_term_focus_keyword">Focus keyword</label></th>
				<td><input type="text" name="wpzeo_term_focus_keyword" id="wpzeo_term_focus_keyword" value="<?php echo esc_attr((string) $focus_keyword); ?>"></td>
			</tr>
			<tr class="form-field">
			<th scope="row"><label for="wpzeo_term_robots">Robots</label></th>
			<td>
				<select name="wpzeo_term_robots" id="wpzeo_term_robots">
					<option value="" <?php selected($robots, ''); ?>>Default</option>
					<option value="index,follow" <?php selected($robots, 'index,follow'); ?>>index,follow</option>
					<option value="noindex,follow" <?php selected($robots, 'noindex,follow'); ?>>noindex,follow</option>
					<option value="index,nofollow" <?php selected($robots, 'index,nofollow'); ?>>index,nofollow</option>
					<option value="noindex,nofollow" <?php selected($robots, 'noindex,nofollow'); ?>>noindex,nofollow</option>
				</select>
			</td>
		</tr>
			<tr class="form-field">
				<th scope="row"><label for="wpzeo_term_canonical">Canonical URL</label></th>
				<td><input type="url" name="wpzeo_term_canonical" id="wpzeo_term_canonical" value="<?php echo esc_attr((string) $canonical); ?>"></td>
			</tr>
			<tr class="form-field">
				<th scope="row"><label for="wpzeo_term_og_image">Social image URL (OG/Twitter)</label></th>
				<td><input type="url" name="wpzeo_term_og_image" id="wpzeo_term_og_image" value="<?php echo esc_attr((string) $og_image); ?>"></td>
			</tr>
			<tr class="form-field">
			<th scope="row"><label for="wpzeo_term_content_editor">Archive content (classic editor)</label></th>
			<td>
				<?php
				wp_editor(
					(string) $term_content,
					'wpzeo_term_content_editor',
					[
						'textarea_name' => 'wpzeo_term_content',
						'textarea_rows' => 8,
						'media_buttons' => true,
						'teeny'         => false,
					]
				);
				?>
				<p class="description">This content is rendered directly on the taxonomy archive page.</p>
			</td>
		</tr>
		<tr class="form-field">
			<th scope="row"><label for="wpzeo_term_content_post_id">Archive content source (Post/Page ID)</label></th>
			<td>
				<input type="number" min="1" name="wpzeo_term_content_post_id" id="wpzeo_term_content_post_id" value="<?php echo esc_attr((string) $content_post_id); ?>">
				<p class="description">Edit this post/page in the normal editor. Its content will be rendered on this taxonomy archive.</p>
			</td>
		</tr>
		<?php
	}

	/**
	 * @param int $term_id
	 */
	public function save_term_fields($term_id)
	{
		if (! isset($_POST['wpzeo_term_meta_nonce']) || ! wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['wpzeo_term_meta_nonce'])), 'wpzeo_term_meta_nonce')) {
			return;
		}

		$this->save_meta_value($term_id, '_wpzeo_term_title', isset($_POST['wpzeo_term_title']) ? sanitize_text_field(wp_unslash($_POST['wpzeo_term_title'])) : '', 'term');
		$this->save_meta_value($term_id, '_wpzeo_term_description', isset($_POST['wpzeo_term_description']) ? sanitize_textarea_field(wp_unslash($_POST['wpzeo_term_description'])) : '', 'term');
		$this->save_meta_value($term_id, '_wpzeo_term_focus_keyword', isset($_POST['wpzeo_term_focus_keyword']) ? sanitize_text_field(wp_unslash($_POST['wpzeo_term_focus_keyword'])) : '', 'term');
		$this->save_meta_value($term_id, '_wpzeo_term_robots', isset($_POST['wpzeo_term_robots']) ? sanitize_text_field(wp_unslash($_POST['wpzeo_term_robots'])) : '', 'term');
			$this->save_meta_value($term_id, '_wpzeo_term_canonical', isset($_POST['wpzeo_term_canonical']) ? esc_url_raw(wp_unslash($_POST['wpzeo_term_canonical'])) : '', 'term');
			$this->save_meta_value($term_id, '_wpzeo_term_og_image', isset($_POST['wpzeo_term_og_image']) ? esc_url_raw(wp_unslash($_POST['wpzeo_term_og_image'])) : '', 'term');
			$this->save_meta_value($term_id, '_wpzeo_term_content', isset($_POST['wpzeo_term_content']) ? wp_kses_post(wp_unslash($_POST['wpzeo_term_content'])) : '', 'term');
		$this->save_meta_value($term_id, '_wpzeo_term_content_post_id', isset($_POST['wpzeo_term_content_post_id']) ? (string) absint(wp_unslash($_POST['wpzeo_term_content_post_id'])) : '', 'term');
	}

	/**
	 * @param \WP_User $user
	 */
	public function render_author_fields($user)
	{
		if (! current_user_can('edit_user', $user->ID)) {
			return;
		}

		$title       = get_user_meta($user->ID, '_wpzeo_author_title', true);
		$description = get_user_meta($user->ID, '_wpzeo_author_description', true);
		$focus_keyword = get_user_meta($user->ID, '_wpzeo_author_focus_keyword', true);
		$author_content = get_user_meta($user->ID, '_wpzeo_author_content', true);
			$robots      = get_user_meta($user->ID, '_wpzeo_author_robots', true);
			$canonical   = get_user_meta($user->ID, '_wpzeo_author_canonical', true);
			$og_image    = get_user_meta($user->ID, '_wpzeo_author_og_image', true);
			$content_post_id = get_user_meta($user->ID, '_wpzeo_author_content_post_id', true);
		wp_nonce_field('wpzeo_author_meta_nonce', 'wpzeo_author_meta_nonce');
		?>
		<h2>wpZeo SEO</h2>
		<table class="form-table" role="presentation">
			<tr>
				<th><label for="wpzeo_author_title">SEO title</label></th>
				<td><input type="text" id="wpzeo_author_title" class="regular-text" name="wpzeo_author_title" value="<?php echo esc_attr((string) $title); ?>"></td>
			</tr>
				<tr>
					<th><label for="wpzeo_author_description">Meta description</label></th>
					<td><textarea id="wpzeo_author_description" class="large-text" name="wpzeo_author_description" rows="3"><?php echo esc_textarea((string) $description); ?></textarea></td>
				</tr>
				<tr>
					<th><label for="wpzeo_author_focus_keyword">Focus keyword</label></th>
					<td><input type="text" id="wpzeo_author_focus_keyword" class="regular-text" name="wpzeo_author_focus_keyword" value="<?php echo esc_attr((string) $focus_keyword); ?>"></td>
				</tr>
				<tr>
				<th><label for="wpzeo_author_robots">Robots</label></th>
				<td>
					<select id="wpzeo_author_robots" name="wpzeo_author_robots">
						<option value="" <?php selected($robots, ''); ?>>Default</option>
						<option value="index,follow" <?php selected($robots, 'index,follow'); ?>>index,follow</option>
						<option value="noindex,follow" <?php selected($robots, 'noindex,follow'); ?>>noindex,follow</option>
						<option value="index,nofollow" <?php selected($robots, 'index,nofollow'); ?>>index,nofollow</option>
						<option value="noindex,nofollow" <?php selected($robots, 'noindex,nofollow'); ?>>noindex,nofollow</option>
					</select>
				</td>
			</tr>
				<tr>
					<th><label for="wpzeo_author_canonical">Canonical URL</label></th>
					<td><input type="url" id="wpzeo_author_canonical" class="regular-text" name="wpzeo_author_canonical" value="<?php echo esc_attr((string) $canonical); ?>"></td>
				</tr>
				<tr>
					<th><label for="wpzeo_author_og_image">Social image URL (OG/Twitter)</label></th>
					<td><input type="url" id="wpzeo_author_og_image" class="regular-text" name="wpzeo_author_og_image" value="<?php echo esc_attr((string) $og_image); ?>"></td>
				</tr>
				<tr>
				<th><label for="wpzeo_author_content_editor">Archive content (classic editor)</label></th>
				<td>
					<?php
					wp_editor(
						(string) $author_content,
						'wpzeo_author_content_editor',
						[
							'textarea_name' => 'wpzeo_author_content',
							'textarea_rows' => 8,
							'media_buttons' => true,
							'teeny'         => false,
						]
					);
					?>
					<p class="description">This content is rendered directly on the author archive page.</p>
				</td>
			</tr>
			<tr>
				<th><label for="wpzeo_author_content_post_id">Archive content source (Post/Page ID)</label></th>
				<td>
					<input type="number" min="1" id="wpzeo_author_content_post_id" class="regular-text" name="wpzeo_author_content_post_id" value="<?php echo esc_attr((string) $content_post_id); ?>">
					<p class="description">Edit this post/page in the normal editor. Its content will be shown on this author archive page.</p>
				</td>
			</tr>
		</table>
		<?php
	}

	/**
	 * @param int $user_id
	 */
	public function save_author_fields($user_id)
	{
		if (! isset($_POST['wpzeo_author_meta_nonce']) || ! wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['wpzeo_author_meta_nonce'])), 'wpzeo_author_meta_nonce')) {
			return;
		}

		if (! current_user_can('edit_user', $user_id)) {
			return;
		}

		$this->save_meta_value($user_id, '_wpzeo_author_title', isset($_POST['wpzeo_author_title']) ? sanitize_text_field(wp_unslash($_POST['wpzeo_author_title'])) : '', 'user');
		$this->save_meta_value($user_id, '_wpzeo_author_description', isset($_POST['wpzeo_author_description']) ? sanitize_textarea_field(wp_unslash($_POST['wpzeo_author_description'])) : '', 'user');
		$this->save_meta_value($user_id, '_wpzeo_author_focus_keyword', isset($_POST['wpzeo_author_focus_keyword']) ? sanitize_text_field(wp_unslash($_POST['wpzeo_author_focus_keyword'])) : '', 'user');
		$this->save_meta_value($user_id, '_wpzeo_author_robots', isset($_POST['wpzeo_author_robots']) ? sanitize_text_field(wp_unslash($_POST['wpzeo_author_robots'])) : '', 'user');
			$this->save_meta_value($user_id, '_wpzeo_author_canonical', isset($_POST['wpzeo_author_canonical']) ? esc_url_raw(wp_unslash($_POST['wpzeo_author_canonical'])) : '', 'user');
			$this->save_meta_value($user_id, '_wpzeo_author_og_image', isset($_POST['wpzeo_author_og_image']) ? esc_url_raw(wp_unslash($_POST['wpzeo_author_og_image'])) : '', 'user');
			$this->save_meta_value($user_id, '_wpzeo_author_content', isset($_POST['wpzeo_author_content']) ? wp_kses_post(wp_unslash($_POST['wpzeo_author_content'])) : '', 'user');
		$this->save_meta_value($user_id, '_wpzeo_author_content_post_id', isset($_POST['wpzeo_author_content_post_id']) ? (string) absint(wp_unslash($_POST['wpzeo_author_content_post_id'])) : '', 'user');
	}

	/**
	 * @param array<string, string> $parts
	 * @return array<string, string>
	 */
	public function filter_document_title($parts)
	{
		if ($this->defer_to_external_seo) {
			return $parts;
		}

		$title = $this->get_context_title();
		if ('' !== $title) {
			$parts['title'] = $title;
		}
		return $parts;
	}

	/**
	 * @param string $description
	 * @return string
	 */
	public function filter_archive_description($description)
	{
		if (is_admin() || is_feed()) {
			return $description;
		}

		$source_post_id = 0;
		if (is_category() || is_tag() || is_tax()) {
			$term = get_queried_object();
			if ($term instanceof \WP_Term) {
				$direct_content = (string) get_term_meta($term->term_id, '_wpzeo_term_content', true);
				if ('' !== trim(wp_strip_all_tags($direct_content, true))) {
					return apply_filters('the_content', $direct_content);
				}
				$source_post_id = (int) get_term_meta($term->term_id, '_wpzeo_term_content_post_id', true);
			}
		} elseif (is_author()) {
			$author_id = (int) get_query_var('author');
			$direct_content = (string) get_user_meta($author_id, '_wpzeo_author_content', true);
			if ('' !== trim(wp_strip_all_tags($direct_content, true))) {
				return apply_filters('the_content', $direct_content);
			}
			$source_post_id = (int) get_user_meta($author_id, '_wpzeo_author_content_post_id', true);
		}

		if ($source_post_id <= 0) {
			return $description;
		}

		$post = get_post($source_post_id);
		if (! $post || 'publish' !== $post->post_status) {
			return $description;
		}

		return apply_filters('the_content', $post->post_content);
	}

	public function output_meta_tags()
	{
		if ($this->defer_to_external_seo || is_admin() || ! is_main_query()) {
			return;
		}

		$description = $this->get_context_description();
		$canonical   = $this->get_context_canonical();

		if ('' !== $description) {
			echo '<meta name="description" content="' . esc_attr($description) . '">' . "\n";
		}

		if ('' !== $canonical) {
			echo '<link rel="canonical" href="' . esc_url($canonical) . '">' . "\n";
		}
	}

	/**
	 * @param array<string, mixed> $robots
	 * @return array<string, mixed>
	 */
	public function filter_wp_robots($robots)
	{
		if ($this->defer_to_external_seo || is_admin() || ! is_main_query()) {
			return $robots;
		}

		$value = $this->get_context_robots();
		if ('' === $value) {
			return $robots;
		}

		$tokens = array_map('trim', explode(',', strtolower($value)));
		$tokens = array_filter($tokens, static function ($token) {
			return '' !== $token;
		});

		if (in_array('noindex', $tokens, true)) {
			$robots['noindex'] = true;
			unset($robots['index']);
		} elseif (in_array('index', $tokens, true)) {
			unset($robots['noindex']);
		}

		if (in_array('nofollow', $tokens, true)) {
			$robots['nofollow'] = true;
			unset($robots['follow']);
		} elseif (in_array('follow', $tokens, true)) {
			unset($robots['nofollow']);
		}

		if (in_array('noarchive', $tokens, true)) {
			$robots['noarchive'] = true;
		}

		$advanced_directives = $this->parse_advanced_robots($this->get_context_robots_advanced());
		foreach ($advanced_directives as $directive) {
			if (in_array($directive, ['noarchive', 'nosnippet', 'noimageindex', 'notranslate', 'nositelinkssearchbox'], true)) {
				$robots[$directive] = true;
				continue;
			}

			if (0 === strpos($directive, 'max-snippet:')) {
				$robots['max-snippet'] = substr($directive, 12);
				continue;
			}
			if (0 === strpos($directive, 'max-image-preview:')) {
				$robots['max-image-preview'] = substr($directive, 18);
				continue;
			}
			if (0 === strpos($directive, 'max-video-preview:')) {
				$robots['max-video-preview'] = substr($directive, 18);
			}
		}

		return $robots;
	}

	/**
	 * @return string
	 */
	private function get_context_robots_advanced()
	{
		if (is_singular()) {
			$post_id = get_queried_object_id();
			return trim((string) get_post_meta($post_id, '_wpzeo_robots_advanced', true));
		}

		return '';
	}

	/**
	 * @param string $input
	 * @return string
	 */
	private function sanitize_advanced_robots($input)
	{
		$tokens = $this->parse_advanced_robots($input);
		return implode(',', $tokens);
	}

	/**
	 * @param string $input
	 * @return array<int, string>
	 */
	private function parse_advanced_robots($input)
	{
		$input = strtolower(trim((string) $input));
		if ('' === $input) {
			return [];
		}

		$raw_tokens = preg_split('/[\s,]+/', $input);
		$raw_tokens = is_array($raw_tokens) ? $raw_tokens : [];
		$allowed_simple = ['noarchive', 'nosnippet', 'noimageindex', 'notranslate', 'nositelinkssearchbox'];
		$valid = [];

		foreach ($raw_tokens as $token) {
			$token = trim((string) $token);
			if ('' === $token) {
				continue;
			}

			if (in_array($token, $allowed_simple, true)) {
				$valid[] = $token;
				continue;
			}

			if (preg_match('/^max-snippet:-?\d+$/', $token)) {
				$valid[] = $token;
				continue;
			}

			if (preg_match('/^max-video-preview:-?\d+$/', $token)) {
				$valid[] = $token;
				continue;
			}

			if (preg_match('/^max-image-preview:(none|standard|large)$/', $token)) {
				$valid[] = $token;
			}
		}

		return array_values(array_unique($valid));
	}

	/**
	 * @return string
	 */
	public function get_context_title()
	{
		if (is_singular()) {
			$post_id = get_queried_object_id();
			$value   = (string) get_post_meta($post_id, '_wpzeo_title', true);
			$value = trim($value);
			if ('' !== $value) {
				return $value;
			}

			$post = get_post($post_id);
			if ($post instanceof \WP_Post) {
				$template = $this->get_post_type_template('title', (string) $post->post_type);
				$resolved = $this->resolve_template($template, $this->get_post_template_vars($post_id, $post));
				if ('' !== $resolved) {
					return $resolved;
				}
			}

			return wp_strip_all_tags((string) get_the_title($post_id), true);
		}

		if (is_category() || is_tag() || is_tax()) {
			$term = get_queried_object();
			if ($term instanceof \WP_Term) {
				$value = (string) get_term_meta($term->term_id, '_wpzeo_term_title', true);
				$value = trim($value);
				if ('' !== $value) {
					return $value;
				}

				$template = $this->get_taxonomy_template('title', (string) $term->taxonomy);
				$resolved = $this->resolve_template($template, $this->get_term_template_vars($term));
				if ('' !== $resolved) {
					return $resolved;
				}

				return wp_strip_all_tags((string) single_term_title('', false), true);
			}
		}

		if (is_author()) {
			$author_id = (int) get_query_var('author');
			$value     = (string) get_user_meta($author_id, '_wpzeo_author_title', true);
			$value = trim($value);
			if ('' !== $value) {
				return $value;
			}

			$template = $this->get_author_template('title');
			$resolved = $this->resolve_template($template, $this->get_author_template_vars($author_id));
			if ('' !== $resolved) {
				return $resolved;
			}

			$user = get_userdata($author_id);
			if ($user) {
				return wp_strip_all_tags((string) $user->display_name, true);
			}
		}

		return '';
	}

	/**
	 * @return string
	 */
	private function get_context_description()
	{
		if (is_singular()) {
			$post_id = get_queried_object_id();
			$value   = trim((string) get_post_meta($post_id, '_wpzeo_description', true));
			if ('' !== $value) {
				return $value;
			}

			$post = get_post($post_id);
			if ($post instanceof \WP_Post) {
				$template = $this->get_post_type_template('description', (string) $post->post_type);
				$resolved = $this->resolve_template($template, $this->get_post_template_vars($post_id, $post));
				if ('' !== $resolved) {
					return $resolved;
				}
			}

			return wp_strip_all_tags((string) get_the_excerpt($post_id), true);
		}

		if (is_category() || is_tag() || is_tax()) {
			$term = get_queried_object();
			if ($term instanceof \WP_Term) {
				$value = trim((string) get_term_meta($term->term_id, '_wpzeo_term_description', true));
				if ('' !== $value) {
					return $value;
				}

				$template = $this->get_taxonomy_template('description', (string) $term->taxonomy);
				$resolved = $this->resolve_template($template, $this->get_term_template_vars($term));
				if ('' !== $resolved) {
					return $resolved;
				}

				$source_post_id = (int) get_term_meta($term->term_id, '_wpzeo_term_content_post_id', true);
				if ($source_post_id > 0) {
					$source_excerpt = wp_strip_all_tags((string) get_the_excerpt($source_post_id), true);
					if ('' !== $source_excerpt) {
						return $source_excerpt;
					}
				}

				$term_desc = wp_strip_all_tags((string) term_description($term), true);
				if ('' !== $term_desc) {
					return $term_desc;
				}

				return wp_strip_all_tags((string) single_term_title('', false), true);
			}
		}

		if (is_author()) {
			$author_id = (int) get_query_var('author');
			$value     = trim((string) get_user_meta($author_id, '_wpzeo_author_description', true));
			if ('' !== $value) {
				return $value;
			}

			$template = $this->get_author_template('description');
			$resolved = $this->resolve_template($template, $this->get_author_template_vars($author_id));
			if ('' !== $resolved) {
				return $resolved;
			}

			$source_post_id = (int) get_user_meta($author_id, '_wpzeo_author_content_post_id', true);
			if ($source_post_id > 0) {
				$source_excerpt = wp_strip_all_tags((string) get_the_excerpt($source_post_id), true);
				if ('' !== $source_excerpt) {
					return $source_excerpt;
				}
			}

			$user = get_userdata($author_id);
			if ($user) {
				$author_desc = wp_strip_all_tags((string) get_the_author_meta('description', $user->ID), true);
				if ('' !== $author_desc) {
					return $author_desc;
				}

				return wp_strip_all_tags((string) $user->display_name, true);
			}
		}

		return wp_strip_all_tags((string) get_bloginfo('description'), true);
	}

	/**
	 * @param string $kind title|description
	 * @param string $post_type
	 * @return string
	 */
	private function get_post_type_template($kind, $post_type)
	{
		$general = $this->settings->get_general();
		$key = ('description' === $kind) ? 'description_template_post_types' : 'title_template_post_types';
		$templates = isset($general[$key]) && is_array($general[$key]) ? $general[$key] : [];

		return isset($templates[$post_type]) ? trim((string) $templates[$post_type]) : '';
	}

	/**
	 * @param string $kind title|description
	 * @param string $taxonomy
	 * @return string
	 */
	private function get_taxonomy_template($kind, $taxonomy)
	{
		$general = $this->settings->get_general();
		$key = ('description' === $kind) ? 'description_template_taxonomies' : 'title_template_taxonomies';
		$templates = isset($general[$key]) && is_array($general[$key]) ? $general[$key] : [];

		return isset($templates[$taxonomy]) ? trim((string) $templates[$taxonomy]) : '';
	}

	/**
	 * @param string $kind title|description
	 * @return string
	 */
	private function get_author_template($kind)
	{
		$general = $this->settings->get_general();
		$key = ('description' === $kind) ? 'description_template_author' : 'title_template_author';
		return isset($general[$key]) ? trim((string) $general[$key]) : '';
	}

	/**
	 * @param int $post_id
	 * @param \WP_Post $post
	 * @return array<string, string>
	 */
	private function get_post_template_vars($post_id, $post)
	{
		return [
			'%title%' => wp_strip_all_tags((string) get_the_title($post_id), true),
			'%sitename%' => wp_strip_all_tags((string) get_bloginfo('name'), true),
			'%excerpt%' => $this->get_post_excerpt_for_template($post_id, $post),
		];
	}

	/**
	 * @param \WP_Term $term
	 * @return array<string, string>
	 */
	private function get_term_template_vars($term)
	{
		return [
			'%term%' => wp_strip_all_tags((string) single_term_title('', false), true),
			'%term_description%' => wp_strip_all_tags((string) term_description($term), true),
			'%sitename%' => wp_strip_all_tags((string) get_bloginfo('name'), true),
		];
	}

	/**
	 * @param int $author_id
	 * @return array<string, string>
	 */
	private function get_author_template_vars($author_id)
	{
		$user = get_userdata($author_id);
		$name = $user ? (string) $user->display_name : '';
		$bio = $user ? (string) get_the_author_meta('description', $user->ID) : '';

		return [
			'%author%' => wp_strip_all_tags($name, true),
			'%author_bio%' => wp_strip_all_tags($bio, true),
			'%sitename%' => wp_strip_all_tags((string) get_bloginfo('name'), true),
		];
	}

	/**
	 * @param int $post_id
	 * @param \WP_Post $post
	 * @return string
	 */
	private function get_post_excerpt_for_template($post_id, $post)
	{
		$excerpt = trim(wp_strip_all_tags((string) get_the_excerpt($post_id), true));
		if ('' !== $excerpt) {
			return $excerpt;
		}

		$content = trim(wp_strip_all_tags((string) $post->post_content, true));
		if ('' === $content) {
			return '';
		}

		if (function_exists('mb_substr')) {
			return (string) mb_substr($content, 0, 160);
		}

		return (string) substr($content, 0, 160);
	}

	/**
	 * @param string $template
	 * @param array<string, string> $vars
	 * @return string
	 */
	private function resolve_template($template, $vars)
	{
		$template = trim((string) $template);
		if ('' === $template) {
			return '';
		}

		$resolved = str_ireplace(array_keys($vars), array_values($vars), $template);
		$resolved = wp_strip_all_tags((string) $resolved, true);
		$resolved = preg_replace('/\s+/', ' ', (string) $resolved);
		$resolved = is_string($resolved) ? trim($resolved) : '';
		return $resolved;
	}

	/**
	 * @return string
	 */
	private function get_context_robots()
	{
		$general = $this->settings->get_general();

		if (! empty($general['noindex_search']) && is_search()) {
			return 'noindex,follow';
		}

		if (is_singular()) {
			$post_id = get_queried_object_id();
			$value   = trim((string) get_post_meta($post_id, '_wpzeo_robots', true));
			return '' !== $value ? $value : (string) $general['default_robots'];
		}

		if (is_category() || is_tag() || is_tax()) {
			$term = get_queried_object();
			if ($term instanceof \WP_Term) {
				$value = trim((string) get_term_meta($term->term_id, '_wpzeo_term_robots', true));
				return '' !== $value ? $value : (string) $general['default_robots'];
			}
		}

		if (is_author()) {
			$author_id = (int) get_query_var('author');
			$value     = trim((string) get_user_meta($author_id, '_wpzeo_author_robots', true));
			return '' !== $value ? $value : (string) $general['default_robots'];
		}

		return (string) $general['default_robots'];
	}

	/**
	 * @return string
	 */
	private function get_context_canonical()
	{
		if (is_paged()) {
			return (string) get_pagenum_link(max(1, (int) get_query_var('paged')));
		}

		if (is_singular()) {
			$post_id = get_queried_object_id();
			$value   = trim((string) get_post_meta($post_id, '_wpzeo_canonical', true));
			return '' !== $value ? $value : (string) get_permalink($post_id);
		}

		if (is_category() || is_tag() || is_tax()) {
			$term = get_queried_object();
			if ($term instanceof \WP_Term) {
				$value = trim((string) get_term_meta($term->term_id, '_wpzeo_term_canonical', true));
				if ('' !== $value) {
					return $value;
				}
				$link = get_term_link($term);
				if (! is_wp_error($link)) {
					return (string) $link;
				}
			}
		}

		if (is_author()) {
			$author_id = (int) get_query_var('author');
			$value     = trim((string) get_user_meta($author_id, '_wpzeo_author_canonical', true));
			if ('' !== $value) {
				return $value;
			}
			return (string) get_author_posts_url($author_id);
		}

		if (is_home() || is_front_page()) {
			return home_url('/');
		}

		return '';
	}

	/**
	 * @return bool
	 */
	private function should_defer_to_external_seo()
	{
		$known_constants = [
			'WPSEO_VERSION',               // Yoast.
			'RANK_MATH_VERSION',           // Rank Math.
			'AIOSEO_VERSION',              // AIOSEO.
			'SEOPRESS_VERSION',            // SEOPress.
			'THE_SEO_FRAMEWORK_VERSION',   // The SEO Framework.
		];

		$detected = false;
		foreach ($known_constants as $constant_name) {
			if (defined($constant_name)) {
				$detected = true;
				break;
			}
		}

		/**
		 * Allow overriding automatic defer behavior.
		 *
		 * @param bool $defer Whether wpZeo should suppress frontend SEO output.
		 * @param bool $detected Whether a known SEO plugin was detected.
		 */
		return (bool) apply_filters('wpzeo/defer_to_external_seo', $detected, $detected);
	}

	/**
	 * @param int $object_id
	 * @param string $meta_key
	 * @param string $value
	 * @param string $type
	 */
	private function save_meta_value($object_id, $meta_key, $value, $type)
	{
		$value = trim($value);

		if ('' === $value) {
			if ('post' === $type) {
				delete_post_meta($object_id, $meta_key);
			} elseif ('term' === $type) {
				delete_term_meta($object_id, $meta_key);
			} elseif ('user' === $type) {
				delete_user_meta($object_id, $meta_key);
			}
			return;
		}

		if ('post' === $type) {
			update_post_meta($object_id, $meta_key, $value);
		} elseif ('term' === $type) {
			update_term_meta($object_id, $meta_key, $value);
		} elseif ('user' === $type) {
			update_user_meta($object_id, $meta_key, $value);
		}
	}
}
