<?php

namespace WPZEO\Core;

if (! defined('ABSPATH')) {
	exit;
}

class Settings
{
	const OPTION_MODULES = 'wpzeo_modules';
	const OPTION_GENERAL = 'wpzeo_settings_general';
	const OPTION_SITEMAP = 'wpzeo_settings_sitemap';
	const OPTION_SOCIAL = 'wpzeo_settings_social';
	const OPTION_ROBOTS = 'wpzeo_settings_robots';
	const OPTION_REDIRECTS = 'wpzeo_settings_redirects';
	const OPTION_REDIRECT_STATS = 'wpzeo_settings_redirect_stats';
	const OPTION_MIGRATION = 'wpzeo_settings_migration';
	const OPTION_LAST_MIGRATION_REPORT = 'wpzeo_last_migration_report';

	/**
	 * @return array<string, bool>
	 */
	public function get_modules()
	{
		$modules  = get_option(self::OPTION_MODULES, []);
		$defaults = $this->default_modules();
		$merged   = array_merge($defaults, is_array($modules) ? $modules : []);

		return [
			'meta'      => ! empty($merged['meta']),
			'sitemap'   => ! empty($merged['sitemap']),
			'robots'    => ! empty($merged['robots']),
			'social'    => ! empty($merged['social']),
			'redirects' => ! empty($merged['redirects']),
		];
	}

	/**
	 * @return array<string, mixed>
	 */
	public function get_general()
	{
		$defaults = [
			'default_robots' => 'index,follow',
			'noindex_search' => 1,
			'title_template_post_types' => $this->default_post_type_templates('title'),
			'description_template_post_types' => $this->default_post_type_templates('description'),
			'title_template_taxonomies' => $this->default_taxonomy_templates('title'),
			'description_template_taxonomies' => $this->default_taxonomy_templates('description'),
			'title_template_author' => '%author% | %sitename%',
			'description_template_author' => '%author_bio%',
		];

		$value = get_option(self::OPTION_GENERAL, []);
		$merged = array_merge($defaults, is_array($value) ? $value : []);

		// ZeroConfig fallback for existing installs with empty template sets.
		if (empty($merged['title_template_post_types']) || ! is_array($merged['title_template_post_types'])) {
			$merged['title_template_post_types'] = $defaults['title_template_post_types'];
		}
		if (empty($merged['description_template_post_types']) || ! is_array($merged['description_template_post_types'])) {
			$merged['description_template_post_types'] = $defaults['description_template_post_types'];
		}
		if (empty($merged['title_template_taxonomies']) || ! is_array($merged['title_template_taxonomies'])) {
			$merged['title_template_taxonomies'] = $defaults['title_template_taxonomies'];
		}
		if (empty($merged['description_template_taxonomies']) || ! is_array($merged['description_template_taxonomies'])) {
			$merged['description_template_taxonomies'] = $defaults['description_template_taxonomies'];
		}
		if (! isset($merged['title_template_author']) || '' === trim((string) $merged['title_template_author'])) {
			$merged['title_template_author'] = $defaults['title_template_author'];
		}
		if (! isset($merged['description_template_author']) || '' === trim((string) $merged['description_template_author'])) {
			$merged['description_template_author'] = $defaults['description_template_author'];
		}

		return $merged;
	}

	/**
	 * @return array<string, mixed>
	 */
	public function get_sitemap()
	{
		$public_post_types = get_post_types(['public' => true], 'names');
		$public_taxonomies = get_taxonomies(['public' => true], 'names');

		$defaults = [
			'enabled'         => 1,
			'include_authors' => 1,
			'limit'           => 1000,
			'include_post_types' => array_values($public_post_types),
			'include_taxonomies' => array_values($public_taxonomies),
			'exclude_post_ids'   => [],
			'exclude_term_ids'   => [],
		];

		$value = get_option(self::OPTION_SITEMAP, []);
		$merged = array_merge($defaults, is_array($value) ? $value : []);
		$merged['include_post_types'] = is_array($merged['include_post_types']) ? array_values($merged['include_post_types']) : $defaults['include_post_types'];
		$merged['include_taxonomies'] = is_array($merged['include_taxonomies']) ? array_values($merged['include_taxonomies']) : $defaults['include_taxonomies'];
		$merged['exclude_post_ids'] = is_array($merged['exclude_post_ids']) ? array_values(array_map('absint', $merged['exclude_post_ids'])) : [];
		$merged['exclude_term_ids'] = is_array($merged['exclude_term_ids']) ? array_values(array_map('absint', $merged['exclude_term_ids'])) : [];

		return $merged;
	}

	/**
	 * @return array<string, mixed>
	 */
	public function get_social()
	{
		$defaults = [
			'default_image' => '',
			'twitter_card'  => 'summary_large_image',
		];

		$value = get_option(self::OPTION_SOCIAL, []);
		return array_merge($defaults, is_array($value) ? $value : []);
	}

	/**
	 * @return array<string, mixed>
	 */
	public function get_robots()
	{
		$defaults = [
			'custom_rules' => '',
		];

		$value = get_option(self::OPTION_ROBOTS, []);
		return array_merge($defaults, is_array($value) ? $value : []);
	}

	/**
	 * @return array<string, mixed>
	 */
	public function get_migration()
	{
		$defaults = [
			'enabled'     => 0,
			'auto_enable' => 1,
		];

		$value = get_option(self::OPTION_MIGRATION, []);
		return array_merge($defaults, is_array($value) ? $value : []);
	}

	/**
	 * @return array<int, array<string, mixed>>
	 */
	public function get_redirect_rules()
	{
		$value = get_option(self::OPTION_REDIRECTS, []);
		if (! is_array($value) || empty($value['rules']) || ! is_array($value['rules'])) {
			return [];
		}
		$stats = $this->get_redirect_stats();

		$rules = [];
		foreach ($value['rules'] as $rule) {
			if (! is_array($rule)) {
				continue;
			}

			$source = isset($rule['source']) ? trim((string) $rule['source']) : '';
			$target = isset($rule['target']) ? trim((string) $rule['target']) : '';
			$code   = isset($rule['code']) ? (string) $rule['code'] : '301';
			$type   = isset($rule['type']) ? (string) $rule['type'] : 'exact';

			if ('' === $source || '' === $target) {
				continue;
			}

			if (! in_array($code, ['301', '302'], true)) {
				$code = '301';
			}
			if (! in_array($type, ['exact', 'regex'], true)) {
				$type = 'exact';
			}

			$rule_key = $this->get_redirect_rule_key($source, $target, $code, $type);
			$rule_stats = isset($stats[$rule_key]) && is_array($stats[$rule_key]) ? $stats[$rule_key] : [];
			$hits = isset($rule_stats['hits']) ? absint($rule_stats['hits']) : 0;
			$last_hit = isset($rule_stats['last_hit']) ? (string) $rule_stats['last_hit'] : '';

				$rules[] = [
					'source' => $source,
					'target' => $target,
					'code'   => $code,
					'type'   => $type,
					'hits'   => $hits,
					'last_hit' => $last_hit,
				];
		}

		return $rules;
	}

	/**
	 * @return array<string, array<string, mixed>>
	 */
	public function get_redirect_stats()
	{
		$value = get_option(self::OPTION_REDIRECT_STATS, []);
		return is_array($value) ? $value : [];
	}

	public function ensure_defaults()
	{
		if (! get_option(self::OPTION_MODULES)) {
			add_option(self::OPTION_MODULES, $this->default_modules());
		}

		if (! get_option(self::OPTION_GENERAL)) {
			add_option(self::OPTION_GENERAL, $this->get_general());
		}

		if (! get_option(self::OPTION_SITEMAP)) {
			add_option(self::OPTION_SITEMAP, $this->get_sitemap());
		}

		if (! get_option(self::OPTION_SOCIAL)) {
			add_option(self::OPTION_SOCIAL, $this->get_social());
		}

		if (! get_option(self::OPTION_ROBOTS)) {
			add_option(self::OPTION_ROBOTS, $this->get_robots());
		}

		if (! get_option(self::OPTION_REDIRECTS)) {
			add_option(self::OPTION_REDIRECTS, ['rules' => []]);
		}
		if (! get_option(self::OPTION_REDIRECT_STATS)) {
			add_option(self::OPTION_REDIRECT_STATS, []);
		}

		if (! get_option(self::OPTION_MIGRATION)) {
			add_option(self::OPTION_MIGRATION, $this->get_migration());
		}
		if (! get_option(self::OPTION_LAST_MIGRATION_REPORT)) {
			add_option(self::OPTION_LAST_MIGRATION_REPORT, []);
		}
	}

	/**
	 * @return array<string, int>
	 */
	private function default_modules()
	{
		return [
			'meta'      => 1,
			'sitemap'   => 1,
			'robots'    => 1,
			'social'    => 1,
			'redirects' => 1,
		];
	}

	/**
	 * @param string $kind title|description
	 * @return array<string, string>
	 */
	private function default_post_type_templates($kind)
	{
		$post_types = get_post_types(['public' => true], 'names');
		$templates = [];
		$value = ('description' === $kind) ? '%excerpt%' : '%title% | %sitename%';

		foreach ($post_types as $post_type) {
			$templates[(string) $post_type] = $value;
		}

		return $templates;
	}

	/**
	 * @param string $kind title|description
	 * @return array<string, string>
	 */
	private function default_taxonomy_templates($kind)
	{
		$taxonomies = get_taxonomies(['public' => true], 'names');
		$templates = [];
		$value = ('description' === $kind) ? '%term_description%' : '%term% | %sitename%';

		foreach ($taxonomies as $taxonomy) {
			$templates[(string) $taxonomy] = $value;
		}

		return $templates;
	}

	/**
	 * @param string $source
	 * @param string $target
	 * @param string $code
	 * @param string $type
	 * @return string
	 */
	private function get_redirect_rule_key($source, $target, $code, $type)
	{
		return sha1($source . '|' . $target . '|' . $code . '|' . $type);
	}
}
