<?php

namespace WPZEO\Admin;

use WPZEO\Core\Settings;
use WPZEO\Modules\Migration;

if (! defined('ABSPATH')) {
	exit;
}

class Admin
{
	/**
	 * @var Settings
	 */
	private $settings;

	/**
	 * @param Settings $settings
	 */
	public function __construct(Settings $settings)
	{
		$this->settings = $settings;

		add_action('admin_menu', [$this, 'register_menu']);
		add_action('admin_init', [$this, 'register_settings']);
		add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);
	}

	/**
	 * @param string $hook
	 */
	public function enqueue_assets($hook)
	{
		if ('settings_page_wpzeo-zero-seo' !== $hook) {
			return;
		}

		wp_enqueue_script(
			'wpzeo-admin-redirects',
			WPZEO_URL . 'assets/admin/redirects.js',
			[],
			WPZEO_VERSION,
			true
		);
		wp_enqueue_script(
			'wpzeo-admin-meta',
			WPZEO_URL . 'assets/admin/meta.js',
			[],
			WPZEO_VERSION,
			true
		);
		wp_enqueue_script(
			'wpzeo-admin-migration',
			WPZEO_URL . 'assets/admin/migration.js',
			[],
			WPZEO_VERSION,
			true
		);

			wp_localize_script(
				'wpzeo-admin-redirects',
				'wpzeoAdmin',
				[
					'restBase' => esc_url_raw(rest_url('wpzeo/v1/redirects')),
					'restExport' => esc_url_raw(rest_url('wpzeo/v1/redirects-export')),
					'restImport' => esc_url_raw(rest_url('wpzeo/v1/redirects-import')),
					'nonce'    => wp_create_nonce('wp_rest'),
					'i18n'     => [
						'saving'  => 'Saving redirects via REST...',
						'loading' => 'Loading redirects via REST...',
						'saved'   => 'Redirects saved.',
						'loaded'  => 'Redirects loaded.',
						'exporting' => 'Exporting redirects CSV...',
						'importing' => 'Importing redirects CSV...',
						'exported' => 'CSV exported.',
						'imported' => 'CSV import completed.',
						'error'   => 'Request failed.',
					],
				]
			);
		wp_localize_script(
			'wpzeo-admin-meta',
			'wpzeoMetaAdmin',
			[
				'restBase' => esc_url_raw(rest_url('wpzeo/v1/meta')),
				'nonce'    => wp_create_nonce('wp_rest'),
				'i18n'     => [
					'loading' => 'Loading meta via REST...',
					'saving'  => 'Saving meta via REST...',
					'loaded'  => 'Meta loaded.',
					'saved'   => 'Meta saved.',
					'error'   => 'Request failed.',
				],
			]
		);
		wp_localize_script(
			'wpzeo-admin-migration',
			'wpzeoMigrationAdmin',
			[
				'restStatus' => esc_url_raw(rest_url('wpzeo/v1/migration-status')),
				'restImport' => esc_url_raw(rest_url('wpzeo/v1/migration-import')),
				'nonce'      => wp_create_nonce('wp_rest'),
				'i18n'       => [
					'loadingStatus' => 'Loading migration status...',
					'runningDryRun' => 'Running migration dry run...',
					'runningImport' => 'Running migration import...',
					'statusLoaded'  => 'Migration status loaded.',
					'dryRunDone'    => 'Dry run completed.',
					'importDone'    => 'Migration import completed.',
					'error'         => 'Request failed.',
				],
			]
		);
	}

	public function register_menu()
	{
		add_options_page(
			'wpZeo - Zero SEO',
			'wpZeo Zero SEO',
			'manage_options',
			'wpzeo-zero-seo',
			[$this, 'render_settings_page']
		);
	}

	public function register_settings()
	{
		register_setting('wpzeo_settings', Settings::OPTION_MODULES, [$this, 'sanitize_modules']);
		register_setting('wpzeo_settings', Settings::OPTION_GENERAL, [$this, 'sanitize_general']);
		register_setting('wpzeo_settings', Settings::OPTION_SITEMAP, [$this, 'sanitize_sitemap']);
		register_setting('wpzeo_settings', Settings::OPTION_SOCIAL, [$this, 'sanitize_social']);
		register_setting('wpzeo_settings', Settings::OPTION_ROBOTS, [$this, 'sanitize_robots']);
		register_setting('wpzeo_settings', Settings::OPTION_REDIRECTS, [$this, 'sanitize_redirects']);
		register_setting('wpzeo_settings', Settings::OPTION_MIGRATION, [$this, 'sanitize_migration']);
	}

	/**
	 * @param mixed $value
	 * @return array<string, int>
	 */
	public function sanitize_modules($value)
	{
		$value = is_array($value) ? $value : [];
		return [
			'meta'      => ! empty($value['meta']) ? 1 : 0,
			'sitemap'   => ! empty($value['sitemap']) ? 1 : 0,
			'robots'    => ! empty($value['robots']) ? 1 : 0,
			'social'    => ! empty($value['social']) ? 1 : 0,
			'redirects' => ! empty($value['redirects']) ? 1 : 0,
		];
	}

	/**
	 * @param mixed $value
	 * @return array<string, mixed>
	 */
	public function sanitize_general($value)
	{
		$value = is_array($value) ? $value : [];
		$robots = isset($value['default_robots']) ? sanitize_text_field((string) $value['default_robots']) : 'index,follow';
		if (! in_array($robots, ['index,follow', 'noindex,follow', 'noindex,nofollow', 'index,nofollow'], true)) {
			$robots = 'index,follow';
		}

		$public_post_types = get_post_types(['public' => true], 'names');
		$public_taxonomies = get_taxonomies(['public' => true], 'names');

		$title_template_post_types = $this->sanitize_template_map(
			isset($value['title_template_post_types']) ? $value['title_template_post_types'] : [],
			$public_post_types
		);
		$description_template_post_types = $this->sanitize_template_map(
			isset($value['description_template_post_types']) ? $value['description_template_post_types'] : [],
			$public_post_types
		);
		$title_template_taxonomies = $this->sanitize_template_map(
			isset($value['title_template_taxonomies']) ? $value['title_template_taxonomies'] : [],
			$public_taxonomies
		);
		$description_template_taxonomies = $this->sanitize_template_map(
			isset($value['description_template_taxonomies']) ? $value['description_template_taxonomies'] : [],
			$public_taxonomies
		);

		return [
			'default_robots' => $robots,
			'noindex_search' => ! empty($value['noindex_search']) ? 1 : 0,
			'title_template_post_types' => $title_template_post_types,
			'description_template_post_types' => $description_template_post_types,
			'title_template_taxonomies' => $title_template_taxonomies,
			'description_template_taxonomies' => $description_template_taxonomies,
			'title_template_author' => isset($value['title_template_author']) ? sanitize_text_field((string) $value['title_template_author']) : '',
			'description_template_author' => isset($value['description_template_author']) ? sanitize_text_field((string) $value['description_template_author']) : '',
		];
	}

	/**
	 * @param mixed $templates
	 * @param array<string, string> $allowed_keys
	 * @return array<string, string>
	 */
	private function sanitize_template_map($templates, $allowed_keys)
	{
		$templates = is_array($templates) ? $templates : [];
		$sanitized = [];

		foreach ($templates as $key => $template) {
			$key = sanitize_key((string) $key);
			if (! isset($allowed_keys[$key])) {
				continue;
			}
			$value = sanitize_text_field((string) $template);
			if ('' === trim($value)) {
				continue;
			}
			$sanitized[$key] = $value;
		}

		return $sanitized;
	}

	/**
	 * @param mixed $value
	 * @return array<string, mixed>
	 */
	public function sanitize_sitemap($value)
	{
		$value = is_array($value) ? $value : [];
		$limit = isset($value['limit']) ? absint($value['limit']) : 1000;
		if ($limit < 100) {
			$limit = 100;
		}
		if ($limit > 5000) {
			$limit = 5000;
		}

		$public_post_types = get_post_types(['public' => true], 'names');
		$public_taxonomies = get_taxonomies(['public' => true], 'names');

		$selected_post_types = isset($value['include_post_types']) && is_array($value['include_post_types']) ? $value['include_post_types'] : [];
		$selected_taxonomies = isset($value['include_taxonomies']) && is_array($value['include_taxonomies']) ? $value['include_taxonomies'] : [];

		$include_post_types = [];
		foreach ($selected_post_types as $post_type) {
			$post_type = sanitize_key((string) $post_type);
			if (isset($public_post_types[$post_type])) {
				$include_post_types[] = $post_type;
			}
		}

		$include_taxonomies = [];
		foreach ($selected_taxonomies as $taxonomy) {
			$taxonomy = sanitize_key((string) $taxonomy);
			if (isset($public_taxonomies[$taxonomy])) {
				$include_taxonomies[] = $taxonomy;
			}
		}

		$exclude_post_ids = isset($value['exclude_post_ids']) ? (string) $value['exclude_post_ids'] : '';
		$exclude_term_ids = isset($value['exclude_term_ids']) ? (string) $value['exclude_term_ids'] : '';

		$exclude_post_parts = preg_split('/[\s,]+/', $exclude_post_ids);
		$exclude_term_parts = preg_split('/[\s,]+/', $exclude_term_ids);
		$exclude_post_parts = is_array($exclude_post_parts) ? $exclude_post_parts : [];
		$exclude_term_parts = is_array($exclude_term_parts) ? $exclude_term_parts : [];

		$exclude_post_ids = array_values(array_filter(array_map('absint', $exclude_post_parts), static function ($id) {
			return $id > 0;
		}));
		$exclude_term_ids = array_values(array_filter(array_map('absint', $exclude_term_parts), static function ($id) {
			return $id > 0;
		}));

		return [
			'enabled'         => ! empty($value['enabled']) ? 1 : 0,
			'include_authors' => ! empty($value['include_authors']) ? 1 : 0,
			'limit'           => $limit,
			'include_post_types' => $include_post_types,
			'include_taxonomies' => $include_taxonomies,
			'exclude_post_ids'   => $exclude_post_ids,
			'exclude_term_ids'   => $exclude_term_ids,
		];
	}

	/**
	 * @param mixed $value
	 * @return array<string, string>
	 */
	public function sanitize_social($value)
	{
		$value = is_array($value) ? $value : [];

		$card = isset($value['twitter_card']) ? sanitize_text_field((string) $value['twitter_card']) : 'summary_large_image';
		if (! in_array($card, ['summary', 'summary_large_image'], true)) {
			$card = 'summary_large_image';
		}

		return [
			'default_image' => isset($value['default_image']) ? esc_url_raw((string) $value['default_image']) : '',
			'twitter_card'  => $card,
		];
	}

	/**
	 * @param mixed $value
	 * @return array<string, string>
	 */
	public function sanitize_robots($value)
	{
		$value = is_array($value) ? $value : [];
		return [
			'custom_rules' => isset($value['custom_rules']) ? sanitize_textarea_field((string) $value['custom_rules']) : '',
		];
	}

	/**
	 * @param mixed $value
	 * @return array<string, int>
	 */
	public function sanitize_migration($value)
	{
		$value = is_array($value) ? $value : [];
		return [
			'enabled'     => ! empty($value['enabled']) ? 1 : 0,
			'auto_enable' => ! empty($value['auto_enable']) ? 1 : 0,
		];
	}

	/**
	 * Rules format: one per line => /source|https://target.tld/path|301|exact
	 *
	 * @param mixed $value
	 * @return array<string, array<int, array<string, string>>>
	 */
	public function sanitize_redirects($value)
	{
		$value = is_array($value) ? $value : [];
		$raw   = isset($value['raw']) ? (string) $value['raw'] : '';
		$lines = preg_split('/\r\n|\r|\n/', $raw);

		$rules = [];
		foreach ((array) $lines as $line) {
			$line = trim($line);
			if ('' === $line) {
				continue;
			}

			$parts = array_map('trim', explode('|', $line));
			if (count($parts) < 2) {
				continue;
			}

			$source = $parts[0];
			$target = $parts[1];
			$code   = isset($parts[2]) ? $parts[2] : '301';
			$type   = isset($parts[3]) ? sanitize_key((string) $parts[3]) : 'exact';

			if (! in_array($code, ['301', '302'], true)) {
				$code = '301';
			}
			if (! in_array($type, ['exact', 'regex'], true)) {
				$type = 'exact';
			}

			if ('' === $source || '' === $target) {
				continue;
			}

			if ('exact' === $type) {
				$source = '/' . ltrim($source, '/');
			} else {
				if (false === @preg_match($source, '')) {
					continue;
				}
			}

			if (0 === strpos($target, '/')) {
				$target = home_url($target);
			} else {
				$target = esc_url_raw($target);
			}

			if ('' === $target) {
				continue;
			}

			$rules[] = [
				'source' => $source,
				'target' => $target,
				'code'   => $code,
				'type'   => $type,
			];
		}

		return ['rules' => $rules];
	}

	public function render_settings_page()
	{
		if (! current_user_can('manage_options')) {
			return;
		}

		$modules   = $this->settings->get_modules();
		$general   = $this->settings->get_general();
		$sitemap   = $this->settings->get_sitemap();
		$social    = $this->settings->get_social();
		$robots    = $this->settings->get_robots();
		$redirects = $this->settings->get_redirect_rules();
		$migration = $this->settings->get_migration();
		$detected_plugins = Migration::detect_supported_plugins();
		$last_migration_report = get_option(Settings::OPTION_LAST_MIGRATION_REPORT, []);
		$api_routes = function_exists('rest_get_server') ? rest_get_server()->get_routes() : [];
		$api_health_ok = is_array($api_routes) && isset($api_routes['/wpzeo/v1/migration-status']);

			$redirect_lines = [];
			foreach ($redirects as $rule) {
				$type = isset($rule['type']) ? (string) $rule['type'] : 'exact';
				$redirect_lines[] = $rule['source'] . '|' . $rule['target'] . '|' . $rule['code'] . '|' . $type;
			}
		?>
		<div class="wrap">
			<h1>wpZeo - Zero SEO</h1>
			<p>Lean SEO settings for metadata, taxonomies, authors, sitemap, robots, social and redirects.</p>

			<h2>Health</h2>
			<table class="widefat striped" style="max-width:900px; margin-bottom:18px;">
				<tbody>
					<tr>
						<td><strong>wpZeo REST API</strong></td>
						<td><?php echo $api_health_ok ? 'reachable' : 'not reachable'; ?></td>
					</tr>
					<tr>
						<td><strong>Migration mode (effective)</strong></td>
						<td><?php echo (! empty($migration['enabled']) || (! empty($migration['auto_enable']) && (bool) array_filter($detected_plugins, static function ($plugin) { return ! empty($plugin['installed']); }))) ? 'enabled' : 'disabled'; ?></td>
					</tr>
					<tr>
						<td><strong>Last migration run</strong></td>
						<td><?php echo esc_html(isset($last_migration_report['run_at']) ? (string) $last_migration_report['run_at'] : 'none'); ?></td>
					</tr>
					<tr>
						<td><strong>Last migration source</strong></td>
						<td><?php echo esc_html(isset($last_migration_report['source']) ? (string) $last_migration_report['source'] : 'n/a'); ?></td>
					</tr>
				</tbody>
			</table>

			<form method="post" action="options.php">
				<?php settings_fields('wpzeo_settings'); ?>

				<h2>Modules</h2>
				<table class="form-table" role="presentation">
					<tr>
						<th scope="row">Enabled modules</th>
						<td>
							<label><input type="checkbox" name="<?php echo esc_attr(Settings::OPTION_MODULES); ?>[meta]" value="1" <?php checked($modules['meta']); ?>> Meta</label><br>
							<label><input type="checkbox" name="<?php echo esc_attr(Settings::OPTION_MODULES); ?>[sitemap]" value="1" <?php checked($modules['sitemap']); ?>> Sitemap</label><br>
							<label><input type="checkbox" name="<?php echo esc_attr(Settings::OPTION_MODULES); ?>[robots]" value="1" <?php checked($modules['robots']); ?>> Robots.txt</label><br>
							<label><input type="checkbox" name="<?php echo esc_attr(Settings::OPTION_MODULES); ?>[social]" value="1" <?php checked($modules['social']); ?>> Social</label><br>
							<label><input type="checkbox" name="<?php echo esc_attr(Settings::OPTION_MODULES); ?>[redirects]" value="1" <?php checked($modules['redirects']); ?>> Redirects</label>
						</td>
					</tr>
				</table>

				<h2>General</h2>
				<table class="form-table" role="presentation">
					<tr>
						<th scope="row"><label for="wpzeo_default_robots">Default robots</label></th>
						<td>
							<select id="wpzeo_default_robots" name="<?php echo esc_attr(Settings::OPTION_GENERAL); ?>[default_robots]">
								<option value="index,follow" <?php selected($general['default_robots'], 'index,follow'); ?>>index,follow</option>
								<option value="noindex,follow" <?php selected($general['default_robots'], 'noindex,follow'); ?>>noindex,follow</option>
								<option value="index,nofollow" <?php selected($general['default_robots'], 'index,nofollow'); ?>>index,nofollow</option>
								<option value="noindex,nofollow" <?php selected($general['default_robots'], 'noindex,nofollow'); ?>>noindex,nofollow</option>
							</select>
						</td>
					</tr>
					<tr>
						<th scope="row">Noindex search results</th>
						<td>
							<label><input type="checkbox" name="<?php echo esc_attr(Settings::OPTION_GENERAL); ?>[noindex_search]" value="1" <?php checked(! empty($general['noindex_search'])); ?>> Add noindex to search pages</label>
						</td>
					</tr>
					<tr>
						<th scope="row">Title templates (post types)</th>
						<td>
							<?php foreach (get_post_types(['public' => true], 'objects') as $post_type) : ?>
								<?php
								$pt_name = (string) $post_type->name;
								$pt_template = isset($general['title_template_post_types'][$pt_name]) ? (string) $general['title_template_post_types'][$pt_name] : '';
								?>
								<label for="wpzeo_title_tpl_pt_<?php echo esc_attr($pt_name); ?>"><?php echo esc_html($post_type->labels->singular_name . ' (' . $pt_name . ')'); ?></label><br>
								<input id="wpzeo_title_tpl_pt_<?php echo esc_attr($pt_name); ?>" class="regular-text" type="text" name="<?php echo esc_attr(Settings::OPTION_GENERAL); ?>[title_template_post_types][<?php echo esc_attr($pt_name); ?>]" value="<?php echo esc_attr($pt_template); ?>"><br><br>
							<?php endforeach; ?>
							<p class="description">Available placeholders: <code>%title%</code>, <code>%sitename%</code>, <code>%excerpt%</code>.</p>
						</td>
					</tr>
					<tr>
						<th scope="row">Description templates (post types)</th>
						<td>
							<?php foreach (get_post_types(['public' => true], 'objects') as $post_type) : ?>
								<?php
								$pt_name = (string) $post_type->name;
								$pt_template = isset($general['description_template_post_types'][$pt_name]) ? (string) $general['description_template_post_types'][$pt_name] : '';
								?>
								<label for="wpzeo_desc_tpl_pt_<?php echo esc_attr($pt_name); ?>"><?php echo esc_html($post_type->labels->singular_name . ' (' . $pt_name . ')'); ?></label><br>
								<input id="wpzeo_desc_tpl_pt_<?php echo esc_attr($pt_name); ?>" class="regular-text" type="text" name="<?php echo esc_attr(Settings::OPTION_GENERAL); ?>[description_template_post_types][<?php echo esc_attr($pt_name); ?>]" value="<?php echo esc_attr($pt_template); ?>"><br><br>
							<?php endforeach; ?>
							<p class="description">Available placeholders: <code>%title%</code>, <code>%sitename%</code>, <code>%excerpt%</code>.</p>
						</td>
					</tr>
					<tr>
						<th scope="row">Title templates (taxonomies)</th>
						<td>
							<?php foreach (get_taxonomies(['public' => true], 'objects') as $taxonomy) : ?>
								<?php
								$tx_name = (string) $taxonomy->name;
								$tx_template = isset($general['title_template_taxonomies'][$tx_name]) ? (string) $general['title_template_taxonomies'][$tx_name] : '';
								?>
								<label for="wpzeo_title_tpl_tx_<?php echo esc_attr($tx_name); ?>"><?php echo esc_html($taxonomy->labels->singular_name . ' (' . $tx_name . ')'); ?></label><br>
								<input id="wpzeo_title_tpl_tx_<?php echo esc_attr($tx_name); ?>" class="regular-text" type="text" name="<?php echo esc_attr(Settings::OPTION_GENERAL); ?>[title_template_taxonomies][<?php echo esc_attr($tx_name); ?>]" value="<?php echo esc_attr($tx_template); ?>"><br><br>
							<?php endforeach; ?>
							<p class="description">Available placeholders: <code>%term%</code>, <code>%term_description%</code>, <code>%sitename%</code>.</p>
						</td>
					</tr>
					<tr>
						<th scope="row">Description templates (taxonomies)</th>
						<td>
							<?php foreach (get_taxonomies(['public' => true], 'objects') as $taxonomy) : ?>
								<?php
								$tx_name = (string) $taxonomy->name;
								$tx_template = isset($general['description_template_taxonomies'][$tx_name]) ? (string) $general['description_template_taxonomies'][$tx_name] : '';
								?>
								<label for="wpzeo_desc_tpl_tx_<?php echo esc_attr($tx_name); ?>"><?php echo esc_html($taxonomy->labels->singular_name . ' (' . $tx_name . ')'); ?></label><br>
								<input id="wpzeo_desc_tpl_tx_<?php echo esc_attr($tx_name); ?>" class="regular-text" type="text" name="<?php echo esc_attr(Settings::OPTION_GENERAL); ?>[description_template_taxonomies][<?php echo esc_attr($tx_name); ?>]" value="<?php echo esc_attr($tx_template); ?>"><br><br>
							<?php endforeach; ?>
							<p class="description">Available placeholders: <code>%term%</code>, <code>%term_description%</code>, <code>%sitename%</code>.</p>
						</td>
					</tr>
					<tr>
						<th scope="row">Author templates</th>
						<td>
							<label for="wpzeo_title_template_author">Title template (author archives)</label><br>
							<input id="wpzeo_title_template_author" class="regular-text" type="text" name="<?php echo esc_attr(Settings::OPTION_GENERAL); ?>[title_template_author]" value="<?php echo esc_attr(isset($general['title_template_author']) ? (string) $general['title_template_author'] : ''); ?>"><br><br>
							<label for="wpzeo_description_template_author">Description template (author archives)</label><br>
							<input id="wpzeo_description_template_author" class="regular-text" type="text" name="<?php echo esc_attr(Settings::OPTION_GENERAL); ?>[description_template_author]" value="<?php echo esc_attr(isset($general['description_template_author']) ? (string) $general['description_template_author'] : ''); ?>">
							<p class="description">Available placeholders: <code>%author%</code>, <code>%author_bio%</code>, <code>%sitename%</code>.</p>
						</td>
					</tr>
				</table>

				<h2>Sitemap</h2>
				<table class="form-table" role="presentation">
					<tr>
						<th scope="row">Enable sitemap</th>
						<td><label><input type="checkbox" name="<?php echo esc_attr(Settings::OPTION_SITEMAP); ?>[enabled]" value="1" <?php checked(! empty($sitemap['enabled'])); ?>> Enable at <code>/wpzeo-sitemap.xml</code></label></td>
					</tr>
					<tr>
						<th scope="row">Include authors</th>
						<td><label><input type="checkbox" name="<?php echo esc_attr(Settings::OPTION_SITEMAP); ?>[include_authors]" value="1" <?php checked(! empty($sitemap['include_authors'])); ?>> Include author archive URLs</label></td>
					</tr>
					<tr>
						<th scope="row"><label for="wpzeo_sitemap_limit">URL limit</label></th>
						<td><input id="wpzeo_sitemap_limit" type="number" min="100" max="5000" name="<?php echo esc_attr(Settings::OPTION_SITEMAP); ?>[limit]" value="<?php echo esc_attr((string) $sitemap['limit']); ?>"></td>
					</tr>
					<tr>
						<th scope="row">Include post types</th>
						<td>
							<?php foreach (get_post_types(['public' => true], 'objects') as $post_type) : ?>
								<?php $pt_name = (string) $post_type->name; ?>
								<label>
									<input type="checkbox" name="<?php echo esc_attr(Settings::OPTION_SITEMAP); ?>[include_post_types][]" value="<?php echo esc_attr($pt_name); ?>" <?php checked(in_array($pt_name, (array) $sitemap['include_post_types'], true)); ?>>
									<?php echo esc_html($post_type->labels->singular_name . ' (' . $pt_name . ')'); ?>
								</label><br>
							<?php endforeach; ?>
						</td>
					</tr>
					<tr>
						<th scope="row">Include taxonomies</th>
						<td>
							<?php foreach (get_taxonomies(['public' => true], 'objects') as $taxonomy) : ?>
								<?php $tx_name = (string) $taxonomy->name; ?>
								<label>
									<input type="checkbox" name="<?php echo esc_attr(Settings::OPTION_SITEMAP); ?>[include_taxonomies][]" value="<?php echo esc_attr($tx_name); ?>" <?php checked(in_array($tx_name, (array) $sitemap['include_taxonomies'], true)); ?>>
									<?php echo esc_html($taxonomy->labels->singular_name . ' (' . $tx_name . ')'); ?>
								</label><br>
							<?php endforeach; ?>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="wpzeo_exclude_post_ids">Exclude post IDs</label></th>
						<td>
							<input id="wpzeo_exclude_post_ids" class="regular-text" type="text" name="<?php echo esc_attr(Settings::OPTION_SITEMAP); ?>[exclude_post_ids]" value="<?php echo esc_attr(implode(',', (array) $sitemap['exclude_post_ids'])); ?>">
							<p class="description">Comma or space separated IDs, e.g. <code>12,45,97</code>.</p>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="wpzeo_exclude_term_ids">Exclude term IDs</label></th>
						<td>
							<input id="wpzeo_exclude_term_ids" class="regular-text" type="text" name="<?php echo esc_attr(Settings::OPTION_SITEMAP); ?>[exclude_term_ids]" value="<?php echo esc_attr(implode(',', (array) $sitemap['exclude_term_ids'])); ?>">
							<p class="description">Comma or space separated IDs, e.g. <code>3,10</code>.</p>
						</td>
					</tr>
				</table>

				<h2>Social</h2>
				<table class="form-table" role="presentation">
					<tr>
						<th scope="row"><label for="wpzeo_default_image">Default social image URL</label></th>
						<td><input id="wpzeo_default_image" class="regular-text" type="url" name="<?php echo esc_attr(Settings::OPTION_SOCIAL); ?>[default_image]" value="<?php echo esc_attr((string) $social['default_image']); ?>"></td>
					</tr>
					<tr>
						<th scope="row"><label for="wpzeo_twitter_card">Twitter/X card</label></th>
						<td>
							<select id="wpzeo_twitter_card" name="<?php echo esc_attr(Settings::OPTION_SOCIAL); ?>[twitter_card]">
								<option value="summary" <?php selected($social['twitter_card'], 'summary'); ?>>summary</option>
								<option value="summary_large_image" <?php selected($social['twitter_card'], 'summary_large_image'); ?>>summary_large_image</option>
							</select>
						</td>
					</tr>
				</table>

				<h2>Robots.txt</h2>
				<table class="form-table" role="presentation">
					<tr>
						<th scope="row"><label for="wpzeo_custom_robots">Custom rules</label></th>
						<td>
							<textarea id="wpzeo_custom_robots" class="large-text code" rows="8" name="<?php echo esc_attr(Settings::OPTION_ROBOTS); ?>[custom_rules]"><?php echo esc_textarea((string) $robots['custom_rules']); ?></textarea>
							<p class="description">Each line is appended to generated robots.txt output.</p>
						</td>
					</tr>
				</table>

				<h2>Redirects</h2>
				<table class="form-table" role="presentation">
					<tr>
						<th scope="row"><label for="wpzeo_redirects_raw">Rules</label></th>
						<td>
								<textarea id="wpzeo_redirects_raw" class="large-text code" rows="10" name="<?php echo esc_attr(Settings::OPTION_REDIRECTS); ?>[raw]"><?php echo esc_textarea(implode("\n", $redirect_lines)); ?></textarea>
								<p class="description">One rule per line: <code>/old-path|https://example.com/new-path|301|exact</code> or <code>#^/old/(.+)$#|/new/$1|301|regex</code></p>
								<p>
									<button type="button" class="button" id="wpzeo-load-redirects">Load via REST</button>
									<button type="button" class="button button-secondary" id="wpzeo-save-redirects">Save via REST</button>
									<button type="button" class="button" id="wpzeo-export-redirects-csv">Export CSV</button>
									<button type="button" class="button" id="wpzeo-import-redirects-csv">Import CSV</button>
									<input type="file" id="wpzeo-import-redirects-file" accept=".csv,text/csv" style="display:none;">
								</p>
								<div id="wpzeo-redirects-status" class="notice inline" style="display:none;"></div>
							</td>
						</tr>
				</table>

				<h2>REST Meta Manager</h2>
				<table class="form-table" role="presentation">
					<tr>
						<th scope="row"><label for="wpzeo_meta_object_type">Object</label></th>
						<td>
							<select id="wpzeo_meta_object_type">
								<option value="post">Post</option>
								<option value="term">Term</option>
								<option value="author">Author</option>
							</select>
							<input type="number" min="1" id="wpzeo_meta_object_id" placeholder="Object ID" style="width:140px;">
							<button type="button" class="button" id="wpzeo-meta-load">Load via REST</button>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="wpzeo_meta_title">SEO title</label></th>
						<td><input type="text" id="wpzeo_meta_title" class="regular-text"></td>
					</tr>
					<tr>
						<th scope="row"><label for="wpzeo_meta_description">Meta description</label></th>
						<td><textarea id="wpzeo_meta_description" class="large-text" rows="3"></textarea></td>
					</tr>
					<tr>
						<th scope="row"><label for="wpzeo_meta_focus_keyword">Focus keyword</label></th>
						<td><input type="text" id="wpzeo_meta_focus_keyword" class="regular-text"></td>
					</tr>
					<tr>
						<th scope="row"><label for="wpzeo_meta_robots">Robots</label></th>
						<td>
							<select id="wpzeo_meta_robots">
								<option value="">Default</option>
								<option value="index,follow">index,follow</option>
								<option value="noindex,follow">noindex,follow</option>
								<option value="index,nofollow">index,nofollow</option>
								<option value="noindex,nofollow">noindex,nofollow</option>
							</select>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="wpzeo_meta_canonical">Canonical URL</label></th>
						<td><input type="url" id="wpzeo_meta_canonical" class="regular-text"></td>
					</tr>
					<tr>
						<th scope="row"><label for="wpzeo_meta_og_image">OG image URL</label></th>
						<td><input type="url" id="wpzeo_meta_og_image" class="regular-text"></td>
					</tr>
					<tr class="wpzeo-meta-post-only">
						<th scope="row"><label for="wpzeo_meta_og_title">Open Graph title</label></th>
						<td><input type="text" id="wpzeo_meta_og_title" class="regular-text"></td>
					</tr>
					<tr class="wpzeo-meta-post-only">
						<th scope="row"><label for="wpzeo_meta_og_description">Open Graph description</label></th>
						<td><textarea id="wpzeo_meta_og_description" class="large-text" rows="2"></textarea></td>
					</tr>
					<tr class="wpzeo-meta-post-only">
						<th scope="row"><label for="wpzeo_meta_twitter_title">Twitter/X title</label></th>
						<td><input type="text" id="wpzeo_meta_twitter_title" class="regular-text"></td>
					</tr>
					<tr class="wpzeo-meta-post-only">
						<th scope="row"><label for="wpzeo_meta_twitter_description">Twitter/X description</label></th>
						<td><textarea id="wpzeo_meta_twitter_description" class="large-text" rows="2"></textarea></td>
					</tr>
					<tr class="wpzeo-meta-post-only">
						<th scope="row"><label for="wpzeo_meta_twitter_image">Twitter/X image URL</label></th>
						<td><input type="url" id="wpzeo_meta_twitter_image" class="regular-text"></td>
					</tr>
					<tr class="wpzeo-meta-post-only">
						<th scope="row"><label for="wpzeo_meta_robots_advanced">Advanced robots</label></th>
						<td><input type="text" id="wpzeo_meta_robots_advanced" class="regular-text" placeholder="noarchive,nosnippet,max-image-preview:large"></td>
					</tr>
					<tr class="wpzeo-meta-post-only">
						<th scope="row">Exclude from sitemap</th>
						<td><label><input type="checkbox" id="wpzeo_meta_exclude_sitemap" value="1"> Exclude this post from wpZeo sitemap</label></td>
					</tr>
					<tr>
						<th scope="row"></th>
						<td>
							<button type="button" class="button button-primary" id="wpzeo-meta-save">Save via REST</button>
							<div id="wpzeo-meta-status" class="notice inline" style="display:none; margin-top:10px;"></div>
						</td>
					</tr>
				</table>

				<h2>Migration Mode</h2>
				<table class="form-table" role="presentation">
					<tr>
						<th scope="row">Migration mode</th>
						<td>
							<label><input type="checkbox" name="<?php echo esc_attr(Settings::OPTION_MIGRATION); ?>[enabled]" value="1" <?php checked(! empty($migration['enabled'])); ?>> Force-enable migration mode</label><br>
							<label><input type="checkbox" name="<?php echo esc_attr(Settings::OPTION_MIGRATION); ?>[auto_enable]" value="1" <?php checked(! empty($migration['auto_enable'])); ?>> Auto-enable when legacy SEO plugins are installed</label>
						</td>
					</tr>
					<tr>
						<th scope="row">Detected plugins</th>
						<td>
							<?php foreach ($detected_plugins as $plugin) : ?>
								<?php
								$status_parts = [];
								$status_parts[] = ! empty($plugin['installed']) ? 'installed' : 'not installed';
								$status_parts[] = ! empty($plugin['active']) ? 'active' : 'inactive';
								$version = ! empty($plugin['version']) ? 'v' . (string) $plugin['version'] : '';
								?>
								<div>
									<strong><?php echo esc_html((string) $plugin['name']); ?></strong>:
									<?php echo esc_html(implode(', ', $status_parts)); ?>
									<?php if ('' !== $version) : ?>
										(<?php echo esc_html($version); ?>)
									<?php endif; ?>
								</div>
							<?php endforeach; ?>
							<p class="description">REST status endpoint: <code>/wp-json/wpzeo/v1/migration-status</code></p>
						</td>
					</tr>
					<tr>
						<th scope="row">Migration assistant</th>
						<td>
							<p>
								<button type="button" class="button" id="wpzeo-migration-refresh-status">Refresh status</button>
							</p>
							<p>
								<label for="wpzeo_migration_source"><strong>Source</strong></label><br>
								<select id="wpzeo_migration_source">
									<option value="auto">auto</option>
									<option value="yoast">yoast</option>
									<option value="rank_math">rank_math</option>
								</select>
							</p>
							<p>
								<label for="wpzeo_migration_redirect_conflict"><strong>Redirect conflict strategy</strong></label><br>
								<select id="wpzeo_migration_redirect_conflict">
									<option value="skip">skip (keep existing)</option>
									<option value="overwrite">overwrite (replace existing)</option>
									<option value="append">append (allow multiple by source)</option>
								</select>
							</p>
							<p>
								<label><input type="checkbox" id="wpzeo_migration_include_posts" checked> Include posts</label><br>
								<label><input type="checkbox" id="wpzeo_migration_include_terms" checked> Include terms</label><br>
								<label><input type="checkbox" id="wpzeo_migration_include_authors" checked> Include authors</label><br>
								<label><input type="checkbox" id="wpzeo_migration_include_redirects" checked> Include redirects</label><br>
								<label><input type="checkbox" id="wpzeo_migration_overwrite"> Overwrite existing wpZeo values</label>
							</p>
							<p>
								<button type="button" class="button" id="wpzeo-migration-dry-run">Dry run</button>
								<button type="button" class="button button-primary" id="wpzeo-migration-run">Run import</button>
							</p>
							<div id="wpzeo-migration-status" class="notice inline" style="display:none; margin-top:10px;"></div>
							<pre id="wpzeo-migration-result" style="display:none; max-height:320px; overflow:auto; background:#fff; border:1px solid #ccd0d4; padding:10px; margin-top:10px;"></pre>
						</td>
					</tr>
				</table>

				<?php submit_button('Save wpZeo settings'); ?>
			</form>
		</div>
		<?php
	}
}
