(function () {
  "use strict";

  if (typeof wpzeoMetaAdmin === "undefined") {
    return;
  }

  var objectType = document.getElementById("wpzeo_meta_object_type");
  var objectId = document.getElementById("wpzeo_meta_object_id");
  var title = document.getElementById("wpzeo_meta_title");
  var description = document.getElementById("wpzeo_meta_description");
  var focusKeyword = document.getElementById("wpzeo_meta_focus_keyword");
  var robots = document.getElementById("wpzeo_meta_robots");
  var canonical = document.getElementById("wpzeo_meta_canonical");
  var ogImage = document.getElementById("wpzeo_meta_og_image");
  var ogTitle = document.getElementById("wpzeo_meta_og_title");
  var ogDescription = document.getElementById("wpzeo_meta_og_description");
  var twitterTitle = document.getElementById("wpzeo_meta_twitter_title");
  var twitterDescription = document.getElementById("wpzeo_meta_twitter_description");
  var twitterImage = document.getElementById("wpzeo_meta_twitter_image");
  var robotsAdvanced = document.getElementById("wpzeo_meta_robots_advanced");
  var excludeSitemap = document.getElementById("wpzeo_meta_exclude_sitemap");
  var postOnlyRows = document.querySelectorAll(".wpzeo-meta-post-only");
  var loadBtn = document.getElementById("wpzeo-meta-load");
  var saveBtn = document.getElementById("wpzeo-meta-save");
  var statusEl = document.getElementById("wpzeo-meta-status");

  if (
    !objectType ||
    !objectId ||
    !title ||
    !description ||
    !focusKeyword ||
    !robots ||
    !canonical ||
    !ogImage ||
    !ogTitle ||
    !ogDescription ||
    !twitterTitle ||
    !twitterDescription ||
    !twitterImage ||
    !robotsAdvanced ||
    !excludeSitemap ||
    !loadBtn ||
    !saveBtn ||
    !statusEl
  ) {
    return;
  }

  function setStatus(message, type) {
    statusEl.className = "notice inline";
    statusEl.style.display = "block";
    if (type === "success") {
      statusEl.classList.add("notice-success");
    } else if (type === "warning") {
      statusEl.classList.add("notice-warning");
    } else if (type === "error") {
      statusEl.classList.add("notice-error");
    } else {
      statusEl.classList.add("notice-info");
    }
    statusEl.innerHTML = "<p>" + message + "</p>";
  }

  function getEndpoint() {
    var type = objectType.value;
    var id = Number(objectId.value || 0);
    if (!id || id < 1) {
      return null;
    }
    return wpzeoMetaAdmin.restBase + "/" + encodeURIComponent(type) + "/" + id;
  }

  function request(method, endpoint, body) {
    return fetch(endpoint, {
      method: method,
      credentials: "same-origin",
      headers: {
        "Content-Type": "application/json",
        "X-WP-Nonce": wpzeoMetaAdmin.nonce,
      },
      body: body ? JSON.stringify(body) : undefined,
    }).then(function (response) {
      return response.json().then(function (json) {
        return { ok: response.ok, status: response.status, json: json };
      });
    });
  }

  function fillForm(meta) {
    title.value = meta.title || "";
    description.value = meta.description || "";
    focusKeyword.value = meta.focus_keyword || "";
    robots.value = meta.robots || "";
    canonical.value = meta.canonical || "";
    ogImage.value = meta.og_image || "";
    ogTitle.value = meta.og_title || "";
    ogDescription.value = meta.og_description || "";
    twitterTitle.value = meta.twitter_title || "";
    twitterDescription.value = meta.twitter_description || "";
    twitterImage.value = meta.twitter_image || "";
    robotsAdvanced.value = meta.robots_advanced || "";
    excludeSitemap.checked = String(meta.exclude_sitemap || "") === "1";
  }

  function updatePostOnlyState() {
    var isPost = objectType.value === "post";
    postOnlyRows.forEach(function (row) {
      row.style.display = isPost ? "" : "none";
    });
    ogTitle.disabled = !isPost;
    ogDescription.disabled = !isPost;
    twitterTitle.disabled = !isPost;
    twitterDescription.disabled = !isPost;
    twitterImage.disabled = !isPost;
    robotsAdvanced.disabled = !isPost;
    excludeSitemap.disabled = !isPost;
  }

  objectType.addEventListener("change", function () {
    updatePostOnlyState();
  });
  updatePostOnlyState();

  loadBtn.addEventListener("click", function () {
    var endpoint = getEndpoint();
    if (!endpoint) {
      setStatus("Please provide a valid object ID.", "warning");
      return;
    }

    setStatus(wpzeoMetaAdmin.i18n.loading, "info");
    request("GET", endpoint)
      .then(function (result) {
        if (!result.ok) {
          var message =
            (result.json && (result.json.message || result.json.error)) ||
            wpzeoMetaAdmin.i18n.error;
          setStatus(message, "error");
          return;
        }
        fillForm(result.json.meta || {});
        setStatus(wpzeoMetaAdmin.i18n.loaded, "success");
      })
      .catch(function () {
        setStatus(wpzeoMetaAdmin.i18n.error, "error");
      });
  });

  saveBtn.addEventListener("click", function () {
    var endpoint = getEndpoint();
    if (!endpoint) {
      setStatus("Please provide a valid object ID.", "warning");
      return;
    }

    var payload = {
      title: title.value,
      description: description.value,
      focus_keyword: focusKeyword.value,
      robots: robots.value,
      canonical: canonical.value,
      og_image: ogImage.value,
    };
    if (objectType.value === "post") {
      payload.og_title = ogTitle.value;
      payload.og_description = ogDescription.value;
      payload.twitter_title = twitterTitle.value;
      payload.twitter_description = twitterDescription.value;
      payload.twitter_image = twitterImage.value;
      payload.robots_advanced = robotsAdvanced.value;
      payload.exclude_sitemap = excludeSitemap.checked ? 1 : 0;
    }

    setStatus(wpzeoMetaAdmin.i18n.saving, "info");
    request("POST", endpoint, payload)
      .then(function (result) {
        if (!result.ok) {
          var message =
            (result.json && (result.json.message || result.json.error)) ||
            wpzeoMetaAdmin.i18n.error;
          setStatus(message, "error");
          return;
        }
        fillForm(result.json.meta || {});
        setStatus(wpzeoMetaAdmin.i18n.saved, "success");
      })
      .catch(function () {
        setStatus(wpzeoMetaAdmin.i18n.error, "error");
      });
  });
})();
