<?php

declare(strict_types=1);

namespace Agent\Modules\System;

use Agent\Plugin;
use Agent\Support\Options;

final class ChangesetService
{
    private Options $options;

    public function __construct(Options $options)
    {
        $this->options = $options;
    }

    public function apply(array $actions, string $auditId): array
    {
        $inverseActions = [];
        $applied = [];

        foreach ($actions as $index => $action) {
            $inverse = $this->buildInverseAction($action);
            $this->executeAction($action);
            array_unshift($inverseActions, $inverse);

            $applied[] = [
                'index' => $index,
                'id' => sanitize_text_field((string) ($action['id'] ?? ('action_' . (string) $index))),
                'operation' => sanitize_key((string) ($action['operation'] ?? '')),
            ];
        }

        $rollbackToken = $this->storeRollbackTemplate($auditId, $inverseActions);

        return [
            'applied' => $applied,
            'rollback_token' => $rollbackToken,
            'inverse_count' => count($inverseActions),
        ];
    }

    public function rollback(string $rollbackToken): array|null
    {
        $templates = $this->getRollbackTemplates();

        foreach ($templates as $i => $template) {
            if (! is_array($template) || ($template['rollback_token'] ?? '') !== $rollbackToken) {
                continue;
            }

            if (($template['status'] ?? '') === 'rolled_back') {
                return ['status' => 'already_rolled_back'];
            }

            $inverseActions = is_array($template['inverse_actions'] ?? null) ? $template['inverse_actions'] : [];
            foreach ($inverseActions as $action) {
                if (! is_array($action)) {
                    continue;
                }
                $this->executeAction($action);
            }

            $templates[$i]['status'] = 'rolled_back';
            $templates[$i]['rolled_back_at'] = gmdate('c');
            $this->saveRollbackTemplates($templates);

            return [
                'status' => 'rolled_back',
                'audit_id' => (string) ($templates[$i]['audit_id'] ?? ''),
                'inverse_count' => count($inverseActions),
            ];
        }

        return null;
    }

    private function buildInverseAction(array $action): array
    {
        $operation = sanitize_key((string) ($action['operation'] ?? ''));

        return match ($operation) {
            'set_readonly_mode' => [
                'operation' => 'set_readonly_mode',
                'payload' => ['enabled' => (bool) $this->options->get('readonly_mode', true)],
            ],
            'set_emergency_lock' => [
                'operation' => 'set_emergency_lock',
                'payload' => ['enabled' => (bool) $this->options->get('emergency_lock', false)],
            ],
            'set_module' => $this->buildModuleInverse($action),
            'set_rate_limit' => [
                'operation' => 'set_rate_limit',
                'payload' => ['value' => (int) $this->options->get('rate_limit_per_minute', 60)],
            ],
            default => [
                'operation' => 'noop',
                'payload' => [],
            ],
        };
    }

    private function buildModuleInverse(array $action): array
    {
        $payload = is_array($action['payload'] ?? null) ? $action['payload'] : [];
        $module = sanitize_key((string) ($payload['module'] ?? ''));

        return [
            'operation' => 'set_module',
            'payload' => [
                'module' => $module,
                'enabled' => (bool) $this->options->get('modules.' . $module, false),
            ],
        ];
    }

    private function executeAction(array $action): void
    {
        $operation = sanitize_key((string) ($action['operation'] ?? ''));
        $payload = is_array($action['payload'] ?? null) ? $action['payload'] : [];

        if ($operation === 'set_readonly_mode') {
            $this->options->set('readonly_mode', (bool) ($payload['enabled'] ?? true));
            return;
        }

        if ($operation === 'set_emergency_lock') {
            $this->options->set('emergency_lock', (bool) ($payload['enabled'] ?? false));
            return;
        }

        if ($operation === 'set_module') {
            $module = sanitize_key((string) ($payload['module'] ?? ''));
            if ($module !== '') {
                $this->options->set('modules.' . $module, (bool) ($payload['enabled'] ?? false));
            }
            return;
        }

        if ($operation === 'set_rate_limit') {
            $value = max(1, (int) ($payload['value'] ?? 60));
            $this->options->set('rate_limit_per_minute', $value);
        }
    }

    private function storeRollbackTemplate(string $auditId, array $inverseActions): string
    {
        $templates = $this->getRollbackTemplates();
        $rollbackToken = $this->generateToken();

        $templates[] = [
            'rollback_token' => $rollbackToken,
            'audit_id' => $auditId,
            'created_at' => gmdate('c'),
            'actor_user_id' => get_current_user_id(),
            'status' => 'pending',
            'inverse_actions' => $inverseActions,
        ];

        if (count($templates) > 100) {
            $templates = array_slice($templates, -100);
        }

        $this->saveRollbackTemplates($templates);

        return $rollbackToken;
    }

    private function getRollbackTemplates(): array
    {
        $raw = get_option(Plugin::OPTION_ROLLBACKS, []);
        return is_array($raw) ? $raw : [];
    }

    private function saveRollbackTemplates(array $templates): void
    {
        update_option(Plugin::OPTION_ROLLBACKS, $templates, false);
    }

    private function generateToken(): string
    {
        try {
            return bin2hex(random_bytes(16));
        } catch (\Throwable) {
            return md5((string) microtime(true) . ':' . (string) wp_rand());
        }
    }
}
