<?php

declare(strict_types=1);

namespace Agent\Modules\StyleGuide;

use Agent\Modules\OpenRouter\OpenRouterService;
use Agent\Support\Options;

final class StyleGuideModule
{
    private const POST_TYPE = 'wpai_style_guide';
    private const RECIPE_POST_TYPE = 'wpai_prompt_recipe';
    private const RECIPE_OPTION_MAPPING = 'agent_prompt_recipe_posts';
    private const RECIPE_PREFIX = 'prompt_recipe_';
    private const RULE_KEYS = ['style_manifest', 'style_exemplars', 'seo_rules'];

    private const DEFAULT_RULES = [
        [
            'key' => 'style_manifest',
            'slug' => 'stilmanifest',
            'title' => 'Stilmanifest',
            'content' => "Anrede: Du/Sie\nTon: Klar, hilfreich, praezise\nSatzlaenge: Kurz bis mittel\nStruktur: Problem -> Loesung -> naechster Schritt",
        ],
        [
            'key' => 'style_exemplars',
            'slug' => 'beispielabsaetze',
            'title' => 'Beispielabsätze',
            'content' => "Beispiel 1: Kurzer Absatz im Zielstil.\n\nBeispiel 2: Kurzer Absatz mit klarer Handlungsanweisung.\n\nBeispiel 3: Kurzer Absatz mit technischer Erklaerung.",
        ],
        [
            'key' => 'seo_rules',
            'slug' => 'seo-regeln',
            'title' => 'SEO-Regeln',
            'content' => "- H2 alle 300-500 Woerter\n- Codebloecke in Markdown\n- Keywords natuerlich platzieren\n- Klare Meta-Description-Struktur",
        ],
    ];

    public function register(): void
    {
        register_post_type(
            self::POST_TYPE,
            [
                'labels' => [
                    'name' => 'Blog Voice & Regeln',
                    'singular_name' => 'Style Guide Regel',
                    'add_new_item' => 'Neue Regel erstellen',
                    'edit_item' => 'Regel bearbeiten',
                ],
                'public' => false,
                'show_ui' => true,
                'show_in_menu' => 'agent',
                'menu_icon' => 'dashicons-edit-page',
                'supports' => ['title', 'editor', 'revisions'],
                'show_in_rest' => false,
            ]
        );

        add_filter('use_block_editor_for_post_type', [$this, 'disableBlockEditorForStyleGuide'], 10, 2);
        add_filter('wp_editor_settings', [$this, 'forceEasyMDETextareaEditor'], 10, 2);
        add_action('admin_init', [$this, 'ensureDefaultRulePosts']);
        add_action('add_meta_boxes', [$this, 'registerMetaBoxes']);
        add_action('admin_enqueue_scripts', [$this, 'enqueueStyleGuideEditorAssets']);
        add_action('save_post_' . self::POST_TYPE, [$this, 'saveMetaBoxes']);
        add_action('wp_ajax_wpai_style_guide_generate', [$this, 'ajaxGenerateRule']);
        add_action('manage_' . self::POST_TYPE . '_posts_custom_column', [$this, 'renderRuleKeyColumn'], 10, 2);
        add_filter('manage_' . self::POST_TYPE . '_posts_columns', [$this, 'addRuleKeyColumn']);
    }

    public function disableBlockEditorForStyleGuide(bool $useBlockEditor, string $postType): bool
    {
        if ($postType === self::POST_TYPE) {
            return false;
        }

        return $useBlockEditor;
    }

    public function forceEasyMDETextareaEditor(array $settings, string $editorId): array
    {
        if ($editorId !== 'content') {
            return $settings;
        }
        if (! $this->isStyleGuideEditorScreen()) {
            return $settings;
        }

        // EasyMDE takes over the plain textarea. Disable native visual/text tab stack.
        $settings['tinymce'] = false;
        $settings['quicktags'] = false;
        $settings['media_buttons'] = false;

        return $settings;
    }

    public function enqueueStyleGuideEditorAssets(string $hookSuffix): void
    {
        if (! in_array($hookSuffix, ['post.php', 'post-new.php'], true)) {
            return;
        }
        if (! $this->isStyleGuideEditorScreen()) {
            return;
        }

        wp_enqueue_style(
            'agent-easymde',
            plugin_dir_url(__FILE__) . '../../../assets/vendor/easymde/easymde.min.css',
            [],
            '2.20.0'
        );
        wp_add_inline_style(
            'agent-easymde',
            '.EasyMDEContainer .editor-toolbar a{background:transparent !important;}
            .EasyMDEContainer .editor-toolbar a:hover,
            .EasyMDEContainer .editor-toolbar a.active{background:transparent !important;}
            .EasyMDEContainer .editor-toolbar i.separator{border-left:1px solid #dcdcde;}
            .EasyMDEContainer .editor-toolbar #wpai-md-placeholder-select{height:28px;margin:0 6px 0 4px;padding:0 28px 0 8px;font-size:12px;border:1px solid #c3c4c7;border-radius:3px;background-color:#fff;background-image:url(\"data:image/svg+xml,%3Csvg xmlns=%27http://www.w3.org/2000/svg%27 width=%2710%27 height=%276%27 viewBox=%270 0 10 6%27%3E%3Cpath fill=%27%2350575e%27 d=%27M0 0l5 6 5-6z%27/%3E%3C/svg%3E\");background-repeat:no-repeat;background-position:right 8px center;background-size:10px 6px;-webkit-appearance:none;-moz-appearance:none;appearance:none;vertical-align:middle;}
            #wp-content-editor-tools{display:none !important;}'
        );
        wp_enqueue_script(
            'agent-easymde',
            plugin_dir_url(__FILE__) . '../../../assets/vendor/easymde/easymde.min.js',
            [],
            '2.20.0',
            true
        );
        $categoryLabel = __('Categories');
        $tagLabel = __('Tags');
        if (function_exists('get_taxonomy')) {
            $categoryTax = get_taxonomy('category');
            if (is_object($categoryTax) && isset($categoryTax->labels->name)) {
                $categoryLabel = (string) $categoryTax->labels->name;
            }
            $tagTax = get_taxonomy('post_tag');
            if (is_object($tagTax) && isset($tagTax->labels->name)) {
                $tagLabel = (string) $tagTax->labels->name;
            }
        }

        $placeholderOptions = [
            ['token' => '{{site_name}}', 'label' => __('Site Title')],
            ['token' => '{{site_tagline}}', 'label' => __('Tagline')],
            ['token' => '{{site_language}}', 'label' => __('Site Language')],
            ['token' => '{{site_url}}', 'label' => __('Site Address (URL)')],
            ['token' => '{{categories}}', 'label' => $categoryLabel],
            ['token' => '{{tags}}', 'label' => $tagLabel],
            ['token' => '{{recent_posts_titles}}', 'label' => __('Recent Post Titles', 'default')],
            ['token' => '{{recent_posts_excerpts}}', 'label' => __('Recent Post Excerpts', 'default')],
            ['token' => '{{recent_post_titles}}', 'label' => __('Recent Post Titles (Alias)', 'default')],
            ['token' => '{{recent_post_excerpts}}', 'label' => __('Recent Post Excerpts (Alias)', 'default')],
            ['token' => '{{seo_plugin}}', 'label' => __('SEO Plugin', 'default')],
            ['token' => '{{target_audience_hint}}', 'label' => __('Target Audience Hint', 'default')],
            ['token' => '{{post_count}}', 'label' => __('Post Count', 'default')],
            ['token' => '{{business_model_hint}}', 'label' => __('Business Model Hint', 'default')],
        ];
        $placeholderOptionsJson = wp_json_encode($placeholderOptions);
        $placeholderTitle = esc_js(__('WP Placeholders', 'default'));

        $inlineScript = <<<'JS'
(function () {
    if (window.__agentEasyMDEInitialized) {
        return;
    }

    var init = function () {
        var t = document.getElementById('content');
        if (!t || window.__agentEasyMDEInitialized) {
            return;
        }

        window.wpaiStyleGuideEasyMDE = new EasyMDE({
            element: t,
            autoDownloadFontAwesome: true,
            spellChecker: false,
            status: ['lines', 'words'],
            forceSync: true,
            toolbar: ['bold', 'italic', 'heading', '|', 'quote', 'unordered-list', 'ordered-list', '|', 'link', 'image', 'table', 'horizontal-rule', '|', 'preview', 'guide']
        });

        var container = t.closest('.EasyMDEContainer') || t.parentNode;
        if (container) {
            var toolbar = container.querySelector('.editor-toolbar');
            if (toolbar && !document.getElementById('wpai-md-placeholder-select')) {
                var placeholders = __WPAI_PLACEHOLDERS__;

                var select = document.createElement('select');
                select.id = 'wpai-md-placeholder-select';
                select.setAttribute('aria-label', '__WPAI_PLACEHOLDER_TITLE__');

                var placeholderOption = document.createElement('option');
                placeholderOption.value = '';
                placeholderOption.textContent = '__WPAI_PLACEHOLDER_TITLE__';
                select.appendChild(placeholderOption);

                placeholders.forEach(function (item) {
                    var opt = document.createElement('option');
                    opt.value = item.token;
                    opt.textContent = item.label + ' (' + item.token + ')';
                    select.appendChild(opt);
                });

                select.addEventListener('change', function () {
                    var token = select.value;
                    if (!token || !window.wpaiStyleGuideEasyMDE || !window.wpaiStyleGuideEasyMDE.codemirror) {
                        return;
                    }
                    var cm = window.wpaiStyleGuideEasyMDE.codemirror;
                    cm.replaceSelection(token);
                    cm.focus();
                    select.value = '';
                });

                toolbar.appendChild(select);
            }
        }

        if (container && !document.getElementById('wpai-easymde-attribution')) {
            var note = document.createElement('p');
            note.id = 'wpai-easymde-attribution';
            note.style.margin = '8px 0 0';
            note.style.fontSize = '12px';
            note.style.color = '#646970';
            note.innerHTML = 'EasyMDE by <a href="https://github.com/Ionaru/easy-markdown-editor" target="_blank" rel="noopener noreferrer">Jeroen Akkerman</a>';
            container.parentNode.insertBefore(note, container.nextSibling);
        }

        window.__agentEasyMDEInitialized = true;
    };

    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }
})();
JS;
        $inlineScript = str_replace('__WPAI_PLACEHOLDERS__', (string) $placeholderOptionsJson, $inlineScript);
        $inlineScript = str_replace('__WPAI_PLACEHOLDER_TITLE__', $placeholderTitle, $inlineScript);
        wp_add_inline_script('agent-easymde', $inlineScript, 'after');
    }

    private function isStyleGuideEditorScreen(): bool
    {
        $postType = '';

        if (function_exists('get_current_screen')) {
            $screen = get_current_screen();
            if (is_object($screen) && isset($screen->post_type)) {
                $postType = (string) $screen->post_type;
            }
        }

        if ($postType === '' && isset($_GET['post_type'])) {
            $postType = sanitize_key((string) $_GET['post_type']);
        }

        if ($postType === '' && isset($_GET['post'])) {
            $post = get_post((int) $_GET['post']);
            if ($post instanceof \WP_Post) {
                $postType = (string) $post->post_type;
            }
        }

        if ($postType === '' && isset($_POST['post_type'])) {
            $postType = sanitize_key((string) $_POST['post_type']);
        }

        return $postType === self::POST_TYPE;
    }

    public function ensureDefaultRulePosts(): void
    {
        if (! current_user_can('manage_options')) {
            return;
        }

        $this->upsertDefaultRulePosts();
    }

    public static function seedDefaultRulePosts(): void
    {
        $module = new self();
        $module->upsertDefaultRulePosts();
    }

    public function registerMetaBoxes(): void
    {
        add_meta_box(
            'wpai-style-guide-rule-assignment',
            'Regel-Zuordnung',
            [$this, 'renderRuleAssignmentMetaBox'],
            self::POST_TYPE,
            'side',
            'high'
        );

        add_meta_box(
            'wpai-style-guide-openrouter',
            'OpenRouter',
            [$this, 'renderOpenRouterMetaBox'],
            self::POST_TYPE,
            'side',
            'high'
        );
    }

    public function renderOpenRouterMetaBox(\WP_Post $post): void
    {
        wp_nonce_field('wpai_style_guide_openrouter_save', 'wpai_style_guide_openrouter_nonce');

        $model = (string) get_post_meta($post->ID, '_wpai_or_model', true);
        $temperature = (string) get_post_meta($post->ID, '_wpai_or_temperature', true);
        $maxTokens = (string) get_post_meta($post->ID, '_wpai_or_max_tokens', true);

        $settings = get_option('agent_settings', []);
        $openRouter = is_array($settings['openrouter'] ?? null) ? $settings['openrouter'] : [];
        $enabled = (bool) ($openRouter['enabled'] ?? false);
        $hasKey = ((string) ($openRouter['api_key'] ?? '')) !== '';
        $ajaxNonce = wp_create_nonce('wpai_style_guide_generate');
        $postId = (int) $post->ID;

        ?>
        <p><strong>Status:</strong> <?php echo esc_html($enabled ? ($hasKey ? 'Aktiv' : 'Kein API-Key') : 'Deaktiviert'); ?></p>
        <p>
            <label for="wpai-or-model"><strong>Modell</strong></label><br />
            <input id="wpai-or-model" type="text" style="width:100%" name="wpai_or_model" value="<?php echo esc_attr($model); ?>" placeholder="openrouter/free" />
        </p>
        <p>
            <label for="wpai-or-temperature"><strong>Temperature</strong></label><br />
            <input id="wpai-or-temperature" type="number" step="0.1" min="0" max="2" style="width:100%" name="wpai_or_temperature" value="<?php echo esc_attr($temperature); ?>" placeholder="0.7" />
        </p>
        <p>
            <label for="wpai-or-max-tokens"><strong>Max Tokens</strong></label><br />
            <input id="wpai-or-max-tokens" type="number" min="1" max="200000" style="width:100%" name="wpai_or_max_tokens" value="<?php echo esc_attr($maxTokens); ?>" placeholder="1200" />
        </p>
        <p>
            <button type="button" class="button button-primary" id="wpai-or-auto-generate">Autogenerieren</button>
            <button type="button" class="button" id="wpai-or-context-generate">Generieren aus Kontext</button>
        </p>
        <p id="wpai-or-generation-status"><em></em></p>
        <script>
            (function () {
                var autoBtn = document.getElementById('wpai-or-auto-generate');
                var ctxBtn = document.getElementById('wpai-or-context-generate');
                var statusEl = document.getElementById('wpai-or-generation-status');
                if (!autoBtn || !ctxBtn || !statusEl) {
                    return;
                }

                function getEditorContent() {
                    if (window.wpaiStyleGuideEasyMDE && typeof window.wpaiStyleGuideEasyMDE.value === 'function') {
                        return window.wpaiStyleGuideEasyMDE.value() || '';
                    }
                    if (window.tinyMCE && window.tinyMCE.get('content')) {
                        return window.tinyMCE.get('content').getContent({format: 'text'}) || '';
                    }
                    var contentEl = document.getElementById('content');
                    return contentEl ? (contentEl.value || '') : '';
                }

                function setEditorContent(text) {
                    if (window.wpaiStyleGuideEasyMDE && typeof window.wpaiStyleGuideEasyMDE.value === 'function') {
                        window.wpaiStyleGuideEasyMDE.value(text || '');
                    }
                    if (window.tinyMCE && window.tinyMCE.get('content')) {
                        window.tinyMCE.get('content').setContent(text);
                    }
                    var contentEl = document.getElementById('content');
                    if (contentEl) {
                        contentEl.value = text;
                    }
                }

                function setBusy(isBusy, text) {
                    autoBtn.disabled = isBusy;
                    ctxBtn.disabled = isBusy;
                    statusEl.innerHTML = '<em>' + (text || '') + '</em>';
                }

                function requestGeneration(mode) {
                    setBusy(true, 'Generierung laeuft...');
                    var body = new URLSearchParams();
                    body.set('action', 'wpai_style_guide_generate');
                    body.set('nonce', <?php echo wp_json_encode($ajaxNonce); ?>);
                    body.set('post_id', <?php echo wp_json_encode((string) $postId); ?>);
                    body.set('mode', mode);
                    if (mode === 'context') {
                        body.set('current_content', getEditorContent());
                    }

                    fetch(ajaxurl, {
                        method: 'POST',
                        headers: {'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8'},
                        body: body.toString(),
                        credentials: 'same-origin'
                    }).then(function (res) {
                        return res.text().then(function (text) {
                            var parsed = null;
                            try {
                                parsed = JSON.parse(text);
                            } catch (e) {
                                parsed = {success: false, data: {message: 'Serverantwort ist kein gueltiges JSON. HTTP ' + res.status + '.'}};
                            }
                            return parsed;
                        });
                    }).then(function (json) {
                        if (!json || !json.success) {
                            var msg = json && json.data && json.data.message ? json.data.message : 'Generierung fehlgeschlagen.';
                            setBusy(false, msg);
                            return;
                        }
                        if (json.data && json.data.content) {
                            setEditorContent(json.data.content);
                        }
                        setBusy(false, 'Generierung erfolgreich gespeichert.');
                    }).catch(function () {
                        setBusy(false, 'Netzwerkfehler bei der Generierung.');
                    });
                }

                autoBtn.addEventListener('click', function () {
                    requestGeneration('auto');
                });
                ctxBtn.addEventListener('click', function () {
                    requestGeneration('context');
                });
            })();
        </script>
        <?php
    }

    public function ajaxGenerateRule(): void
    {
        try {
        $nonce = isset($_POST['nonce']) ? (string) $_POST['nonce'] : '';
        if (! wp_verify_nonce($nonce, 'wpai_style_guide_generate')) {
            wp_send_json_error(['message' => 'Ungueltige Anfrage.'], 403);
        }

        $postId = isset($_POST['post_id']) ? (int) $_POST['post_id'] : 0;
        $mode = isset($_POST['mode']) ? sanitize_key((string) $_POST['mode']) : 'auto';
        $currentContent = isset($_POST['current_content']) ? wp_unslash((string) $_POST['current_content']) : '';
        if (! in_array($mode, ['auto', 'context'], true)) {
            $mode = 'auto';
        }

        $post = get_post($postId);
        if (! ($post instanceof \WP_Post) || $post->post_type !== self::POST_TYPE) {
            wp_send_json_error(['message' => 'Ungueltiger Regel-Post.'], 400);
        }

        if (! current_user_can('edit_post', $postId)) {
            wp_send_json_error(['message' => 'Keine Berechtigung.'], 403);
        }

        $settings = (array) get_option('agent_settings', []);
        $openRouter = is_array($settings['openrouter'] ?? null) ? $settings['openrouter'] : [];
        if (! (bool) ($openRouter['enabled'] ?? false)) {
            wp_send_json_error(['message' => 'OpenRouter ist deaktiviert.'], 400);
        }
        if (((string) ($openRouter['api_key'] ?? '')) === '') {
            wp_send_json_error(['message' => 'OpenRouter API-Key fehlt.'], 400);
        }

        $service = new OpenRouterService(new Options('agent_settings', \Agent\Plugin::defaultSettings()));
        $ruleKey = (string) get_post_meta($postId, '_wpai_rule_key', true);
        $model = (string) get_post_meta($postId, '_wpai_or_model', true);
        if ($model === '') {
            $model = (string) ($openRouter['writer_model'] ?? 'openrouter/free');
        }
        if ($model === '') {
            $model = 'openrouter/free';
        }

        $temperatureRaw = (string) get_post_meta($postId, '_wpai_or_temperature', true);
        $maxTokensRaw = (string) get_post_meta($postId, '_wpai_or_max_tokens', true);
        $temperature = $temperatureRaw === '' ? 0.4 : max(0.0, min(2.0, (float) $temperatureRaw));
        $defaultMaxTokens = $ruleKey === 'seo_rules' ? 3000 : 1200;
        $maxTokens = $maxTokensRaw === '' ? $defaultMaxTokens : max(100, min(12000, (int) $maxTokensRaw));

        $prompt = $mode === 'context'
            ? $this->buildContextPrompt($post, $ruleKey, $currentContent)
            : $this->buildAutoPrompt($post, $ruleKey);

        $payload = [
            'model' => $model,
            'temperature' => $temperature,
            'max_tokens' => $maxTokens,
            'messages' => [
                [
                    'role' => 'system',
                    'content' => 'Du erstellst strukturierte Regeltexte fuer WordPress Blog Voice und SEO in deutscher Sprache.',
                ],
                [
                    'role' => 'user',
                    'content' => $prompt,
                ],
            ],
        ];

        $result = $service->chatCompletions($payload);
        if (is_wp_error($result)) {
            $status = (int) ($result->data['status'] ?? 502);
            $error = [
                'message' => $result->get_error_message(),
            ];
            if (isset($result->data['retry_after'])) {
                $error['retry_after'] = (int) $result->data['retry_after'];
            }
            wp_send_json_error($error, $status);
        }

        $generated = trim($this->extractGeneratedText($result));
        if ($generated === '') {
            wp_send_json_error(['message' => 'Leere Antwort vom Modell.'], 502);
        }

        if ($this->isTruncatedResponse($result)) {
            $continued = $this->completeTruncatedGeneration($service, $payload, $generated, 3);
            if (is_wp_error($continued)) {
                $status = (int) ($continued->data['status'] ?? 502);
                wp_send_json_error(['message' => $continued->get_error_message()], $status);
            }
            $generated = trim($continued);
        }

        wp_update_post(
            [
                'ID' => $postId,
                'post_content' => $generated,
            ]
        );

        wp_send_json_success(
            [
                'content' => $generated,
                'mode' => $mode,
                'model' => $model,
            ]
        );
        } catch (\Throwable $e) {
            wp_send_json_error(
                [
                    'message' => 'Serverfehler: ' . $e->getMessage(),
                ],
                500
            );
        }
    }

    private function isTruncatedResponse(array $result): bool
    {
        $choices = $result['choices'] ?? null;
        if (! is_array($choices) || ! isset($choices[0]) || ! is_array($choices[0])) {
            return false;
        }

        $finishReason = (string) ($choices[0]['finish_reason'] ?? '');
        return $finishReason === 'length' || $finishReason === 'max_tokens';
    }

    private function completeTruncatedGeneration(OpenRouterService $service, array $basePayload, string $initialText, int $maxRounds): string|\WP_Error
    {
        $combined = $initialText;
        $rounds = max(1, $maxRounds);

        for ($i = 0; $i < $rounds; $i++) {
            $messages = $basePayload['messages'] ?? [];
            if (! is_array($messages)) {
                $messages = [];
            }

            $messages[] = [
                'role' => 'assistant',
                'content' => $combined,
            ];
            $messages[] = [
                'role' => 'user',
                'content' => 'Bitte direkt nahtlos fortsetzen. Nur der fehlende Rest, ohne Wiederholung.',
            ];

            $nextPayload = $basePayload;
            $nextPayload['messages'] = $messages;

            $next = $service->chatCompletions($nextPayload);
            if (is_wp_error($next)) {
                return $next;
            }

            $part = trim($this->extractGeneratedText($next));
            if ($part === '') {
                break;
            }

            $combined = rtrim($combined) . "\n" . $part;

            if (! $this->isTruncatedResponse($next)) {
                break;
            }
        }

        return $combined;
    }

    public function renderRuleAssignmentMetaBox(\WP_Post $post): void
    {
        wp_nonce_field('wpai_style_guide_rule_assignment_save', 'wpai_style_guide_rule_assignment_nonce');
        $current = (string) get_post_meta($post->ID, '_wpai_rule_key', true);

        echo '<p><strong>Maschinenlesbarer Regelschluessel</strong></p>';
        echo '<p><select name="wpai_rule_key" style="width:100%;">';
        echo '<option value="">Keine Zuordnung</option>';
        foreach (self::RULE_KEYS as $ruleKey) {
            echo '<option value="' . esc_attr($ruleKey) . '" ' . selected($current, $ruleKey, false) . '>' . esc_html($ruleKey) . '</option>';
        }
        echo '</select></p>';
        echo '<p><em>Jeder Schluessel darf nur einem Post zugeordnet sein.</em></p>';
    }

    public function saveMetaBoxes(int $postId): void
    {
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }

        if (! current_user_can('edit_post', $postId)) {
            return;
        }

        if (isset($_POST['wpai_style_guide_openrouter_nonce']) && wp_verify_nonce((string) $_POST['wpai_style_guide_openrouter_nonce'], 'wpai_style_guide_openrouter_save')) {
            $model = isset($_POST['wpai_or_model']) ? sanitize_text_field((string) $_POST['wpai_or_model']) : '';
            $temperature = isset($_POST['wpai_or_temperature']) ? (string) max(0, min(2, (float) $_POST['wpai_or_temperature'])) : '';
            $maxTokens = isset($_POST['wpai_or_max_tokens']) ? (string) max(1, (int) $_POST['wpai_or_max_tokens']) : '';

            update_post_meta($postId, '_wpai_or_model', $model);
            update_post_meta($postId, '_wpai_or_temperature', $temperature);
            update_post_meta($postId, '_wpai_or_max_tokens', $maxTokens);
        }

        if (isset($_POST['wpai_style_guide_rule_assignment_nonce']) && wp_verify_nonce((string) $_POST['wpai_style_guide_rule_assignment_nonce'], 'wpai_style_guide_rule_assignment_save')) {
            $requested = isset($_POST['wpai_rule_key']) ? sanitize_key((string) $_POST['wpai_rule_key']) : '';
            if (! in_array($requested, self::RULE_KEYS, true)) {
                $requested = '';
            }

            if ($requested === '') {
                delete_post_meta($postId, '_wpai_rule_key');
            } else {
                $conflicts = get_posts(
                    [
                        'post_type' => self::POST_TYPE,
                        'post_status' => ['publish', 'draft', 'pending', 'private', 'future'],
                        'posts_per_page' => -1,
                        'fields' => 'ids',
                        'meta_key' => '_wpai_rule_key',
                        'meta_value' => $requested,
                        'exclude' => [$postId],
                    ]
                );
                foreach ($conflicts as $conflictId) {
                    delete_post_meta((int) $conflictId, '_wpai_rule_key');
                }
                update_post_meta($postId, '_wpai_rule_key', $requested);
            }
            $this->syncRuleMapping();
        }
    }

    public function addRuleKeyColumn(array $columns): array
    {
        $out = [];
        foreach ($columns as $key => $label) {
            $out[$key] = $label;
            if ($key === 'title') {
                $out['wpai_rule_key'] = 'Regelschlüssel';
            }
        }

        return $out;
    }

    public function renderRuleKeyColumn(string $column, int $postId): void
    {
        if ($column !== 'wpai_rule_key') {
            return;
        }

        $ruleKey = (string) get_post_meta($postId, '_wpai_rule_key', true);
        echo esc_html($ruleKey !== '' ? $ruleKey : '-');
    }

    private function upsertDefaultRulePosts(): void
    {
        foreach (self::DEFAULT_RULES as $rule) {
            $ruleKey = (string) ($rule['key'] ?? '');
            $slug = (string) ($rule['slug'] ?? '');
            $title = (string) ($rule['title'] ?? '');
            $content = (string) ($rule['content'] ?? '');

            if ($ruleKey === '' || $slug === '' || $title === '') {
                continue;
            }

            $existing = get_page_by_path($slug, OBJECT, self::POST_TYPE);

            if (! ($existing instanceof \WP_Post)) {
                $newId = wp_insert_post(
                    [
                        'post_type' => self::POST_TYPE,
                        'post_status' => 'publish',
                        'post_name' => $slug,
                        'post_title' => $title,
                        'post_content' => $content,
                    ]
                );
                if (is_int($newId) && $newId > 0) {
                    update_post_meta($newId, '_wpai_rule_key', $ruleKey);
                }
                continue;
            }

            if ($existing->post_title !== $title || $existing->post_name !== $slug) {
                wp_update_post(
                    [
                        'ID' => $existing->ID,
                        'post_title' => $title,
                        'post_name' => $slug,
                    ]
                );
            }

            update_post_meta($existing->ID, '_wpai_rule_key', $ruleKey);
        }
        $this->syncRuleMapping();
    }

    private function syncRuleMapping(): void
    {
        $mapping = [];
        $posts = get_posts(
            [
                'post_type' => self::POST_TYPE,
                'post_status' => ['publish', 'draft', 'pending', 'private', 'future'],
                'posts_per_page' => -1,
                'fields' => 'ids',
                'orderby' => 'ID',
                'order' => 'ASC',
            ]
        );

        foreach ($posts as $id) {
            $ruleKey = (string) get_post_meta((int) $id, '_wpai_rule_key', true);
            if (in_array($ruleKey, self::RULE_KEYS, true)) {
                $mapping[$ruleKey] = (int) $id;
            }
        }

        update_option('agent_style_guide_rule_posts', $mapping, false);
    }

    private function buildAutoPrompt(\WP_Post $post, string $ruleKey): string
    {
        $siteName = (string) get_bloginfo('name');
        $tagline = (string) get_bloginfo('description');
        $language = (string) get_bloginfo('language');
        $siteUrl = (string) get_site_url();
        $categories = $this->collectTermNames('category', 20);
        $tags = $this->collectTermNames('post_tag', 25);
        $recentTitles = $this->collectRecentPostTitles(8);
        $recentExcerpts = $this->collectRecentPostExcerpts(5);
        $targetAudienceHint = (string) get_post_meta((int) $post->ID, '_wpai_target_audience_hint', true);
        $businessModelHint = (string) get_post_meta((int) $post->ID, '_wpai_business_model_hint', true);
        $seoPlugin = $this->detectSeoPlugin();
        $postCount = $this->getPublishedPostCount();
        if ($targetAudienceHint === '') {
            $targetAudienceHint = '(none)';
        }
        if ($businessModelHint === '') {
            $businessModelHint = '(unknown)';
        }

        $template = $this->loadPromptRecipeTemplate($ruleKey);
        if ($template === '') {
            $template = $this->defaultPromptTemplateForRuleKey($ruleKey);
        }

        $replacements = [
            '{{site_name}}' => $siteName !== '' ? $siteName : '(empty)',
            '{{site_tagline}}' => $tagline !== '' ? $tagline : '(empty)',
            '{{site_language}}' => $language !== '' ? $language : '(empty)',
            '{{site_url}}' => $siteUrl !== '' ? $siteUrl : '(empty)',
            '{{categories}}' => $categories !== '' ? $categories : '(none)',
            '{{tags}}' => $tags !== '' ? $tags : '(none)',
            '{{recent_posts_titles}}' => $recentTitles !== '' ? $recentTitles : '(none)',
            '{{recent_posts_excerpts}}' => $recentExcerpts !== '' ? $recentExcerpts : '(none)',
            '{{recent_post_titles}}' => $recentTitles !== '' ? $recentTitles : '(none)',
            '{{recent_post_excerpts}}' => $recentExcerpts !== '' ? $recentExcerpts : '(none)',
            '{{seo_plugin}}' => $seoPlugin,
            '{{target_audience_hint}}' => $targetAudienceHint,
            '{{post_count}}' => (string) $postCount,
            '{{business_model_hint}}' => $businessModelHint,
        ];

        return strtr($template, $replacements);
    }

    private function loadPromptRecipeTemplate(string $ruleKey): string
    {
        $recipeKey = self::RECIPE_PREFIX . $ruleKey;
        $mapping = get_option(self::RECIPE_OPTION_MAPPING, []);
        if (! is_array($mapping)) {
            return '';
        }
        $recipePostId = isset($mapping[$recipeKey]) ? (int) $mapping[$recipeKey] : 0;
        if ($recipePostId <= 0) {
            return '';
        }

        $recipePost = get_post($recipePostId);
        if (! ($recipePost instanceof \WP_Post) || $recipePost->post_type !== self::RECIPE_POST_TYPE) {
            return '';
        }

        return trim((string) $recipePost->post_content);
    }

    private function defaultPromptTemplateForRuleKey(string $ruleKey): string
    {
        return match ($ruleKey) {
            'seo_rules' => $this->defaultSeoPromptTemplate(),
            'style_exemplars' => $this->defaultStyleExemplarsPromptTemplate(),
            default => $this->defaultStyleGuidePromptTemplate(),
        };
    }

    private function defaultStyleGuidePromptTemplate(): string
    {
        return <<<'PROMPT'
You are an expert editorial strategist and content architect.

Your task is to generate a comprehensive, professional style guide for a WordPress-based website using only the information provided below.

Website Name:
{{site_name}}

Tagline / Description:
{{site_tagline}}

Primary Language:
{{site_language}}

Categories:
{{categories}}

Tags:
{{tags}}

Recent Post Titles:
{{recent_posts_titles}}

Recent Post Excerpts:
{{recent_posts_excerpts}}

Optional Target Audience Hint:
{{target_audience_hint}}

Instructions:

1. Infer the website’s primary focus, expertise level, and positioning.
2. Deduce tone, voice tendencies, and implied brand personality.
3. Identify recurring themes or subject clusters.
4. Determine the likely target audience (be specific).
5. Propose a consistent terminology policy.
6. Define formatting and structural standards.
7. Establish SEO tone guidance (without keyword stuffing).
8. Provide rules for internal linking philosophy.
9. Define clarity standards (sentence length, paragraph size, use of examples).
10. Suggest do’s and don’ts to prevent stylistic drift.

Output Requirements:

- Write a structured, production-ready style guide.
- Use clear section headings.
- Be prescriptive and actionable.
- Do not speculate wildly; base conclusions only on provided data.
- If information is insufficient, define adaptable guidance instead of guessing.
- Do not include meta commentary.
- Do not mention AI or prompt mechanics.

Structure the output as:

1. Editorial Positioning
2. Target Audience Profile
3. Brand Voice & Tone
4. Terminology & Language Rules
5. Structural & Formatting Guidelines
6. SEO & Discoverability Philosophy
7. Internal Linking Strategy
8. Content Depth & Technical Expectations
9. Consistency Rules
10. Style Guardrails (Do’s and Don’ts)
PROMPT;
    }

    private function defaultStyleExemplarsPromptTemplate(): string
    {
        return <<<'PROMPT'
You are an expert editorial strategist and content architect.

Your task is to generate 3-5 short style exemplar paragraphs for a WordPress-based website using only the information provided below.

Website Name:
{{site_name}}

Tagline / Description:
{{site_tagline}}

Primary Language:
{{site_language}}

Categories:
{{categories}}

Tags:
{{tags}}

Recent Post Titles:
{{recent_posts_titles}}

Recent Post Excerpts:
{{recent_posts_excerpts}}

Optional Target Audience Hint:
{{target_audience_hint}}

Instructions:

1. Infer voice and tone from the provided data.
2. Produce compact, realistic exemplar paragraphs in the inferred brand style.
3. Vary structure and intent between examples (intro, explanation, CTA, practical tip).
4. Keep each example concise and production-ready.

Output Requirements:

- Return 3-5 clearly separated paragraphs.
- Use markdown.
- No meta commentary.
- No mention of AI or prompt mechanics.
PROMPT;
    }

    private function defaultSeoPromptTemplate(): string
    {
        return <<<'PROMPT'
You are an expert Technical SEO strategist and content systems architect.

Your task is to generate a comprehensive, best-practice SEO ruleset tailored specifically to the WordPress website described below.

Website Information:
Site Name: {{site_name}}
Tagline / Description: {{site_tagline}}
Primary Language: {{site_language}}
Site URL: {{site_url}}

Categories:
{{categories}}

Tags:
{{tags}}

Recent Post Titles:
{{recent_post_titles}}

Recent Post Excerpts:
{{recent_post_excerpts}}

Existing SEO Plugin (if any):
{{seo_plugin}}

Target Audience Hint (optional):
{{target_audience_hint}}

Content Volume:
{{post_count}}

Primary Business Model (if known):
{{business_model_hint}}

Instructions:

1. Analyze the thematic structure and topic clusters implied by categories and tags.
2. Infer search intent patterns from post titles and excerpts.
3. Deduce content depth level (beginner, intermediate, expert).
4. Identify whether the site appears informational, transactional, commercial, or mixed.
5. Generate a best-practice SEO ruleset tailored to this specific site profile.
6. Base recommendations only on the information provided. If data is insufficient, provide adaptable best-practice guidance instead of speculation.
7. Avoid generic SEO clichés. Provide actionable, system-level rules.

The ruleset must include the following structured sections:

1. SEO Positioning Strategy
   - Core search intent focus
   - Topic authority strategy
   - Pillar/cluster model recommendations

2. On-Page SEO Standards
   - Title tag rules
   - Meta description rules
   - Heading hierarchy standards
   - URL structure guidelines
   - Internal anchor text policy
   - Image optimization standards

3. Content Optimization Framework
   - Keyword integration philosophy
   - Semantic SEO recommendations
   - Content depth expectations
   - E-E-A-T alignment guidelines

4. Internal Linking Architecture
   - Topic clustering rules
   - Link density standards
   - Anchor text variation policy
   - Orphan page prevention rules

5. Technical SEO Baselines (WordPress-Specific)
   - Sitemap standards
   - Indexing control rules
   - Canonicalization strategy
   - Pagination handling
   - Category and tag archive treatment
   - Performance expectations (Core Web Vitals principles)

6. Structured Data Recommendations
   - Schema types to implement
   - Article schema standards
   - Breadcrumb guidelines

7. Content Governance Rules
   - Update cadence policy
   - Thin content detection rules
   - Content consolidation strategy
   - Duplicate topic avoidance rules

8. Scaling Strategy (Based on {{post_count}})
   - Authority building roadmap
   - Expansion into related keyword clusters
   - Long-term topical dominance strategy

Output Requirements:

- Produce a structured, production-ready SEO ruleset.
- Be prescriptive and operational.
- Avoid meta commentary.
- Do not reference AI or prompt mechanics.
- Do not use vague phrases like “it depends.”
- Provide rules that can be implemented directly in a WordPress workflow.
- Keep language precise, professional, and implementation-focused.
PROMPT;
    }

    private function buildContextPrompt(\WP_Post $post, string $ruleKey, string $currentContent): string
    {
        $base = $this->buildAutoPrompt($post, $ruleKey);
        $otherRules = $this->collectOtherRuleSummaries((int) $post->ID);

        return implode("\n\n", [
            $base,
            'Zusatzkontext fuer "Generieren aus Kontext":',
            'Aktueller Inhalt dieses Regelposts:',
            $currentContent !== '' ? $currentContent : '(leer)',
            'Weitere vorhandene Regelposts:',
            $otherRules !== '' ? $otherRules : '(keine)',
            'Achte auf Konsistenz zwischen den Regelposts und aktualisiere den Text entsprechend.',
        ]);
    }

    private function collectOtherRuleSummaries(int $currentPostId): string
    {
        $posts = get_posts(
            [
                'post_type' => self::POST_TYPE,
                'post_status' => ['publish', 'draft', 'pending', 'private', 'future'],
                'posts_per_page' => -1,
                'exclude' => [$currentPostId],
            ]
        );

        $lines = [];
        foreach ($posts as $post) {
            if (! ($post instanceof \WP_Post)) {
                continue;
            }
            $key = (string) get_post_meta((int) $post->ID, '_wpai_rule_key', true);
            $content = trim((string) $post->post_content);
            if ($content !== '') {
                $content = substr($content, 0, 800);
            }
            $lines[] = '- ' . $post->post_title . ' [' . $key . "]:\n" . ($content !== '' ? $content : '(leer)');
        }

        return implode("\n\n", $lines);
    }

    private function extractGeneratedText(array $result): string
    {
        $choices = $result['choices'] ?? null;
        if (! is_array($choices) || ! isset($choices[0]) || ! is_array($choices[0])) {
            return '';
        }

        $first = $choices[0];
        $message = $first['message'] ?? null;
        if (is_array($message)) {
            $content = $message['content'] ?? '';
            if (is_string($content)) {
                return $content;
            }
            if (is_array($content)) {
                $parts = [];
                foreach ($content as $item) {
                    if (! is_array($item)) {
                        continue;
                    }
                    if (($item['type'] ?? '') === 'text' && is_string($item['text'] ?? null)) {
                        $parts[] = $item['text'];
                    }
                }
                return trim(implode("\n", $parts));
            }
        }

        if (is_string($first['text'] ?? null)) {
            return (string) $first['text'];
        }

        return '';
    }

    private function collectTermNames(string $taxonomy, int $limit): string
    {
        if (! function_exists('get_terms')) {
            return '';
        }

        $terms = get_terms(
            [
                'taxonomy' => $taxonomy,
                'hide_empty' => false,
                'number' => $limit,
                'orderby' => 'count',
                'order' => 'DESC',
            ]
        );
        if (! is_array($terms) || is_wp_error($terms)) {
            return '';
        }

        $names = [];
        foreach ($terms as $term) {
            $name = '';
            if (is_object($term) && isset($term->name)) {
                $name = trim((string) $term->name);
            } elseif (is_array($term) && isset($term['name'])) {
                $name = trim((string) $term['name']);
            }
            if ($name !== '') {
                $names[] = $name;
            }
        }

        return $names === [] ? '' : implode(', ', array_slice($names, 0, $limit));
    }

    private function collectRecentPostTitles(int $limit): string
    {
        $posts = $this->loadRecentPosts($limit);
        if ($posts === []) {
            return '';
        }

        $titles = [];
        foreach ($posts as $post) {
            if (! ($post instanceof \WP_Post)) {
                continue;
            }
            $title = trim((string) $post->post_title);
            if ($title !== '') {
                $titles[] = '- ' . $title;
            }
        }

        return implode("\n", $titles);
    }

    private function collectRecentPostExcerpts(int $limit): string
    {
        $posts = $this->loadRecentPosts($limit);
        if ($posts === []) {
            return '';
        }

        $rows = [];
        foreach ($posts as $post) {
            if (! ($post instanceof \WP_Post)) {
                continue;
            }
            $title = trim((string) $post->post_title);
            $excerpt = trim((string) $post->post_excerpt);
            if ($excerpt === '') {
                $content = trim((string) $post->post_content);
                if ($content !== '') {
                    if (function_exists('wp_strip_all_tags')) {
                        $content = wp_strip_all_tags($content);
                    }
                    if (function_exists('wp_trim_words')) {
                        $excerpt = (string) wp_trim_words($content, 40, '...');
                    } else {
                        $excerpt = substr($content, 0, 260);
                    }
                }
            }
            if ($excerpt === '') {
                continue;
            }

            $label = $title !== '' ? $title : ('Post #' . (string) $post->ID);
            $rows[] = '- ' . $label . ': ' . $excerpt;
        }

        return implode("\n", $rows);
    }

    private function loadRecentPosts(int $limit): array
    {
        $posts = get_posts(
            [
                'post_type' => 'post',
                'post_status' => 'publish',
                'posts_per_page' => $limit,
                'orderby' => 'date',
                'order' => 'DESC',
            ]
        );

        return is_array($posts) ? $posts : [];
    }

    private function getPublishedPostCount(): int
    {
        if (! function_exists('wp_count_posts')) {
            return 0;
        }

        $countObj = wp_count_posts('post');
        if (is_object($countObj) && isset($countObj->publish)) {
            return max(0, (int) $countObj->publish);
        }

        return 0;
    }

    private function detectSeoPlugin(): string
    {
        $active = get_option('active_plugins', []);
        if (! is_array($active) || $active === []) {
            return '(none)';
        }

        $known = [
            'wordpress-seo/wp-seo.php' => 'Yoast SEO',
            'seo-by-rank-math/rank-math.php' => 'Rank Math',
            'all-in-one-seo-pack/all_in_one_seo_pack.php' => 'All in One SEO',
            'autodescription/autodescription.php' => 'The SEO Framework',
            'seopress/seopress.php' => 'SEOPress',
        ];

        foreach ($known as $pluginFile => $label) {
            if (in_array($pluginFile, $active, true)) {
                return $label;
            }
        }

        return '(none detected)';
    }
}
