<?php

declare(strict_types=1);

namespace Agent\Modules\PromptRecipe;

final class PromptRecipeModule
{
    private const POST_TYPE = 'wpai_prompt_recipe';
    private const OPTION_MAPPING = 'agent_prompt_recipe_posts';
    private const RECIPE_KEYS = [
        'prompt_recipe_style_manifest',
        'prompt_recipe_style_exemplars',
        'prompt_recipe_seo_rules',
    ];

    private const DEFAULT_RECIPES = [
        [
            'key' => 'prompt_recipe_style_manifest',
            'slug' => 'prompt-recipe-style-manifest',
            'title' => 'Prompt Recipe Style Manifest',
        ],
        [
            'key' => 'prompt_recipe_style_exemplars',
            'slug' => 'prompt-recipe-style-exemplars',
            'title' => 'Prompt Recipe Style Exemplars',
        ],
        [
            'key' => 'prompt_recipe_seo_rules',
            'slug' => 'prompt-recipe-seo-rules',
            'title' => 'Prompt Recipe SEO Rules',
        ],
    ];

    public function register(): void
    {
        register_post_type(
            self::POST_TYPE,
            [
                'labels' => [
                    'name' => 'Prompt Recipes',
                    'singular_name' => 'Prompt Recipe',
                    'add_new_item' => 'Neues Prompt Recipe erstellen',
                    'edit_item' => 'Prompt Recipe bearbeiten',
                ],
                'public' => false,
                'show_ui' => true,
                'show_in_menu' => 'agent',
                'menu_icon' => 'dashicons-editor-code',
                'supports' => ['title', 'editor', 'revisions'],
                'show_in_rest' => false,
            ]
        );

        add_filter('use_block_editor_for_post_type', [$this, 'disableBlockEditorForPromptRecipe'], 10, 2);
        add_filter('wp_editor_settings', [$this, 'forceEasyMDETextareaEditor'], 10, 2);
        add_action('admin_init', [$this, 'ensureDefaultRecipePosts']);
        add_action('admin_enqueue_scripts', [$this, 'enqueuePromptRecipeEditorAssets']);
        add_action('add_meta_boxes', [$this, 'registerMetaBoxes']);
        add_action('save_post_' . self::POST_TYPE, [$this, 'saveMetaBoxes']);
        add_action('manage_' . self::POST_TYPE . '_posts_custom_column', [$this, 'renderRecipeKeyColumn'], 10, 2);
        add_filter('manage_' . self::POST_TYPE . '_posts_columns', [$this, 'addRecipeKeyColumn']);
    }

    public static function seedDefaultRecipePosts(): void
    {
        $module = new self();
        $module->upsertDefaultRecipePosts();
    }

    public function disableBlockEditorForPromptRecipe(bool $useBlockEditor, string $postType): bool
    {
        if ($postType === self::POST_TYPE) {
            return false;
        }

        return $useBlockEditor;
    }

    public function forceEasyMDETextareaEditor(array $settings, string $editorId): array
    {
        if ($editorId !== 'content') {
            return $settings;
        }
        if (! $this->isPromptRecipeEditorScreen()) {
            return $settings;
        }

        $settings['tinymce'] = false;
        $settings['quicktags'] = false;
        $settings['media_buttons'] = false;

        return $settings;
    }

    public function enqueuePromptRecipeEditorAssets(string $hookSuffix): void
    {
        if (! in_array($hookSuffix, ['post.php', 'post-new.php'], true)) {
            return;
        }
        if (! $this->isPromptRecipeEditorScreen()) {
            return;
        }

        wp_enqueue_style(
            'agent-easymde',
            plugin_dir_url(__FILE__) . '../../../assets/vendor/easymde/easymde.min.css',
            [],
            '2.20.0'
        );
        wp_add_inline_style(
            'agent-easymde',
            '.EasyMDEContainer .editor-toolbar a{background:transparent !important;}
            .EasyMDEContainer .editor-toolbar a:hover,
            .EasyMDEContainer .editor-toolbar a.active{background:transparent !important;}
            .EasyMDEContainer .editor-toolbar i.separator{border-left:1px solid #dcdcde;}
            .EasyMDEContainer .editor-toolbar #wpai-md-placeholder-select{height:28px;margin:0 6px 0 4px;padding:0 28px 0 8px;font-size:12px;border:1px solid #c3c4c7;border-radius:3px;background-color:#fff;background-image:url(\"data:image/svg+xml,%3Csvg xmlns=%27http://www.w3.org/2000/svg%27 width=%2710%27 height=%276%27 viewBox=%270 0 10 6%27%3E%3Cpath fill=%27%2350575e%27 d=%27M0 0l5 6 5-6z%27/%3E%3C/svg%3E\");background-repeat:no-repeat;background-position:right 8px center;background-size:10px 6px;-webkit-appearance:none;-moz-appearance:none;appearance:none;vertical-align:middle;}
            #wp-content-editor-tools{display:none !important;}'
        );
        wp_enqueue_script(
            'agent-easymde',
            plugin_dir_url(__FILE__) . '../../../assets/vendor/easymde/easymde.min.js',
            [],
            '2.20.0',
            true
        );

        $categoryLabel = __('Categories');
        $tagLabel = __('Tags');
        if (function_exists('get_taxonomy')) {
            $categoryTax = get_taxonomy('category');
            if (is_object($categoryTax) && isset($categoryTax->labels->name)) {
                $categoryLabel = (string) $categoryTax->labels->name;
            }
            $tagTax = get_taxonomy('post_tag');
            if (is_object($tagTax) && isset($tagTax->labels->name)) {
                $tagLabel = (string) $tagTax->labels->name;
            }
        }

        $placeholderOptions = [
            ['token' => '{{site_name}}', 'label' => __('Site Title')],
            ['token' => '{{site_tagline}}', 'label' => __('Tagline')],
            ['token' => '{{site_language}}', 'label' => __('Site Language')],
            ['token' => '{{site_url}}', 'label' => __('Site Address (URL)')],
            ['token' => '{{categories}}', 'label' => $categoryLabel],
            ['token' => '{{tags}}', 'label' => $tagLabel],
            ['token' => '{{recent_posts_titles}}', 'label' => __('Recent Post Titles', 'default')],
            ['token' => '{{recent_posts_excerpts}}', 'label' => __('Recent Post Excerpts', 'default')],
            ['token' => '{{recent_post_titles}}', 'label' => __('Recent Post Titles (Alias)', 'default')],
            ['token' => '{{recent_post_excerpts}}', 'label' => __('Recent Post Excerpts (Alias)', 'default')],
            ['token' => '{{seo_plugin}}', 'label' => __('SEO Plugin', 'default')],
            ['token' => '{{target_audience_hint}}', 'label' => __('Target Audience Hint', 'default')],
            ['token' => '{{post_count}}', 'label' => __('Post Count', 'default')],
            ['token' => '{{business_model_hint}}', 'label' => __('Business Model Hint', 'default')],
        ];
        $placeholderOptionsJson = wp_json_encode($placeholderOptions);
        $placeholderTitle = esc_js(__('WP Placeholders', 'default'));

        $inlineScript = <<<'JS'
(function () {
    if (window.__agentEasyMDEInitialized) {
        return;
    }

    var init = function () {
        var t = document.getElementById('content');
        if (!t || window.__agentEasyMDEInitialized) {
            return;
        }

        window.wpaiStyleGuideEasyMDE = new EasyMDE({
            element: t,
            autoDownloadFontAwesome: true,
            spellChecker: false,
            status: ['lines', 'words'],
            forceSync: true,
            toolbar: ['bold', 'italic', 'heading', '|', 'quote', 'unordered-list', 'ordered-list', '|', 'link', 'image', 'table', 'horizontal-rule', '|', 'preview', 'guide']
        });

        var container = t.closest('.EasyMDEContainer') || t.parentNode;
        if (container) {
            var toolbar = container.querySelector('.editor-toolbar');
            if (toolbar && !document.getElementById('wpai-md-placeholder-select')) {
                var placeholders = __WPAI_PLACEHOLDERS__;

                var select = document.createElement('select');
                select.id = 'wpai-md-placeholder-select';
                select.setAttribute('aria-label', '__WPAI_PLACEHOLDER_TITLE__');

                var placeholderOption = document.createElement('option');
                placeholderOption.value = '';
                placeholderOption.textContent = '__WPAI_PLACEHOLDER_TITLE__';
                select.appendChild(placeholderOption);

                placeholders.forEach(function (item) {
                    var opt = document.createElement('option');
                    opt.value = item.token;
                    opt.textContent = item.label + ' (' + item.token + ')';
                    select.appendChild(opt);
                });

                select.addEventListener('change', function () {
                    var token = select.value;
                    if (!token || !window.wpaiStyleGuideEasyMDE || !window.wpaiStyleGuideEasyMDE.codemirror) {
                        return;
                    }
                    var cm = window.wpaiStyleGuideEasyMDE.codemirror;
                    cm.replaceSelection(token);
                    cm.focus();
                    select.value = '';
                });

                toolbar.appendChild(select);
            }
        }

        if (container && !document.getElementById('wpai-easymde-attribution')) {
            var note = document.createElement('p');
            note.id = 'wpai-easymde-attribution';
            note.style.margin = '8px 0 0';
            note.style.fontSize = '12px';
            note.style.color = '#646970';
            note.innerHTML = 'EasyMDE by <a href="https://github.com/Ionaru/easy-markdown-editor" target="_blank" rel="noopener noreferrer">Jeroen Akkerman</a>';
            container.parentNode.insertBefore(note, container.nextSibling);
        }

        window.__agentEasyMDEInitialized = true;
    };

    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }
})();
JS;
        $inlineScript = str_replace('__WPAI_PLACEHOLDERS__', (string) $placeholderOptionsJson, $inlineScript);
        $inlineScript = str_replace('__WPAI_PLACEHOLDER_TITLE__', $placeholderTitle, $inlineScript);
        wp_add_inline_script('agent-easymde', $inlineScript, 'after');
    }

    public function ensureDefaultRecipePosts(): void
    {
        if (! current_user_can('manage_options')) {
            return;
        }

        $this->upsertDefaultRecipePosts();
    }

    public function registerMetaBoxes(): void
    {
        add_meta_box(
            'wpai-prompt-recipe-assignment',
            'Rezept-Zuordnung',
            [$this, 'renderRecipeAssignmentMetaBox'],
            self::POST_TYPE,
            'side',
            'high'
        );
    }

    public function renderRecipeAssignmentMetaBox(\WP_Post $post): void
    {
        wp_nonce_field('wpai_prompt_recipe_assignment_save', 'wpai_prompt_recipe_assignment_nonce');
        $current = (string) get_post_meta($post->ID, '_wpai_prompt_recipe_key', true);

        echo '<p><strong>Maschinenlesbarer Rezeptschluessel</strong></p>';
        echo '<p><select name="wpai_prompt_recipe_key" style="width:100%;">';
        echo '<option value="">Keine Zuordnung</option>';
        foreach (self::RECIPE_KEYS as $recipeKey) {
            echo '<option value="' . esc_attr($recipeKey) . '" ' . selected($current, $recipeKey, false) . '>' . esc_html($recipeKey) . '</option>';
        }
        echo '</select></p>';
        echo '<p><em>Jeder Schluessel darf nur einem Post zugeordnet sein.</em></p>';
    }

    public function saveMetaBoxes(int $postId): void
    {
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }

        if (! current_user_can('edit_post', $postId)) {
            return;
        }

        if (! isset($_POST['wpai_prompt_recipe_assignment_nonce']) || ! wp_verify_nonce((string) $_POST['wpai_prompt_recipe_assignment_nonce'], 'wpai_prompt_recipe_assignment_save')) {
            return;
        }

        $requested = isset($_POST['wpai_prompt_recipe_key']) ? sanitize_key((string) $_POST['wpai_prompt_recipe_key']) : '';
        if (! in_array($requested, self::RECIPE_KEYS, true)) {
            $requested = '';
        }

        if ($requested === '') {
            delete_post_meta($postId, '_wpai_prompt_recipe_key');
        } else {
            $conflicts = get_posts(
                [
                    'post_type' => self::POST_TYPE,
                    'post_status' => ['publish', 'draft', 'pending', 'private', 'future'],
                    'posts_per_page' => -1,
                    'fields' => 'ids',
                    'meta_key' => '_wpai_prompt_recipe_key',
                    'meta_value' => $requested,
                    'exclude' => [$postId],
                ]
            );
            foreach ($conflicts as $conflictId) {
                delete_post_meta((int) $conflictId, '_wpai_prompt_recipe_key');
            }
            update_post_meta($postId, '_wpai_prompt_recipe_key', $requested);
        }

        $this->syncRecipeMapping();
    }

    public function addRecipeKeyColumn(array $columns): array
    {
        $out = [];
        foreach ($columns as $key => $label) {
            $out[$key] = $label;
            if ($key === 'title') {
                $out['wpai_prompt_recipe_key'] = 'Rezeptschlüssel';
            }
        }

        return $out;
    }

    public function renderRecipeKeyColumn(string $column, int $postId): void
    {
        if ($column !== 'wpai_prompt_recipe_key') {
            return;
        }

        $recipeKey = (string) get_post_meta($postId, '_wpai_prompt_recipe_key', true);
        echo esc_html($recipeKey !== '' ? $recipeKey : '-');
    }

    private function upsertDefaultRecipePosts(): void
    {
        foreach (self::DEFAULT_RECIPES as $recipe) {
            $recipeKey = (string) ($recipe['key'] ?? '');
            $slug = (string) ($recipe['slug'] ?? '');
            $title = (string) ($recipe['title'] ?? '');
            $content = $this->defaultTemplateForRecipeKey($recipeKey);

            if ($recipeKey === '' || $slug === '' || $title === '') {
                continue;
            }

            $existing = get_page_by_path($slug, OBJECT, self::POST_TYPE);

            if (! ($existing instanceof \WP_Post)) {
                $newId = wp_insert_post(
                    [
                        'post_type' => self::POST_TYPE,
                        'post_status' => 'publish',
                        'post_name' => $slug,
                        'post_title' => $title,
                        'post_content' => $content,
                    ]
                );
                if (is_int($newId) && $newId > 0) {
                    update_post_meta($newId, '_wpai_prompt_recipe_key', $recipeKey);
                }
                continue;
            }

            if ($existing->post_title !== $title || $existing->post_name !== $slug) {
                wp_update_post(
                    [
                        'ID' => $existing->ID,
                        'post_title' => $title,
                        'post_name' => $slug,
                    ]
                );
            }

            if ((string) $existing->post_content === '') {
                wp_update_post(
                    [
                        'ID' => $existing->ID,
                        'post_content' => $content,
                    ]
                );
            }

            update_post_meta($existing->ID, '_wpai_prompt_recipe_key', $recipeKey);
        }

        $this->syncRecipeMapping();
    }

    private function syncRecipeMapping(): void
    {
        $mapping = [];
        $posts = get_posts(
            [
                'post_type' => self::POST_TYPE,
                'post_status' => ['publish', 'draft', 'pending', 'private', 'future'],
                'posts_per_page' => -1,
                'fields' => 'ids',
                'orderby' => 'ID',
                'order' => 'ASC',
            ]
        );

        foreach ($posts as $id) {
            $recipeKey = (string) get_post_meta((int) $id, '_wpai_prompt_recipe_key', true);
            if (in_array($recipeKey, self::RECIPE_KEYS, true)) {
                $mapping[$recipeKey] = (int) $id;
            }
        }

        update_option(self::OPTION_MAPPING, $mapping, false);
    }

    private function isPromptRecipeEditorScreen(): bool
    {
        $postType = '';

        if (function_exists('get_current_screen')) {
            $screen = get_current_screen();
            if (is_object($screen) && isset($screen->post_type)) {
                $postType = (string) $screen->post_type;
            }
        }

        if ($postType === '' && isset($_GET['post_type'])) {
            $postType = sanitize_key((string) $_GET['post_type']);
        }

        if ($postType === '' && isset($_GET['post'])) {
            $post = get_post((int) $_GET['post']);
            if ($post instanceof \WP_Post) {
                $postType = (string) $post->post_type;
            }
        }

        if ($postType === '' && isset($_POST['post_type'])) {
            $postType = sanitize_key((string) $_POST['post_type']);
        }

        return $postType === self::POST_TYPE;
    }

    private function defaultTemplateForRecipeKey(string $recipeKey): string
    {
        if ($recipeKey === 'prompt_recipe_seo_rules') {
            return $this->defaultSeoPromptTemplate();
        }
        if ($recipeKey === 'prompt_recipe_style_exemplars') {
            return $this->defaultStyleExemplarsPromptTemplate();
        }

        return $this->defaultStyleGuidePromptTemplate();
    }

    private function defaultStyleGuidePromptTemplate(): string
    {
        return <<<'PROMPT'
You are an expert editorial strategist and content architect.

Your task is to generate a comprehensive, professional style guide for a WordPress-based website using only the information provided below.

Website Name:
{{site_name}}

Tagline / Description:
{{site_tagline}}

Primary Language:
{{site_language}}

Categories:
{{categories}}

Tags:
{{tags}}

Recent Post Titles:
{{recent_posts_titles}}

Recent Post Excerpts:
{{recent_posts_excerpts}}

Optional Target Audience Hint:
{{target_audience_hint}}

Instructions:

1. Infer the website’s primary focus, expertise level, and positioning.
2. Deduce tone, voice tendencies, and implied brand personality.
3. Identify recurring themes or subject clusters.
4. Determine the likely target audience (be specific).
5. Propose a consistent terminology policy.
6. Define formatting and structural standards.
7. Establish SEO tone guidance (without keyword stuffing).
8. Provide rules for internal linking philosophy.
9. Define clarity standards (sentence length, paragraph size, use of examples).
10. Suggest do’s and don’ts to prevent stylistic drift.

Output Requirements:

- Write a structured, production-ready style guide.
- Use clear section headings.
- Be prescriptive and actionable.
- Do not speculate wildly; base conclusions only on provided data.
- If information is insufficient, define adaptable guidance instead of guessing.
- Do not include meta commentary.
- Do not mention AI or prompt mechanics.

Structure the output as:

1. Editorial Positioning
2. Target Audience Profile
3. Brand Voice & Tone
4. Terminology & Language Rules
5. Structural & Formatting Guidelines
6. SEO & Discoverability Philosophy
7. Internal Linking Strategy
8. Content Depth & Technical Expectations
9. Consistency Rules
10. Style Guardrails (Do’s and Don’ts)
PROMPT;
    }

    private function defaultStyleExemplarsPromptTemplate(): string
    {
        return <<<'PROMPT'
You are an expert editorial strategist and content architect.

Your task is to generate 3-5 short style exemplar paragraphs for a WordPress-based website using only the information provided below.

Website Name:
{{site_name}}

Tagline / Description:
{{site_tagline}}

Primary Language:
{{site_language}}

Categories:
{{categories}}

Tags:
{{tags}}

Recent Post Titles:
{{recent_posts_titles}}

Recent Post Excerpts:
{{recent_posts_excerpts}}

Optional Target Audience Hint:
{{target_audience_hint}}

Instructions:

1. Infer voice and tone from the provided data.
2. Produce compact, realistic exemplar paragraphs in the inferred brand style.
3. Vary structure and intent between examples (intro, explanation, CTA, practical tip).
4. Keep each example concise and production-ready.

Output Requirements:

- Return 3-5 clearly separated paragraphs.
- Use markdown.
- No meta commentary.
- No mention of AI or prompt mechanics.
PROMPT;
    }

    private function defaultSeoPromptTemplate(): string
    {
        return <<<'PROMPT'
You are an expert Technical SEO strategist and content systems architect.

Your task is to generate a comprehensive, best-practice SEO ruleset tailored specifically to the WordPress website described below.

Website Information:
Site Name: {{site_name}}
Tagline / Description: {{site_tagline}}
Primary Language: {{site_language}}
Site URL: {{site_url}}

Categories:
{{categories}}

Tags:
{{tags}}

Recent Post Titles:
{{recent_post_titles}}

Recent Post Excerpts:
{{recent_post_excerpts}}

Existing SEO Plugin (if any):
{{seo_plugin}}

Target Audience Hint (optional):
{{target_audience_hint}}

Content Volume:
{{post_count}}

Primary Business Model (if known):
{{business_model_hint}}

Instructions:

1. Analyze the thematic structure and topic clusters implied by categories and tags.
2. Infer search intent patterns from post titles and excerpts.
3. Deduce content depth level (beginner, intermediate, expert).
4. Identify whether the site appears informational, transactional, commercial, or mixed.
5. Generate a best-practice SEO ruleset tailored to this specific site profile.
6. Base recommendations only on the information provided. If data is insufficient, provide adaptable best-practice guidance instead of speculation.
7. Avoid generic SEO clichés. Provide actionable, system-level rules.

The ruleset must include the following structured sections:

1. SEO Positioning Strategy
   - Core search intent focus
   - Topic authority strategy
   - Pillar/cluster model recommendations

2. On-Page SEO Standards
   - Title tag rules
   - Meta description rules
   - Heading hierarchy standards
   - URL structure guidelines
   - Internal anchor text policy
   - Image optimization standards

3. Content Optimization Framework
   - Keyword integration philosophy
   - Semantic SEO recommendations
   - Content depth expectations
   - E-E-A-T alignment guidelines

4. Internal Linking Architecture
   - Topic clustering rules
   - Link density standards
   - Anchor text variation policy
   - Orphan page prevention rules

5. Technical SEO Baselines (WordPress-Specific)
   - Sitemap standards
   - Indexing control rules
   - Canonicalization strategy
   - Pagination handling
   - Category and tag archive treatment
   - Performance expectations (Core Web Vitals principles)

6. Structured Data Recommendations
   - Schema types to implement
   - Article schema standards
   - Breadcrumb guidelines

7. Content Governance Rules
   - Update cadence policy
   - Thin content detection rules
   - Content consolidation strategy
   - Duplicate topic avoidance rules

8. Scaling Strategy (Based on {{post_count}})
   - Authority building roadmap
   - Expansion into related keyword clusters
   - Long-term topical dominance strategy

Output Requirements:

- Produce a structured, production-ready SEO ruleset.
- Be prescriptive and operational.
- Avoid meta commentary.
- Do not reference AI or prompt mechanics.
- Do not use vague phrases like “it depends.”
- Provide rules that can be implemented directly in a WordPress workflow.
- Keep language precise, professional, and implementation-focused.
PROMPT;
    }
}
