<?php

declare(strict_types=1);

namespace Agent\Modules\Planning;

use Agent\Security\RequestGuard;
use Agent\Support\Logger;
use WP_Error;
use WP_REST_Request;
use WP_REST_Response;

final class PlanningController
{
    private PlanningService $planning;
    private RequestGuard $guard;
    private Logger $logger;

    public function __construct(PlanningService $planning, RequestGuard $guard, Logger $logger)
    {
        $this->planning = $planning;
        $this->guard = $guard;
        $this->logger = $logger;
    }

    public function registerRoutes(): void
    {
        register_rest_route(
            'wpai/v1',
            '/planning/run',
            [
                [
                    'methods' => 'GET',
                    'callback' => [$this, 'getRun'],
                    'permission_callback' => [$this, 'canRead'],
                ],
                [
                    'methods' => 'POST',
                    'callback' => [$this, 'run'],
                    'permission_callback' => static fn (): bool => current_user_can('manage_options'),
                ],
            ]
        );

        register_rest_route(
            'wpai/v1',
            '/planning/input',
            [
                'methods' => 'POST',
                'callback' => [$this, 'saveInput'],
                'permission_callback' => static fn (): bool => current_user_can('manage_options'),
            ]
        );
    }

    public function canRead(): bool
    {
        return $this->guard->checkReadAccess();
    }

    public function getRun(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        if (($rate = $this->guard->assertRateLimit('planning_get_run')) instanceof WP_Error) {
            return $rate;
        }

        return new WP_REST_Response($this->planning->currentRun(), 200);
    }

    public function run(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        if (($rate = $this->guard->assertRateLimit('planning_run')) instanceof WP_Error) {
            return $rate;
        }

        $run = $this->planning->runGates();
        $this->logger->log('planning_run_triggered', 'warning', ['run_id' => (string) ($run['run_id'] ?? '')]);

        return new WP_REST_Response($run, 200);
    }

    public function saveInput(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        if (($rate = $this->guard->assertRateLimit('planning_save_input')) instanceof WP_Error) {
            return $rate;
        }

        $payload = $request->get_json_params();
        if (! is_array($payload)) {
            $payload = [];
        }

        $input = is_array($payload['input'] ?? null) ? (array) $payload['input'] : $payload;

        $saved = $this->planning->saveUserInput($input, (int) get_current_user_id());
        if ($saved instanceof WP_Error) {
            return $saved;
        }

        return new WP_REST_Response($saved, 200);
    }
}
