<?php

declare(strict_types=1);

namespace Agent\Modules\OpenRouter;

use Agent\Support\Options;
use WP_Error;

final class OpenRouterService
{
    private Options $options;

    public function __construct(Options $options)
    {
        $this->options = $options;
    }

    public function config(): array
    {
        $cfg = (array) $this->options->get('openrouter', []);

        return [
            'enabled' => (bool) ($cfg['enabled'] ?? false),
            'base_url' => (string) ($cfg['base_url'] ?? 'https://openrouter.ai/api/v1'),
            'site_url' => (string) ($cfg['site_url'] ?? ''),
            'app_name' => (string) ($cfg['app_name'] ?? 'Agent Bridge'),
            'timeout' => max(5, (int) ($cfg['timeout'] ?? 60)),
            'planner_model' => (string) ($cfg['planner_model'] ?? 'openrouter/free'),
            'writer_model' => (string) ($cfg['writer_model'] ?? 'openrouter/free'),
            'image_model' => (string) ($cfg['image_model'] ?? 'openrouter/free'),
            'model_filter_enabled' => (bool) ($cfg['model_filter_enabled'] ?? true),
            'model_filter' => (string) ($cfg['model_filter'] ?? 'openai|x-ai|google|anthropic|meta-llama|free'),
            'api_key' => (string) ($cfg['api_key'] ?? ''),
        ];
    }

    public function models(): array|WP_Error
    {
        return $this->request('GET', '/models', null);
    }

    public function chatCompletions(array $payload): array|WP_Error
    {
        if (($payload['model'] ?? '') === '') {
            return new WP_Error('agent_openrouter_invalid_payload', 'Field "model" is required.', ['status' => 400]);
        }

        if (! isset($payload['messages']) && ! isset($payload['prompt'])) {
            return new WP_Error('agent_openrouter_invalid_payload', 'Either "messages" or "prompt" is required.', ['status' => 400]);
        }

        return $this->request('POST', '/chat/completions', $payload);
    }

    public function embeddings(array $payload): array|WP_Error
    {
        if (($payload['model'] ?? '') === '') {
            return new WP_Error('agent_openrouter_invalid_payload', 'Field "model" is required.', ['status' => 400]);
        }
        if (! array_key_exists('input', $payload)) {
            return new WP_Error('agent_openrouter_invalid_payload', 'Field "input" is required.', ['status' => 400]);
        }

        return $this->request('POST', '/embeddings', $payload);
    }

    public function hasApiKey(): bool
    {
        return $this->config()['api_key'] !== '';
    }

    public function isEnabled(): bool
    {
        return $this->config()['enabled'] === true;
    }

    private function request(string $method, string $path, ?array $payload): array|WP_Error
    {
        $cfg = $this->config();
        $apiKey = (string) $cfg['api_key'];

        if ($apiKey === '') {
            return new WP_Error('agent_openrouter_missing_key', 'OpenRouter API key is not configured.', ['status' => 400]);
        }

        $baseUrl = rtrim((string) $cfg['base_url'], '/');
        $url = $baseUrl . $path;

        $headers = [
            'Authorization' => 'Bearer ' . $apiKey,
            'Content-Type' => 'application/json',
        ];

        $siteUrl = (string) $cfg['site_url'];
        if ($siteUrl === '' && function_exists('get_site_url')) {
            $siteUrl = (string) get_site_url();
        }
        if ($siteUrl !== '') {
            $headers['HTTP-Referer'] = $siteUrl;
        }

        $appName = (string) $cfg['app_name'];
        if ($appName !== '') {
            $headers['X-Title'] = $appName;
        }

        $args = [
            'method' => $method,
            'headers' => $headers,
            'timeout' => (int) $cfg['timeout'],
        ];

        if ($payload !== null) {
            $args['body'] = wp_json_encode($payload);
        }

        $response = wp_remote_request($url, $args);
        if (is_wp_error($response)) {
            return new WP_Error('agent_openrouter_request_failed', $response->get_error_message(), ['status' => 502]);
        }

        $status = (int) wp_remote_retrieve_response_code($response);
        $rawBody = (string) wp_remote_retrieve_body($response);
        $decoded = json_decode($rawBody, true);
        if (! is_array($decoded)) {
            $decoded = ['raw' => $rawBody];
        }

        if ($status < 200 || $status >= 300) {
            if ($status === 429) {
                $retryAfter = $this->extractRetryAfterSeconds($response);
                $message = $this->extractErrorMessage($decoded);
                if ($message === '') {
                    $message = 'OpenRouter rate limit reached.';
                }
                if ($retryAfter !== null && $retryAfter > 0) {
                    $message .= ' Retry after ~' . (string) $retryAfter . 's.';
                }

                return new WP_Error(
                    'agent_openrouter_rate_limited',
                    $message,
                    [
                        'status' => 429,
                        'retry_after' => $retryAfter,
                        'response' => $decoded,
                    ]
                );
            }

            return new WP_Error(
                'agent_openrouter_http_error',
                $this->buildHttpErrorMessage($status, $decoded),
                [
                    'status' => $status,
                    'response' => $decoded,
                ]
            );
        }

        return $decoded;
    }

    private function buildHttpErrorMessage(int $status, array $decoded): string
    {
        $message = $this->extractErrorMessage($decoded);
        if ($message === '') {
            return 'OpenRouter request failed with HTTP status ' . (string) $status;
        }

        return 'OpenRouter request failed (HTTP ' . (string) $status . '): ' . $message;
    }

    private function extractErrorMessage(array $decoded): string
    {
        $base = '';
        if (is_string($decoded['error']['message'] ?? null)) {
            $base = (string) $decoded['error']['message'];
        } elseif (is_string($decoded['message'] ?? null)) {
            $base = (string) $decoded['message'];
        } elseif (is_string($decoded['raw'] ?? null)) {
            $base = (string) $decoded['raw'];
        }

        $metadata = is_array($decoded['error']['metadata'] ?? null) ? (array) $decoded['error']['metadata'] : [];
        $metadataDetail = $this->extractMetadataDetail($metadata);

        if ($base === '') {
            return $metadataDetail;
        }

        if ($metadataDetail !== '' && $this->isGenericProviderMessage($base)) {
            return $base . ': ' . $metadataDetail;
        }

        return $base;
    }

    private function isGenericProviderMessage(string $message): bool
    {
        $normalized = strtolower(trim($message));
        return in_array($normalized, [
            'provider returned error',
            'service temporarily unavailable',
            'upstream request failed',
        ], true);
    }

    private function extractMetadataDetail(array $metadata): string
    {
        if ($metadata === []) {
            return '';
        }

        $detail = $this->extractScalarString($metadata, ['raw', 'message', 'error', 'provider_error', 'detail', 'provider_message']);
        $provider = $this->extractScalarString($metadata, ['provider_name', 'provider']);
        $code = $this->extractScalarString($metadata, ['code', 'provider_code', 'error_code']);

        $parts = [];
        if ($provider !== '') {
            $parts[] = 'provider=' . $provider;
        }
        if ($code !== '') {
            $parts[] = 'code=' . $code;
        }
        if ($detail !== '') {
            $parts[] = $detail;
        }

        if ($parts !== []) {
            return implode('; ', $parts);
        }

        $encoded = wp_json_encode($metadata);
        return is_string($encoded) ? $encoded : '';
    }

    private function extractScalarString(array $input, array $keys): string
    {
        foreach ($keys as $key) {
            $value = $input[$key] ?? null;
            if (is_scalar($value)) {
                $out = trim((string) $value);
                if ($out !== '') {
                    return $out;
                }
            }
            if (is_array($value)) {
                if (is_scalar($value['message'] ?? null)) {
                    $nested = trim((string) $value['message']);
                    if ($nested !== '') {
                        return $nested;
                    }
                }
                $encoded = wp_json_encode($value);
                if (is_string($encoded) && $encoded !== '[]' && $encoded !== '{}') {
                    return $encoded;
                }
            }
        }

        return '';
    }

    private function extractRetryAfterSeconds(array $response): ?int
    {
        $headerValue = '';

        if (function_exists('wp_remote_retrieve_header')) {
            $headerValue = (string) wp_remote_retrieve_header($response, 'retry-after');
        } elseif (isset($response['headers']['retry-after'])) {
            $headerValue = (string) $response['headers']['retry-after'];
        } elseif (isset($response['headers']['Retry-After'])) {
            $headerValue = (string) $response['headers']['Retry-After'];
        }

        $headerValue = trim($headerValue);
        if ($headerValue === '' || ! ctype_digit($headerValue)) {
            return null;
        }

        return max(0, (int) $headerValue);
    }
}
