<?php

declare(strict_types=1);

namespace Agent\Modules\Jobs;

use Agent\Support\Logger;
use Agent\Support\Options;
use WP_Error;

final class JobWorkerScheduler
{
    public const CRON_HOOK = 'agent_jobs_worker_cron';
    public const SCHEDULE_KEY = 'agent_every_minute';
    public const OPTION_STATUS = 'agent_jobs_worker_status';

    private JobQueueService $jobs;
    private JobExecutionService $executor;
    private Options $options;
    private Logger $logger;

    public function __construct(JobQueueService $jobs, JobExecutionService $executor, Options $options, Logger $logger)
    {
        $this->jobs = $jobs;
        $this->executor = $executor;
        $this->options = $options;
        $this->logger = $logger;
    }

    public function register(): void
    {
        add_filter('cron_schedules', [$this, 'addSchedule']);
        add_action(self::CRON_HOOK, [$this, 'run']);
        $this->ensureScheduled();
    }

    public function addSchedule(array $schedules): array
    {
        if (! isset($schedules[self::SCHEDULE_KEY])) {
            $schedules[self::SCHEDULE_KEY] = [
                'interval' => 60,
                'display' => 'Every Minute (Agent Queue Worker)',
            ];
        }

        return $schedules;
    }

    public function ensureScheduled(): void
    {
        if (! function_exists('wp_next_scheduled') || ! function_exists('wp_schedule_event')) {
            return;
        }
        if (wp_next_scheduled(self::CRON_HOOK) !== false) {
            return;
        }

        wp_schedule_event(time() + 30, self::SCHEDULE_KEY, self::CRON_HOOK);
    }

    public static function unschedule(): void
    {
        if (! function_exists('wp_next_scheduled') || ! function_exists('wp_unschedule_event')) {
            return;
        }

        while (($ts = wp_next_scheduled(self::CRON_HOOK)) !== false) {
            wp_unschedule_event($ts, self::CRON_HOOK);
        }
    }

    public function run(): void
    {
        if ((bool) $this->options->get('emergency_lock', false) === true) {
            $this->writeStatus(['state' => 'skipped', 'reason' => 'emergency_lock']);
            return;
        }
        if ((bool) $this->options->get('full_write_access', false) !== true) {
            $this->writeStatus(['state' => 'skipped', 'reason' => 'full_write_disabled']);
            return;
        }

        $maxPerTick = max(1, min(50, (int) $this->options->get('indexing.worker_max_per_tick', 5)));
        $processed = 0;
        $completed = 0;
        $failed = 0;

        for ($i = 0; $i < $maxPerTick; $i++) {
            $result = $this->jobs->runNext(fn (array $job): array|WP_Error => $this->executor->execute($job));
            if ($result === null) {
                break;
            }

            if ($result instanceof WP_Error) {
                if ((string) $result->code === 'agent_job_worker_locked') {
                    $this->writeStatus(['state' => 'locked', 'reason' => 'worker_locked']);
                    break;
                }
                $this->logger->log('jobs_worker_tick_error', 'warning', ['error_code' => (string) $result->code, 'error' => $result->get_error_message()]);
                $this->writeStatus([
                    'state' => 'error',
                    'reason' => 'run_next_error',
                    'error_code' => (string) $result->code,
                    'error_message' => $result->get_error_message(),
                ]);
                break;
            }

            $processed++;
            $status = (string) ($result['status'] ?? '');
            if ($status === 'completed') {
                $completed++;
            } elseif ($status === 'failed') {
                $failed++;
            }

            if ($status === 'failed') {
                // Keep hard-stop semantics: stop this tick after a failed job.
                break;
            }
        }

        $this->writeStatus([
            'state' => 'ok',
            'reason' => '',
            'processed' => $processed,
            'completed' => $completed,
            'failed' => $failed,
            'max_per_tick' => $maxPerTick,
        ]);

        if ($processed > 0 || $failed > 0) {
            $this->logger->log('jobs_worker_tick', 'info', ['processed' => $processed, 'completed' => $completed, 'failed' => $failed, 'max_per_tick' => $maxPerTick]);
        }
    }

    private function writeStatus(array $data): void
    {
        $base = [
            'last_tick_at' => gmdate('c'),
            'state' => 'ok',
            'reason' => '',
            'processed' => 0,
            'completed' => 0,
            'failed' => 0,
            'max_per_tick' => max(1, (int) $this->options->get('indexing.worker_max_per_tick', 5)),
            'error_code' => '',
            'error_message' => '',
        ];

        update_option(self::OPTION_STATUS, array_replace($base, $data), false);
    }
}
