<?php

declare(strict_types=1);

namespace Agent\Modules\Indexing;

use Agent\Security\RequestGuard;
use Agent\Support\Logger;
use WP_Error;
use WP_REST_Request;
use WP_REST_Response;

final class IndexingController
{
    private ChunkPipelineService $pipeline;
    private RequestGuard $guard;
    private Logger $logger;

    public function __construct(ChunkPipelineService $pipeline, RequestGuard $guard, Logger $logger)
    {
        $this->pipeline = $pipeline;
        $this->guard = $guard;
        $this->logger = $logger;
    }

    public function registerRoutes(): void
    {
        register_rest_route(
            'wpai/v1',
            '/index/pipeline/run',
            [
                'methods' => 'POST',
                'callback' => [$this, 'runPipeline'],
                'permission_callback' => [$this, 'canRead'],
            ]
        );

        register_rest_route(
            'wpai/v1',
            '/index/pipeline/prepare',
            [
                'methods' => 'POST',
                'callback' => [$this, 'prepare'],
                'permission_callback' => [$this, 'canRead'],
            ]
        );

        register_rest_route(
            'wpai/v1',
            '/index/pipeline/chunk',
            [
                'methods' => 'POST',
                'callback' => [$this, 'chunk'],
                'permission_callback' => [$this, 'canRead'],
            ]
        );

        register_rest_route(
            'wpai/v1',
            '/index/pipeline/embed',
            [
                'methods' => 'POST',
                'callback' => [$this, 'embed'],
                'permission_callback' => [$this, 'canRead'],
            ]
        );

        register_rest_route(
            'wpai/v1',
            '/index/pipeline/finalize',
            [
                'methods' => 'POST',
                'callback' => [$this, 'finalize'],
                'permission_callback' => [$this, 'canRead'],
            ]
        );

        register_rest_route(
            'wpai/v1',
            '/index/pipeline/(?P<post_id>[\d]+)',
            [
                'methods' => 'GET',
                'callback' => [$this, 'status'],
                'permission_callback' => [$this, 'canRead'],
            ]
        );
    }

    public function canRead(): bool
    {
        return $this->guard->checkReadAccess();
    }

    public function runPipeline(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        if (($rate = $this->guard->assertRateLimit('index_pipeline_run')) instanceof WP_Error) {
            return $rate;
        }
        if (! $this->guard->checkWriteAccess()) {
            return new WP_Error('agent_write_disabled', 'Full write access is disabled.', ['status' => 403]);
        }

        $postId = $this->readPostIdFromBody($request);
        if ($postId <= 0) {
            return new WP_Error('agent_invalid_post_id', 'Field "post_id" is required.', ['status' => 400]);
        }

        $payload = $request->get_json_params();
        if (! is_array($payload)) {
            $payload = [];
        }

        $options = [
            'chunk_size' => (int) ($payload['chunk_size'] ?? 1200),
            'chunk_overlap' => (int) ($payload['chunk_overlap'] ?? 120),
            'embedding_model' => isset($payload['embedding_model']) ? sanitize_text_field((string) $payload['embedding_model']) : null,
            'allow_local_fallback' => array_key_exists('allow_local_fallback', $payload) ? (bool) $payload['allow_local_fallback'] : true,
        ];

        $result = $this->pipeline->run($postId, $options);
        if ($result instanceof WP_Error) {
            return $result;
        }

        $this->logger->log('index_pipeline_run_requested', 'warning', ['post_id' => $postId]);
        return new WP_REST_Response($result, 200);
    }

    public function prepare(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        if (($rate = $this->guard->assertRateLimit('index_pipeline_prepare')) instanceof WP_Error) {
            return $rate;
        }
        if (! $this->guard->checkWriteAccess()) {
            return new WP_Error('agent_write_disabled', 'Full write access is disabled.', ['status' => 403]);
        }

        $postId = $this->readPostIdFromBody($request);
        if ($postId <= 0) {
            return new WP_Error('agent_invalid_post_id', 'Field "post_id" is required.', ['status' => 400]);
        }

        $result = $this->pipeline->prepare($postId);
        if ($result instanceof WP_Error) {
            return $result;
        }

        return new WP_REST_Response($result, 200);
    }

    public function chunk(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        if (($rate = $this->guard->assertRateLimit('index_pipeline_chunk')) instanceof WP_Error) {
            return $rate;
        }
        if (! $this->guard->checkWriteAccess()) {
            return new WP_Error('agent_write_disabled', 'Full write access is disabled.', ['status' => 403]);
        }

        $payload = $request->get_json_params();
        if (! is_array($payload)) {
            $payload = [];
        }
        $postId = (int) ($payload['post_id'] ?? 0);
        if ($postId <= 0) {
            return new WP_Error('agent_invalid_post_id', 'Field "post_id" is required.', ['status' => 400]);
        }

        $chunkSize = (int) ($payload['chunk_size'] ?? 1200);
        $chunkOverlap = (int) ($payload['chunk_overlap'] ?? 120);
        $result = $this->pipeline->chunk($postId, $chunkSize, $chunkOverlap);
        if ($result instanceof WP_Error) {
            return $result;
        }

        return new WP_REST_Response($result, 200);
    }

    public function embed(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        if (($rate = $this->guard->assertRateLimit('index_pipeline_embed')) instanceof WP_Error) {
            return $rate;
        }
        if (! $this->guard->checkWriteAccess()) {
            return new WP_Error('agent_write_disabled', 'Full write access is disabled.', ['status' => 403]);
        }

        $payload = $request->get_json_params();
        if (! is_array($payload)) {
            $payload = [];
        }
        $postId = (int) ($payload['post_id'] ?? 0);
        if ($postId <= 0) {
            return new WP_Error('agent_invalid_post_id', 'Field "post_id" is required.', ['status' => 400]);
        }

        $model = isset($payload['embedding_model']) ? sanitize_text_field((string) $payload['embedding_model']) : null;
        $allowLocalFallback = array_key_exists('allow_local_fallback', $payload) ? (bool) $payload['allow_local_fallback'] : true;
        $result = $this->pipeline->embed($postId, $model, $allowLocalFallback);
        if ($result instanceof WP_Error) {
            return $result;
        }

        return new WP_REST_Response($result, 200);
    }

    public function finalize(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        if (($rate = $this->guard->assertRateLimit('index_pipeline_finalize')) instanceof WP_Error) {
            return $rate;
        }
        if (! $this->guard->checkWriteAccess()) {
            return new WP_Error('agent_write_disabled', 'Full write access is disabled.', ['status' => 403]);
        }

        $postId = $this->readPostIdFromBody($request);
        if ($postId <= 0) {
            return new WP_Error('agent_invalid_post_id', 'Field "post_id" is required.', ['status' => 400]);
        }

        $result = $this->pipeline->finalize($postId);
        if ($result instanceof WP_Error) {
            return $result;
        }

        return new WP_REST_Response($result, 200);
    }

    public function status(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        if (($rate = $this->guard->assertRateLimit('index_pipeline_status')) instanceof WP_Error) {
            return $rate;
        }

        $postId = (int) $request->get_param('post_id');
        if ($postId <= 0) {
            return new WP_Error('agent_invalid_post_id', 'Invalid post id.', ['status' => 400]);
        }

        return new WP_REST_Response($this->pipeline->status($postId), 200);
    }

    private function readPostIdFromBody(WP_REST_Request $request): int
    {
        $payload = $request->get_json_params();
        if (! is_array($payload)) {
            return 0;
        }
        return (int) ($payload['post_id'] ?? 0);
    }
}

