<?php

declare(strict_types=1);

namespace Agent\Modules\Indexing;

use Agent\Modules\Jobs\JobQueueService;
use Agent\Support\Logger;
use Agent\Support\Options;

final class IndexBackfillScheduler
{
    public const CRON_HOOK = 'agent_index_backfill_cron';
    public const SCHEDULE_KEY = 'agent_every_five_minutes';

    private JobQueueService $jobs;
    private Options $options;
    private Logger $logger;

    public function __construct(JobQueueService $jobs, Options $options, Logger $logger)
    {
        $this->jobs = $jobs;
        $this->options = $options;
        $this->logger = $logger;
    }

    public function register(): void
    {
        add_filter('cron_schedules', [$this, 'addSchedule']);
        add_action(self::CRON_HOOK, [$this, 'run']);
        $this->ensureScheduled();
    }

    public function addSchedule(array $schedules): array
    {
        if (! isset($schedules[self::SCHEDULE_KEY])) {
            $schedules[self::SCHEDULE_KEY] = [
                'interval' => 300,
                'display' => 'Every 5 Minutes (Agent Index Backfill)',
            ];
        }
        return $schedules;
    }

    public function ensureScheduled(): void
    {
        if (! function_exists('wp_next_scheduled') || ! function_exists('wp_schedule_event')) {
            return;
        }

        if (wp_next_scheduled(self::CRON_HOOK) !== false) {
            return;
        }

        wp_schedule_event(time() + 60, self::SCHEDULE_KEY, self::CRON_HOOK);
    }

    public static function unschedule(): void
    {
        if (! function_exists('wp_next_scheduled') || ! function_exists('wp_unschedule_event')) {
            return;
        }

        while (($ts = wp_next_scheduled(self::CRON_HOOK)) !== false) {
            wp_unschedule_event($ts, self::CRON_HOOK);
        }
    }

    public function run(): void
    {
        if ((bool) $this->options->get('emergency_lock', false) === true) {
            return;
        }
        if ((bool) $this->options->get('full_write_access', false) !== true) {
            return;
        }

        $limit = max(1, min(50, (int) $this->options->get('indexing.backfill_limit', 10)));
        $scanLimit = max($limit * 3, 30);

        $candidates = $this->findBackfillCandidates($scanLimit);
        if ($candidates === []) {
            return;
        }

        $enqueued = 0;
        foreach ($candidates as $candidate) {
            if ($enqueued >= $limit) {
                break;
            }

            $postId = (int) ($candidate['post_id'] ?? 0);
            $contentHash = (string) ($candidate['content_hash'] ?? '');
            if ($postId <= 0 || $contentHash === '') {
                continue;
            }

            if ($this->jobs->hasPendingJobForPost('reindex_post', $postId)) {
                continue;
            }

            $queueKey = 'reindex_post_' . (string) $postId;
            $job = $this->jobs->enqueue(
                'reindex_post',
                [
                    'post_id' => $postId,
                    'trigger' => 'cron_backfill',
                    'content_hash' => $contentHash,
                ],
                0,
                null,
                $queueKey
            );
            if ($job instanceof \WP_Error) {
                $this->logger->log(
                    'index_backfill_enqueue_failed',
                    'warning',
                    ['post_id' => $postId, 'error_code' => (string) $job->code, 'error' => $job->get_error_message()]
                );
                continue;
            }

            $enqueued++;
        }

        if ($enqueued > 0) {
            $this->logger->log('index_backfill_enqueued', 'info', ['count' => $enqueued, 'scan_limit' => $scanLimit, 'limit' => $limit]);
        }
    }

    private function findBackfillCandidates(int $scanLimit): array
    {
        if (! $this->isDbReady()) {
            return [];
        }

        global $wpdb;
        $postsTable = $wpdb->posts;
        $statusTable = $wpdb->prefix . 'ai_index_status';

        $rows = $wpdb->get_results(
            $wpdb->prepare(
                'SELECT p.ID AS post_id, p.post_title, p.post_content, s.content_hash, s.status
                 FROM ' . $postsTable . ' p
                 LEFT JOIN ' . $statusTable . ' s ON s.post_id = p.ID
                 WHERE p.post_type IN (%s, %s)
                   AND p.post_status IN (%s, %s, %s, %s)
                 ORDER BY p.post_modified_gmt DESC, p.ID DESC
                 LIMIT %d',
                'post',
                'page',
                'publish',
                'draft',
                'pending',
                'private',
                $scanLimit
            ),
            ARRAY_A
        );
        if (! is_array($rows)) {
            return [];
        }

        $candidates = [];
        foreach ($rows as $row) {
            if (! is_array($row)) {
                continue;
            }
            $postId = (int) ($row['post_id'] ?? 0);
            if ($postId <= 0) {
                continue;
            }

            $title = (string) ($row['post_title'] ?? '');
            $content = (string) ($row['post_content'] ?? '');
            $currentHash = hash('sha256', $title . "\n" . $content);
            $indexedHash = (string) ($row['content_hash'] ?? '');
            $indexedStatus = (string) ($row['status'] ?? '');

            if ($indexedHash === $currentHash && $indexedStatus === 'indexed') {
                continue;
            }

            $candidates[] = [
                'post_id' => $postId,
                'content_hash' => $currentHash,
            ];
        }

        return $candidates;
    }

    private function isDbReady(): bool
    {
        global $wpdb;
        if (! isset($wpdb) || ! is_object($wpdb)) {
            return false;
        }
        if (! isset($wpdb->prefix) || ! is_string($wpdb->prefix) || $wpdb->prefix === '') {
            return false;
        }
        if (! isset($wpdb->posts) || ! is_string($wpdb->posts) || $wpdb->posts === '') {
            return false;
        }

        return method_exists($wpdb, 'get_results') && method_exists($wpdb, 'prepare');
    }
}

