<?php

declare(strict_types=1);

namespace Agent\Modules\Indexing;

use Agent\Modules\Jobs\JobQueueService;
use Agent\Support\Logger;
use Agent\Support\Options;

final class ContentChangeIndexer
{
    private JobQueueService $jobs;
    private Options $options;
    private Logger $logger;

    public function __construct(JobQueueService $jobs, Options $options, Logger $logger)
    {
        $this->jobs = $jobs;
        $this->options = $options;
        $this->logger = $logger;
    }

    public function onSavePost(int $postId, \WP_Post $post, bool $update): void
    {
        if ($postId <= 0) {
            return;
        }
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
        if (function_exists('wp_is_post_revision') && wp_is_post_revision($postId)) {
            return;
        }
        if (function_exists('wp_is_post_autosave') && wp_is_post_autosave($postId)) {
            return;
        }
        if (! in_array((string) $post->post_type, ['post', 'page'], true)) {
            return;
        }
        if ((string) $post->post_status === 'auto-draft') {
            return;
        }

        if ((bool) $this->options->get('emergency_lock', false) === true) {
            return;
        }
        if ((bool) $this->options->get('full_write_access', false) !== true) {
            return;
        }

        if ($this->jobs->hasPendingJobForPost('reindex_post', $postId)) {
            return;
        }

        $queueKey = 'reindex_post_' . (string) $postId;
        $job = $this->jobs->enqueue(
            'reindex_post',
            [
                'post_id' => $postId,
                'trigger' => 'save_post',
                'post_type' => (string) $post->post_type,
                'post_status' => (string) $post->post_status,
                'is_update' => $update,
            ],
            (int) get_current_user_id(),
            null,
            $queueKey
        );

        if ($job instanceof \WP_Error) {
            $this->logger->log(
                'index_auto_enqueue_failed',
                'warning',
                ['post_id' => $postId, 'error_code' => (string) $job->code, 'error' => $job->get_error_message()]
            );
            return;
        }

        $this->logger->log(
            'index_auto_enqueued',
            'info',
            ['post_id' => $postId, 'job_id' => (int) ($job['id'] ?? 0), 'queue_key' => $queueKey]
        );
    }
}
