<?php

declare(strict_types=1);

namespace Agent\Admin;

use Agent\Modules\CoreAnalyzer\CoreAnalyzerService;
use Agent\Modules\Jobs\JobWorkerScheduler;
use Agent\Modules\OpenRouter\OpenRouterService;
use Agent\Modules\Planning\PlanningService;
use Agent\Modules\Planning\WpZeoInstaller;
use Agent\Security\RequestGuard;
use Agent\Support\Logger;
use Agent\Support\Options;

final class AdminPage
{
    private CoreAnalyzerService $analyzer;
    private RequestGuard $guard;
    private Logger $logger;
    private Options $options;
    private OpenRouterService $openRouter;
    private PlanningService $planning;
    private WpZeoInstaller $wpZeoInstaller;

    public function __construct(CoreAnalyzerService $analyzer, RequestGuard $guard, Logger $logger, Options $options, OpenRouterService $openRouter, PlanningService $planning, WpZeoInstaller $wpZeoInstaller)
    {
        $this->analyzer = $analyzer;
        $this->guard = $guard;
        $this->logger = $logger;
        $this->options = $options;
        $this->openRouter = $openRouter;
        $this->planning = $planning;
        $this->wpZeoInstaller = $wpZeoInstaller;
    }

    public function registerPage(): void
    {
        add_menu_page(
            'Agent',
            'Agent',
            'manage_options',
            'agent',
            [$this, 'render'],
            'dashicons-superhero',
            58
        );

        add_submenu_page(
            'agent',
            'Agent Control',
            'Control',
            'manage_options',
            'agent',
            [$this, 'render']
        );

        add_submenu_page(
            'agent',
            'OpenRouter',
            'OpenRouter',
            'manage_options',
            'agent-openrouter',
            [$this, 'renderOpenRouter']
        );

        add_submenu_page(
            'agent',
            'Indexing',
            'Indexing',
            'manage_options',
            'agent-indexing',
            [$this, 'renderIndexing']
        );

        add_submenu_page(
            'agent',
            'Planning',
            'Planning',
            'manage_options',
            'agent-planning',
            [$this, 'renderPlanning']
        );
    }

    public function registerSettings(): void
    {
        register_setting(
            'agent_settings_group',
            'agent_settings',
            [
                'type' => 'array',
                'show_in_rest' => false,
                'sanitize_callback' => [$this, 'sanitizeSettings'],
                'default' => [],
            ]
        );
    }

    public function renderPlanningNotice(): void
    {
        if (! current_user_can('manage_options')) {
            return;
        }
        if (isset($_GET['page']) && sanitize_key((string) $_GET['page']) === 'agent-planning') {
            return;
        }

        $run = $this->planning->currentRun();
        $status = (string) ($run['status'] ?? 'idle');
        $state = (string) ($run['current_state'] ?? 'IDLE');
        $blockers = (int) ($run['summary']['blocker_count'] ?? 0);
        $warnings = (int) ($run['summary']['warning_count'] ?? 0);
        $canProceed = (bool) ($run['summary']['can_proceed_to_planning'] ?? false);

        if ($status === 'ready_for_planning' && $canProceed) {
            return;
        }

        $class = 'notice-warning';
        if ($status === 'waiting_for_fix' || $blockers > 0) {
            $class = 'notice-error';
        } elseif ($status === 'idle') {
            $class = 'notice-info';
        }

        $url = admin_url('admin.php?page=agent-planning');
        ?>
        <div class="notice <?php echo esc_attr($class); ?>">
            <p>
                <strong>wpAI befindet sich im Planungsmodus.</strong>
                Bitte richte wpAI ein.
                <a href="<?php echo esc_url($url); ?>">Zum wpAI-Planungsmodus wechseln.</a>
            </p>
        </div>
        <?php
    }

    public function sanitizeSettings(mixed $input): array
    {
        $current = $this->options->all();

        if (! is_array($input)) {
            return $current;
        }

        $current['readonly_mode'] = isset($input['readonly_mode']);
        $current['full_write_access'] = isset($input['full_write_access']);
        $current['emergency_lock'] = isset($input['emergency_lock']);
        $current['rate_limit_per_minute'] = max(1, (int) ($input['rate_limit_per_minute'] ?? 60));

        $apiKey = (string) ($input['api_key'] ?? '');
        $current['api_key'] = sanitize_text_field($apiKey);

        $openRouterInput = is_array($input['openrouter'] ?? null) ? $input['openrouter'] : [];
        $currentOpenRouter = is_array($current['openrouter'] ?? null) ? $current['openrouter'] : [];
        $currentOpenRouter['enabled'] = isset($openRouterInput['enabled']);
        if (array_key_exists('api_key', $openRouterInput)) {
            $currentOpenRouter['api_key'] = sanitize_text_field((string) $openRouterInput['api_key']);
        }
        $currentOpenRouter['base_url'] = isset($openRouterInput['base_url']) ? esc_url_raw((string) $openRouterInput['base_url']) : (string) ($currentOpenRouter['base_url'] ?? 'https://openrouter.ai/api/v1');
        $currentOpenRouter['site_url'] = isset($openRouterInput['site_url']) ? esc_url_raw((string) $openRouterInput['site_url']) : (string) ($currentOpenRouter['site_url'] ?? '');
        $currentOpenRouter['app_name'] = isset($openRouterInput['app_name']) ? sanitize_text_field((string) $openRouterInput['app_name']) : (string) ($currentOpenRouter['app_name'] ?? 'Agent Bridge');
        $currentOpenRouter['timeout'] = isset($openRouterInput['timeout']) ? max(5, min(180, (int) $openRouterInput['timeout'])) : max(5, (int) ($currentOpenRouter['timeout'] ?? 60));
        $currentOpenRouter['planner_model'] = isset($openRouterInput['planner_model']) ? sanitize_text_field((string) $openRouterInput['planner_model']) : (string) ($currentOpenRouter['planner_model'] ?? 'openrouter/free');
        $currentOpenRouter['writer_model'] = isset($openRouterInput['writer_model']) ? sanitize_text_field((string) $openRouterInput['writer_model']) : (string) ($currentOpenRouter['writer_model'] ?? 'openrouter/free');
        $currentOpenRouter['image_model'] = isset($openRouterInput['image_model']) ? sanitize_text_field((string) $openRouterInput['image_model']) : (string) ($currentOpenRouter['image_model'] ?? 'openrouter/free');
        $currentOpenRouter['model_filter_enabled'] = isset($openRouterInput['model_filter_enabled']);
        $currentOpenRouter['model_filter'] = isset($openRouterInput['model_filter']) ? sanitize_text_field((string) $openRouterInput['model_filter']) : (string) ($currentOpenRouter['model_filter'] ?? 'openai|x-ai|google|anthropic|meta-llama|free');
        $current['openrouter'] = $currentOpenRouter;

        $moduleInput = is_array($input['modules'] ?? null) ? $input['modules'] : [];
        foreach ((array) ($current['modules'] ?? []) as $module => $enabled) {
            $current['modules'][$module] = isset($moduleInput[$module]);
        }

        $this->logger->log('settings_updated', 'warning');

        return $current;
    }

    public function render(): void
    {
        if (! current_user_can('manage_options')) {
            return;
        }

        $settings = $this->options->all();
        $report = $this->analyzer->generateReport();
        $logs = $this->logger->latest(20);
        $seoStatus = $this->formatSeoStatus($report);
        $setupStatus = get_option(\Agent\Plugin::OPTION_SETUP_STATUS, []);
        if (! is_array($setupStatus)) {
            $setupStatus = [];
        }
        $apiRows = $this->buildApiFunctionalityRows($settings, $report);
        $pipelineRuns = $this->latestPipelineRuns(15);
        $workerStatus = $this->jobsWorkerStatus();

        ?>
        <div class="wrap">
            <h1>Agent Control</h1>
            <p>Zentrale Übersicht für Systemzustand, API-Status und Notfallsteuerung.</p>

            <form method="post" action="options.php">
                <?php settings_fields('agent_settings_group'); ?>
                <h2>System Status</h2>
                <table class="widefat striped">
                    <tbody>
                        <tr><td>WordPress</td><td><?php echo esc_html((string) $report['system_state']['wordpress_version']); ?></td></tr>
                        <tr><td>PHP</td><td><?php echo esc_html((string) $report['system_state']['php_version']); ?></td></tr>
                        <tr><td>REST API</td><td><?php echo esc_html($report['system_state']['rest_api']['enabled'] ? 'Aktiv' : 'Inaktiv'); ?></td></tr>
                        <tr><td>Permalinks</td><td><?php echo esc_html($report['system_state']['permalinks']['pretty_permalinks'] ? 'Sprechend' : 'Einfach'); ?></td></tr>
                        <tr><td>Object Cache</td><td><?php echo esc_html($report['system_state']['object_cache']['enabled'] ? 'Aktiv' : 'Inaktiv'); ?></td></tr>
                        <tr><td>SEO Plugin</td><td><?php echo esc_html($seoStatus); ?></td></tr>
                    </tbody>
                </table>

                <h2>Einrichtungsstatus</h2>
                <?php if ((array) ($setupStatus['steps'] ?? []) === []) : ?>
                    <p><em>Noch kein automatischer Einrichtungsstatus vorhanden.</em></p>
                <?php else : ?>
                    <table class="widefat striped">
                        <thead>
                            <tr>
                                <th>Schritt</th>
                                <th>Status</th>
                                <th>Details</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ((array) $setupStatus['steps'] as $step) : ?>
                                <tr>
                                    <td><?php echo esc_html((string) ($step['label'] ?? $step['key'] ?? '')); ?></td>
                                    <td><?php echo esc_html((string) ($step['status'] ?? 'unknown')); ?></td>
                                    <td><?php echo esc_html((string) ($step['message'] ?? '')); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                    <?php if (isset($setupStatus['last_run'])) : ?>
                        <p><em>Letzter Auto-Setup Lauf: <?php echo esc_html((string) $setupStatus['last_run']); ?> (UTC)</em></p>
                    <?php endif; ?>
                <?php endif; ?>

                <h2>API-Funktionalitaet</h2>
                <table class="widefat striped">
                    <thead>
                        <tr>
                            <th>Funktion</th>
                            <th>Status</th>
                            <th>Details</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($apiRows as $row) : ?>
                            <tr>
                                <td><?php echo esc_html((string) ($row['label'] ?? '')); ?></td>
                                <td><?php echo esc_html((string) ($row['status'] ?? 'unknown')); ?></td>
                                <td><?php echo esc_html((string) ($row['details'] ?? '')); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>

                <h2>Pipeline-Pruefung (Index/Chunk)</h2>
                <?php if ($pipelineRuns === []) : ?>
                    <p><em>Noch keine Pipeline-Laeufe vorhanden.</em></p>
                <?php else : ?>
                    <table class="widefat striped">
                        <thead>
                            <tr>
                                <th>Run</th>
                                <th>Post ID</th>
                                <th>Status</th>
                                <th>Failed Step</th>
                                <th>Start (UTC)</th>
                                <th>Ende (UTC)</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($pipelineRuns as $run) : ?>
                                <tr>
                                    <td><code><?php echo esc_html((string) ($run['run_id'] ?? '')); ?></code></td>
                                    <td><?php echo esc_html((string) ((int) ($run['post_id'] ?? 0))); ?></td>
                                    <td><?php echo esc_html((string) ($run['status'] ?? 'unknown')); ?></td>
                                    <td><?php echo esc_html((string) ($run['failed_step'] ?? '')); ?></td>
                                    <td><?php echo esc_html((string) ($run['started_at'] ?? '')); ?></td>
                                    <td><?php echo esc_html((string) ($run['finished_at'] ?? '')); ?></td>
                                </tr>
                                <?php if (is_array($run['steps'] ?? null)) : ?>
                                    <tr>
                                        <td colspan="6">
                                            <strong>Schritte:</strong>
                                            <?php
                                            $labels = [];
                                            foreach ((array) $run['steps'] as $stepName => $stepResult) {
                                                if (! is_array($stepResult)) {
                                                    continue;
                                                }
                                                $stepStatus = (string) ($stepResult['status'] ?? '');
                                                if ($stepStatus === '') {
                                                    $stepStatus = $stepResult === [] ? 'ok' : 'done';
                                                }
                                                $labels[] = (string) $stepName . ':' . $stepStatus;
                                            }
                                            echo esc_html(implode(' | ', $labels));
                                            ?>
                                        </td>
                                    </tr>
                                <?php endif; ?>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>

                <h2>Queue Worker Status</h2>
                <table class="widefat striped">
                    <tbody>
                        <tr><td>Letzter Tick (UTC)</td><td><?php echo esc_html((string) ($workerStatus['last_tick_at'] ?? '')); ?></td></tr>
                        <tr><td>Status</td><td><?php echo esc_html((string) ($workerStatus['state'] ?? 'unknown')); ?></td></tr>
                        <tr><td>Grund</td><td><?php echo esc_html((string) ($workerStatus['reason'] ?? '')); ?></td></tr>
                        <tr><td>Verarbeitet</td><td><?php echo esc_html((string) ((int) ($workerStatus['processed'] ?? 0))); ?></td></tr>
                        <tr><td>Erfolgreich</td><td><?php echo esc_html((string) ((int) ($workerStatus['completed'] ?? 0))); ?></td></tr>
                        <tr><td>Fehlgeschlagen</td><td><?php echo esc_html((string) ((int) ($workerStatus['failed'] ?? 0))); ?></td></tr>
                        <tr><td>Max. pro Tick</td><td><?php echo esc_html((string) ((int) ($workerStatus['max_per_tick'] ?? 0))); ?></td></tr>
                        <tr><td>Fehler</td><td><?php echo esc_html((string) ($workerStatus['error_code'] ?? '') . ' ' . (string) ($workerStatus['error_message'] ?? '')); ?></td></tr>
                    </tbody>
                </table>

                <h2>Sicherheit & Laufzeitsteuerung</h2>
                <p>
                    <label>
                        <input type="checkbox" name="agent_settings[readonly_mode]" <?php checked((bool) ($settings['readonly_mode'] ?? true)); ?> />
                        Read-only-Modus
                    </label>
                </p>
                <p>
                    <label>
                        <input type="checkbox" name="agent_settings[full_write_access]" <?php checked((bool) ($settings['full_write_access'] ?? false)); ?> />
                        Voller Schreibzugriff (API-Schreibzugriffe erlaubt)
                    </label>
                </p>
                <p>
                    <label>
                        <input type="checkbox" name="agent_settings[emergency_lock]" <?php checked((bool) ($settings['emergency_lock'] ?? false)); ?> />
                        Notfall-Sperre für API
                    </label>
                </p>
                <p>
                    <label for="agent-rate-limit">Rate Limit pro Minute</label><br />
                    <input id="agent-rate-limit" type="number" min="1" name="agent_settings[rate_limit_per_minute]" value="<?php echo esc_attr((string) ($settings['rate_limit_per_minute'] ?? 60)); ?>" />
                </p>
                <p>
                    <label for="agent-api-key">API-Key (optional für Initialzugriff)</label><br />
                    <input id="agent-api-key" type="text" class="regular-text" name="agent_settings[api_key]" value="<?php echo esc_attr((string) ($settings['api_key'] ?? '')); ?>" autocomplete="off" />
                </p>

                <h2>Module</h2>
                <?php foreach ((array) ($settings['modules'] ?? []) as $module => $enabled) : ?>
                    <p>
                        <label>
                            <input type="checkbox" name="agent_settings[modules][<?php echo esc_attr((string) $module); ?>]" <?php checked((bool) $enabled); ?> />
                            <?php echo esc_html((string) $module); ?>
                        </label>
                    </p>
                <?php endforeach; ?>

                <?php submit_button('Steuerung speichern'); ?>
            </form>

            <h2>Letzte Agenten-Aktivitaeten</h2>
            <table class="widefat striped">
                <thead>
                    <tr>
                        <th>Zeit (UTC)</th>
                        <th>Aktion</th>
                        <th>Severity</th>
                        <th>Benutzer</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if ($logs === []) : ?>
                        <tr><td colspan="4">Noch keine Aktivitaet protokolliert.</td></tr>
                    <?php else : ?>
                        <?php foreach ($logs as $entry) : ?>
                            <tr>
                                <td><?php echo esc_html((string) ($entry['time'] ?? '')); ?></td>
                                <td><?php echo esc_html((string) ($entry['action'] ?? '')); ?></td>
                                <td><?php echo esc_html((string) ($entry['severity'] ?? '')); ?></td>
                                <td><?php echo esc_html((string) ($entry['user_id'] ?? '0')); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
        <?php
    }

    private function buildApiFunctionalityRows(array $settings, array $report): array
    {
        $restEnabled = (bool) ($report['system_state']['rest_api']['enabled'] ?? false);
        $hasNamespace = $this->hasWpAiNamespace();
        $emergencyLock = (bool) ($settings['emergency_lock'] ?? false);
        $fullWrite = (bool) ($settings['full_write_access'] ?? false);
        $readonly = (bool) ($settings['readonly_mode'] ?? true);
        $openRouterCfg = is_array($settings['openrouter'] ?? null) ? $settings['openrouter'] : [];
        $openRouterEnabled = (bool) ($openRouterCfg['enabled'] ?? false);
        $openRouterHasKey = ((string) ($openRouterCfg['api_key'] ?? '')) !== '';
        $coreAnalyzerEnabled = (bool) ($settings['modules']['core_analyzer'] ?? true);

        $readSessionDetails = $emergencyLock
            ? 'Sitzungsbasierter Lesezugriff ist wegen Notfall-Sperre blockiert.'
            : 'Sitzungsbasierter Lesezugriff ist erlaubt (manage_options oder agent_read).';

        $openRouterModelDetails = ! $openRouterEnabled
            ? 'OpenRouter ist deaktiviert.'
            : (! $openRouterHasKey ? 'OpenRouter API-Key fehlt.' : 'Modellliste kann abgefragt werden.');

        $openRouterChatReady = $openRouterEnabled && $openRouterHasKey && $fullWrite && ! $emergencyLock;
        $openRouterChatDetails = ! $openRouterEnabled
            ? 'OpenRouter ist deaktiviert.'
            : (! $openRouterHasKey
                ? 'OpenRouter API-Key fehlt.'
                : (! $fullWrite
                    ? 'Voller Schreibzugriff ist deaktiviert.'
                    : ($emergencyLock ? 'Notfall-Sperre blockiert Schreibanfragen.' : 'Chat-Endpunkt ist schreibbereit.')));
        $workerStatus = $this->jobsWorkerStatus();
        $workerTick = (string) ($workerStatus['last_tick_at'] ?? '');
        $workerState = (string) ($workerStatus['state'] ?? 'unknown');
        $planningRun = $this->planning->currentRun();
        $planningState = (string) ($planningRun['status'] ?? 'idle');
        $planningDetails = 'State: ' . (string) ($planningRun['current_state'] ?? 'IDLE')
            . ', Blocker=' . (string) ((int) ($planningRun['summary']['blocker_count'] ?? 0))
            . ', Hinweise=' . (string) ((int) ($planningRun['summary']['warning_count'] ?? 0));
        $workerDetails = $workerTick === ''
            ? 'Bisher kein Worker-Lauf protokolliert.'
            : ('Status: ' . $workerState
                . ', verarbeitet=' . (string) ((int) ($workerStatus['processed'] ?? 0))
                . ', erfolgreich=' . (string) ((int) ($workerStatus['completed'] ?? 0))
                . ', fehlgeschlagen=' . (string) ((int) ($workerStatus['failed'] ?? 0))
                . ', letzter_lauf=' . $workerTick);

        return [
            [
                'label' => 'REST API',
                'status' => $restEnabled ? 'aktiv' : 'inaktiv',
                'details' => $restEnabled ? 'WordPress REST API ist aktiv.' : 'WordPress REST API ist deaktiviert.',
            ],
            [
                'label' => 'Namespace /wp-json/wpai/v1',
                'status' => $hasNamespace ? 'registriert' : 'fehlt',
                'details' => $hasNamespace ? 'WPAI-REST-Routen sind registriert.' : 'WPAI-REST-Routen wurden nicht erkannt.',
            ],
            [
                'label' => 'Planning Mode',
                'status' => $planningState,
                'details' => $planningDetails,
            ],
            [
                'label' => 'Read-Zugriff (App Passwort)',
                'status' => 'aktiv',
                'details' => 'Basic/Auth mit App Passwort erlaubt Lesezugriff.',
            ],
            [
                'label' => 'Read-Zugriff (Session)',
                'status' => $emergencyLock ? 'gesperrt' : 'aktiv',
                'details' => $readSessionDetails,
            ],
            [
                'label' => 'Write-Zugriff (global)',
                'status' => ($fullWrite && ! $emergencyLock) ? 'aktiv' : 'inaktiv',
                'details' => ($fullWrite && ! $emergencyLock)
                    ? 'Schreiboperationen sind freigeschaltet.'
                    : ($emergencyLock ? 'Notfall-Sperre ist aktiv.' : 'Voller Schreibzugriff ist deaktiviert.'),
            ],
            [
                'label' => 'Readonly-Mode',
                'status' => $readonly ? 'aktiv' : 'inaktiv',
                'details' => $readonly
                    ? 'Plan-Validierung blockiert Schreibaktionen standardmäßig.'
                    : 'Read-only-Modus ist ausgeschaltet.',
            ],
            [
                'label' => 'Core Analyzer Endpunkte',
                'status' => $coreAnalyzerEnabled ? 'aktiv' : 'inaktiv',
                'details' => $coreAnalyzerEnabled
                    ? '/core/report und /system/status sind aktiv.'
                    : 'Modul core_analyzer ist deaktiviert.',
            ],
            [
                'label' => 'OpenRouter Modelle API',
                'status' => ($openRouterEnabled && $openRouterHasKey) ? 'aktiv' : 'inaktiv',
                'details' => $openRouterModelDetails,
            ],
            [
                'label' => 'OpenRouter Chat API',
                'status' => $openRouterChatReady ? 'aktiv' : 'inaktiv',
                'details' => $openRouterChatDetails,
            ],
            [
                'label' => 'Rate Limit',
                'status' => 'aktiv',
                'details' => 'Limit: ' . (string) ((int) ($settings['rate_limit_per_minute'] ?? 60)) . ' Requests pro Minute.',
            ],
            [
                'label' => 'Index/Chunk Pipeline API',
                'status' => ($fullWrite && ! $emergencyLock) ? 'aktiv' : 'nur_lesen',
                'details' => ($fullWrite && ! $emergencyLock)
                    ? '/index/pipeline/* ist fuer Step-Ausfuehrung aktiv.'
                    : 'Statusabfragen sind möglich. Für Ausführung ist voller Schreibzugriff ohne Notfall-Sperre nötig.',
            ],
            [
                'label' => 'Queue Worker Cron',
                'status' => $workerTick === '' ? 'unbekannt' : $workerState,
                'details' => $workerDetails,
            ],
        ];
    }

    private function latestPipelineRuns(int $limit = 15): array
    {
        $limit = max(1, min(100, $limit));
        $runs = get_option('agent_chunk_pipeline_runs', []);
        if (! is_array($runs)) {
            return [];
        }

        return array_slice($runs, 0, $limit);
    }

    private function formatSeoStatus(array $report): string
    {
        $seo = is_array($report['system_state']['seo_plugins'] ?? null) ? $report['system_state']['seo_plugins'] : [];
        $active = is_array($seo['active'] ?? null) ? $seo['active'] : [];
        $installed = is_array($seo['installed'] ?? null) ? $seo['installed'] : [];

        if ($active !== []) {
            $labels = [];
            foreach ($active as $entry) {
                $label = (string) ($entry['label'] ?? '');
                if ($label !== '') {
                    $labels[] = $label;
                }
            }
            if ($labels !== []) {
                return 'Aktiv: ' . implode(', ', array_values(array_unique($labels)));
            }
        }

        if ($installed !== []) {
            $labels = [];
            foreach ($installed as $entry) {
                $label = (string) ($entry['label'] ?? '');
                if ($label !== '') {
                    $labels[] = $label;
                }
            }
            if ($labels !== []) {
                return 'Installiert (inaktiv): ' . implode(', ', array_values(array_unique($labels)));
            }
        }

        return 'Nicht gefunden (wpZeo, Yoast, Rank Math).';
    }

    private function hasWpAiNamespace(): bool
    {
        if (! function_exists('rest_get_server')) {
            return false;
        }

        $server = rest_get_server();
        if (! is_object($server) || ! method_exists($server, 'get_routes')) {
            return false;
        }

        $routes = $server->get_routes();
        if (! is_array($routes)) {
            return false;
        }

        foreach (array_keys($routes) as $route) {
            if (! is_string($route)) {
                continue;
            }
            if (str_starts_with($route, '/wpai/v1/') || $route === '/wpai/v1') {
                return true;
            }
        }

        return false;
    }

    public function renderOpenRouter(): void
    {
        if (! current_user_can('manage_options')) {
            return;
        }

        $settings = $this->options->all();
        $connectionStatus = $this->openRouterConnectionStatus();
        $statusColor = match ((string) ($connectionStatus['state'] ?? 'unknown')) {
            'connected' => '#1d7a1d',
            'disconnected' => '#b42318',
            'disabled' => '#b54708',
            'no_key' => '#6b7280',
            default => '#6b7280',
        };
        ?>
        <div class="wrap">
            <h1>OpenRouter</h1>
            <p>
                <strong style="color: <?php echo esc_attr($statusColor); ?>;">[<?php echo esc_html((string) $connectionStatus['label']); ?>]</strong>
            </p>
            <p>Konfiguration für OpenRouter-Zugriff und Modellverbindung.</p>
            <?php if (($connectionStatus['details'] ?? '') !== '') : ?>
                <p><em><?php echo esc_html((string) $connectionStatus['details']); ?></em></p>
            <?php endif; ?>

            <form method="post" action="options.php">
                <?php settings_fields('agent_settings_group'); ?>
                <p>
                    <label>
                        <input type="checkbox" name="agent_settings[openrouter][enabled]" <?php checked((bool) (($settings['openrouter']['enabled'] ?? false))); ?> />
                        OpenRouter-Integration aktivieren
                    </label>
                </p>
                <p>
                    <label for="agent-openrouter-api-key">OpenRouter API-Key</label><br />
                    <input id="agent-openrouter-api-key" type="password" class="regular-text" name="agent_settings[openrouter][api_key]" value="<?php echo esc_attr((string) ($settings['openrouter']['api_key'] ?? '')); ?>" autocomplete="off" />
                </p>
                <p>
                    <label for="agent-openrouter-base-url">OpenRouter Basis-URL</label><br />
                    <input id="agent-openrouter-base-url" type="url" class="regular-text" name="agent_settings[openrouter][base_url]" value="<?php echo esc_attr((string) ($settings['openrouter']['base_url'] ?? 'https://openrouter.ai/api/v1')); ?>" />
                </p>
                <p>
                    <label for="agent-openrouter-site-url">HTTP-Referer-URL (optional)</label><br />
                    <input id="agent-openrouter-site-url" type="url" class="regular-text" name="agent_settings[openrouter][site_url]" value="<?php echo esc_attr((string) ($settings['openrouter']['site_url'] ?? '')); ?>" />
                </p>
                <p>
                    <label for="agent-openrouter-app-name">X-Title App-Name</label><br />
                    <input id="agent-openrouter-app-name" type="text" class="regular-text" name="agent_settings[openrouter][app_name]" value="<?php echo esc_attr((string) ($settings['openrouter']['app_name'] ?? 'Agent Bridge')); ?>" />
                </p>
                <p>
                    <label for="agent-openrouter-timeout">OpenRouter Timeout (Sekunden)</label><br />
                    <input id="agent-openrouter-timeout" type="number" min="5" max="180" name="agent_settings[openrouter][timeout]" value="<?php echo esc_attr((string) ($settings['openrouter']['timeout'] ?? 60)); ?>" />
                </p>
                <p>
                    <label for="agent-openrouter-planner-model">Planer-Modell</label><br />
                    <input id="agent-openrouter-planner-model" type="text" class="regular-text" name="agent_settings[openrouter][planner_model]" value="<?php echo esc_attr((string) ($settings['openrouter']['planner_model'] ?? 'openrouter/free')); ?>" />
                </p>
                <p>
                    <label for="agent-openrouter-writer-model">Texter-Modell</label><br />
                    <input id="agent-openrouter-writer-model" type="text" class="regular-text" name="agent_settings[openrouter][writer_model]" value="<?php echo esc_attr((string) ($settings['openrouter']['writer_model'] ?? 'openrouter/free')); ?>" />
                </p>
                <p>
                    <label for="agent-openrouter-image-model">Bild-Modell</label><br />
                    <input id="agent-openrouter-image-model" type="text" class="regular-text" name="agent_settings[openrouter][image_model]" value="<?php echo esc_attr((string) ($settings['openrouter']['image_model'] ?? 'openrouter/free')); ?>" />
                </p>
                <p>
                    <label>
                        <input type="checkbox" name="agent_settings[openrouter][model_filter_enabled]" <?php checked((bool) (($settings['openrouter']['model_filter_enabled'] ?? true))); ?> />
                        Modellfilter aktivieren
                    </label>
                </p>
                <p>
                    <label for="agent-openrouter-model-filter">Modellfilter (mit "|" getrennt)</label><br />
                    <input id="agent-openrouter-model-filter" type="text" class="regular-text" name="agent_settings[openrouter][model_filter]" value="<?php echo esc_attr((string) ($settings['openrouter']['model_filter'] ?? 'openai|x-ai|google|anthropic|meta-llama|free')); ?>" />
                </p>

                <?php submit_button('OpenRouter-Einstellungen speichern'); ?>
            </form>

            <?php if (($connectionStatus['state'] ?? '') === 'connected') : ?>
                <?php
                $models = is_array($connectionStatus['models'] ?? null) ? $connectionStatus['models'] : [];
                $totalModels = (int) ($connectionStatus['models_total'] ?? count($models));
                $shownModels = count($models);
                $filterInfo = (string) ($connectionStatus['model_filter_info'] ?? '');
                ?>
                <h2>Verfuegbare Modelle</h2>
                <p>
                    <?php echo esc_html((string) $totalModels); ?> Modelle verfuegbar.
                    <?php if ($shownModels < $totalModels) : ?>
                        Es werden <?php echo esc_html((string) $shownModels); ?> Modelle angezeigt.
                    <?php endif; ?>
                </p>
                <?php if ($filterInfo !== '') : ?>
                    <p><em><?php echo esc_html($filterInfo); ?></em></p>
                <?php endif; ?>

                <?php if ($models === []) : ?>
                    <p><em>Keine Modelle gefunden.</em></p>
                <?php else : ?>
                    <table class="widefat striped">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Name</th>
                                <th>Kontextlaenge</th>
                                <th>Pricing (prompt | completion)</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($models as $model) : ?>
                                <tr>
                                    <td><code><?php echo esc_html((string) ($model['id'] ?? '')); ?></code></td>
                                    <td><?php echo esc_html((string) ($model['name'] ?? '')); ?></td>
                                    <td><?php echo esc_html((string) ($model['context_length'] ?? '')); ?></td>
                                    <td><?php echo esc_html((string) ($model['pricing'] ?? '')); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            <?php endif; ?>
        </div>
        <?php
    }

    public function renderIndexing(): void
    {
        if (! current_user_can('manage_options')) {
            return;
        }

        $postIdFilter = isset($_GET['post_id']) ? max(0, (int) $_GET['post_id']) : 0;
        $chunkRows = $this->chunkRows($postIdFilter > 0 ? $postIdFilter : null, 200);
        $summary = $this->chunkSummary($postIdFilter > 0 ? $postIdFilter : null);
        $pipelineState = $this->pipelineStateRows(100);
        $runs = $this->latestPipelineRuns(30);
        ?>
        <div class="wrap">
            <h1>Indexing</h1>
            <p>Übersicht zur Chunking-Pipeline und zu gespeicherten Chunks.</p>

            <form method="get" action="">
                <input type="hidden" name="page" value="agent-indexing" />
                <p>
                    <label for="agent-indexing-post-id">Post-ID-Filter (optional)</label><br />
                    <input id="agent-indexing-post-id" type="number" min="1" name="post_id" value="<?php echo $postIdFilter > 0 ? esc_attr((string) $postIdFilter) : ''; ?>" />
                    <?php submit_button('Filter anwenden', 'secondary', '', false); ?>
                </p>
            </form>

            <h2>Chunk-Zusammenfassung</h2>
            <table class="widefat striped">
                <tbody>
                    <tr><td>Filter</td><td><?php echo $postIdFilter > 0 ? esc_html('post_id=' . (string) $postIdFilter) : 'kein Filter'; ?></td></tr>
                    <tr><td>Chunks gesamt</td><td><?php echo esc_html((string) ((int) ($summary['chunks_total'] ?? 0))); ?></td></tr>
                    <tr><td>Unterschiedliche Beiträge</td><td><?php echo esc_html((string) ((int) ($summary['distinct_posts'] ?? 0))); ?></td></tr>
                    <tr><td>Chunks mit Embeddings</td><td><?php echo esc_html((string) ((int) ($summary['with_embeddings'] ?? 0))); ?></td></tr>
                </tbody>
            </table>

            <h2>Pipeline-Status</h2>
            <?php if ($pipelineState === []) : ?>
                <p><em>Bisher kein Pipeline-Status vorhanden.</em></p>
            <?php else : ?>
                <table class="widefat striped">
                    <thead>
                        <tr>
                            <th>Post ID</th>
                            <th>Status</th>
                            <th>Hash</th>
                            <th>Updated (UTC)</th>
                            <th>Steps</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($pipelineState as $row) : ?>
                            <tr>
                                <td><?php echo esc_html((string) ((int) ($row['post_id'] ?? 0))); ?></td>
                                <td><?php echo esc_html((string) ($row['status'] ?? 'unknown')); ?></td>
                                <td><code><?php echo esc_html((string) ($row['content_hash'] ?? '')); ?></code></td>
                                <td><?php echo esc_html((string) ($row['updated_at'] ?? '')); ?></td>
                                <td><?php echo esc_html((string) ($row['steps_label'] ?? '')); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>

            <h2>Letzte Pipeline-Läufe</h2>
            <?php if ($runs === []) : ?>
                <p><em>Noch keine Pipeline-Läufe vorhanden.</em></p>
            <?php else : ?>
                <table class="widefat striped">
                    <thead>
                        <tr>
                            <th>Run</th>
                            <th>Post ID</th>
                            <th>Status</th>
                            <th>Fehlgeschritt</th>
                            <th>Start (UTC)</th>
                            <th>Ende (UTC)</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($runs as $run) : ?>
                            <tr>
                                <td><code><?php echo esc_html((string) ($run['run_id'] ?? '')); ?></code></td>
                                <td><?php echo esc_html((string) ((int) ($run['post_id'] ?? 0))); ?></td>
                                <td><?php echo esc_html((string) ($run['status'] ?? 'unknown')); ?></td>
                                <td><?php echo esc_html((string) ($run['failed_step'] ?? '')); ?></td>
                                <td><?php echo esc_html((string) ($run['started_at'] ?? '')); ?></td>
                                <td><?php echo esc_html((string) ($run['finished_at'] ?? '')); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>

            <h2>Chunk-Tabelle (letzte 200)</h2>
            <?php if ($chunkRows === []) : ?>
                <p><em>Keine Chunks gefunden.</em></p>
            <?php else : ?>
                <table class="widefat striped">
                    <thead>
                        <tr>
                            <th>Post ID</th>
                            <th>Chunk No</th>
                            <th>Text Preview</th>
                            <th>Embedding</th>
                            <th>Updated (UTC)</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($chunkRows as $row) : ?>
                            <tr>
                                <td><?php echo esc_html((string) ((int) ($row['post_id'] ?? 0))); ?></td>
                                <td><?php echo esc_html((string) ((int) ($row['chunk_no'] ?? 0))); ?></td>
                                <td><?php echo esc_html((string) ($row['text_preview'] ?? '')); ?></td>
                                <td><?php echo esc_html((string) ($row['embedding_state'] ?? 'none')); ?></td>
                                <td><?php echo esc_html((string) ($row['updated_at'] ?? '')); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
        <?php
    }

    public function renderPlanning(): void
    {
        if (! current_user_can('manage_options')) {
            return;
        }
        if (function_exists('nocache_headers')) {
            nocache_headers();
        }
        if (! headers_sent()) {
            header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
            header('Pragma: no-cache');
            header('Expires: Wed, 11 Jan 1984 05:00:00 GMT');
        }

        $notice = '';
        $noticeType = 'updated';

        if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['agent_planning_action'])) {
            check_admin_referer('agent_planning_action');
            $action = sanitize_key((string) $_POST['agent_planning_action']);

            if ($action === 'run_gates') {
                $this->planning->runGates();
                $notice = 'Prüfung abgeschlossen. Die nächsten Schritte stehen jetzt oben im Bereich "Nächster Schritt".';
            } elseif ($action === 'install_wpzeo_auto') {
                $installed = $this->wpZeoInstaller->installLatest();
                if ($installed instanceof \WP_Error) {
                    $notice = $installed->get_error_message();
                    $noticeType = 'error';
                } else {
                    update_option(\Agent\Modules\Planning\PlanningService::OPTION_SELECTED_SEO_PLUGIN, 'wpzeo-zero-seo/wpzeo-zero-seo.php', false);
                    $notice = 'wpZeo wurde installiert und aktiviert. Der SEO-Blocker wurde erneut geprüft.';
                }
                $this->planning->runGates();
            } elseif ($action === 'fix_seo_single_wpzeo' || $action === 'fix_seo_single_yoast' || $action === 'fix_seo_single_rankmath') {
                $target = 'wpzeo';
                if ($action === 'fix_seo_single_yoast') {
                    $target = 'yoast';
                } elseif ($action === 'fix_seo_single_rankmath') {
                    $target = 'rankmath';
                }
                $result = $this->enforceSingleSeoPlugin($target);
                if ($result instanceof \WP_Error) {
                    $notice = $result->get_error_message();
                    $noticeType = 'error';
                } else {
                    $notice = 'SEO-Konflikt gelöst. Aktiv ist jetzt: ' . (string) ($result['active_label'] ?? $target) . '.';
                }
                $this->planning->runGates();
            } elseif ($action === 'save_input') {
                $rawInput = is_array($_POST['agent_planning_input'] ?? null) ? (array) $_POST['agent_planning_input'] : [];
                $saved = $this->planning->saveUserInput($rawInput, (int) get_current_user_id());
                if ($saved instanceof \WP_Error) {
                    $notice = $saved->get_error_message();
                    $noticeType = 'error';
                } else {
                    $notice = 'Nutzerangaben gespeichert. Du kannst jetzt mit der nächsten Planungsstufe fortfahren.';
                }
            } elseif ($action === 'save_core_baseline') {
                $coreInput = is_array($_POST['agent_core'] ?? null) ? (array) $_POST['agent_core'] : [];
                update_option('blogname', sanitize_text_field((string) ($coreInput['blogname'] ?? '')), false);
                update_option('blogdescription', sanitize_text_field((string) ($coreInput['blogdescription'] ?? '')), false);

                $timezone = sanitize_text_field((string) ($coreInput['timezone_string'] ?? ''));
                if ($timezone !== '' && in_array($timezone, timezone_identifiers_list(), true)) {
                    update_option('timezone_string', $timezone, false);
                }

                $privacyPageId = max(0, (int) ($coreInput['privacy_policy_page'] ?? 0));
                update_option('wp_page_for_privacy_policy', $privacyPageId, false);

                $permalinkStructureRaw = sanitize_text_field((string) ($coreInput['permalink_structure'] ?? ''));

                $permalinkStructure = $this->sanitizePermalinkStructure($permalinkStructureRaw);
                update_option('permalink_structure', $permalinkStructure, false);
                if (function_exists('flush_rewrite_rules')) {
                    flush_rewrite_rules(false);
                }

                $this->planning->runGates();
                $notice = 'WordPress-Basisdaten gespeichert und erneut geprüft.';
            } elseif ($action === 'set_permalink_standard') {
                update_option('permalink_structure', '/%postname%/', false);
                $this->options->set('planning.permalink_strategy', 'standard');
                $this->options->set('planning.cluster_enabled', false);
                if (function_exists('flush_rewrite_rules')) {
                    flush_rewrite_rules(false);
                }
                $this->planning->runGates();
                $notice = 'Standard-Struktur (ohne Kategorie) wurde gesetzt und erneut geprüft.';
            } elseif ($action === 'set_permalink_cluster') {
                update_option('permalink_structure', '/%category%/%postname%/', false);
                $this->options->set('planning.permalink_strategy', 'cluster');
                $this->options->set('planning.cluster_enabled', true);
                if (function_exists('flush_rewrite_rules')) {
                    flush_rewrite_rules(false);
                }
                $this->planning->runGates();
                $notice = 'Cluster-Struktur (mit Kategorie) wurde gesetzt und erneut geprüft.';
            }
        }

        $run = $this->planning->currentRun();
        $canCollectInput = (bool) ($run['summary']['can_collect_input'] ?? false);
        $userInputDone = (string) ($run['gates']['user_input_v1']['status'] ?? 'pending') === 'completed';
        $input = is_array($run['gates']['user_input_v1']['data'] ?? null) ? (array) $run['gates']['user_input_v1']['data'] : [];

        $gateLabels = [
            'dependencies' => 'Gate 1: Abhaengigkeiten',
            'core_baseline' => 'Gate 2: WordPress-Basis',
            'openrouter' => 'Gate 3: OpenRouter',
            'user_input_v1' => 'Gate 4: Nutzerangaben',
        ];
        $gateOrder = ['dependencies', 'core_baseline', 'openrouter', 'user_input_v1'];
        $firstBlockedGate = '';
        foreach (['dependencies', 'core_baseline', 'openrouter'] as $gateKey) {
            $gateState = (string) ($run['gates'][$gateKey]['status'] ?? 'pending');
            if ($gateState === 'blocked') {
                $firstBlockedGate = $gateKey;
                break;
            }
        }

        $nextTitle = 'Noch keine Prüfung gelaufen.';
        $nextMessage = 'Starte jetzt die Grundprüfung. Danach zeigt wpAI dir den konkreten nächsten Schritt.';
        $nextType = 'info';
        $nextActionType = 'run_gates';
        $nextActionLabel = 'Gates jetzt pruefen';
        $nextActionUrl = '';
        $seoInstallButtons = [];
        $seoConflictButtons = [];
        $blockingError = null;

        if ($firstBlockedGate !== '') {
            $nextType = 'error';
            $nextTitle = 'Aktuell blockiert: ' . (string) ($gateLabels[$firstBlockedGate] ?? $firstBlockedGate);
            $nextMessage = 'Behebe den zuerst genannten Blocker und starte danach erneut die Prüfung.';

            $blockers = is_array($run['gates'][$firstBlockedGate]['blockers'] ?? null) ? (array) $run['gates'][$firstBlockedGate]['blockers'] : [];
            $firstAction = [];
            $hasSeoBlocker = false;
            $hasSeoConflictBlocker = false;
            foreach ($blockers as $blockedFinding) {
                if (! is_array($blockedFinding)) {
                    continue;
                }
                $code = (string) ($blockedFinding['code'] ?? '');
                if ($code === 'seo_plugin_missing') {
                    $hasSeoBlocker = true;
                }
                if ($code === 'seo_multiple_active' || $code === 'seo_multiple_installed') {
                    $hasSeoConflictBlocker = true;
                }
            }
            if ($blockers !== []) {
                $firstAction = is_array($blockers[0]['action'] ?? null) ? (array) $blockers[0]['action'] : [];
                $firstMessage = trim((string) ($blockers[0]['message'] ?? ''));
                if ($firstMessage !== '') {
                    $nextMessage = $firstMessage;
                }
                $blockingError = is_array($blockers[0]) ? (array) $blockers[0] : null;
            }
            if ((string) ($firstAction['url'] ?? '') !== '') {
                $nextActionType = 'link';
                $nextActionLabel = (string) ($firstAction['label'] ?? 'Jetzt beheben');
                $nextActionUrl = (string) $firstAction['url'];
            }
            if ($hasSeoBlocker) {
                $seoInstallButtons = $this->planningSeoInstallButtons();
            }
            if ($hasSeoConflictBlocker) {
                $seoConflictButtons = $this->planningSeoConflictButtons();
            }
        } elseif ($canCollectInput && ! $userInputDone) {
            $nextType = 'warning';
            $nextTitle = 'Technische Basis ist bereit.';
            $nextMessage = 'Bitte jetzt die 8 Nutzerangaben ausfüllen. Ohne diese Angaben sind keine sinnvollen Planungsergebnisse möglich.';
            $nextActionType = 'link';
            $nextActionLabel = 'Zu Gate 4 springen';
            $nextActionUrl = '#agent-planning-input';
        } elseif ($canCollectInput && $userInputDone) {
            $nextType = 'success';
            $nextTitle = 'Planungsmodus ist vollständig bereit.';
            $nextMessage = 'Alle Voraussetzungen sind erfüllt. Du kannst jetzt Sitemap-, Navigation- und Seiten-Blueprints erzeugen.';
            $nextActionType = 'none';
            $nextActionLabel = '';
        } elseif ((string) ($run['status'] ?? 'idle') !== 'idle') {
            $nextTitle = 'Status bitte neu prüfen.';
            $nextMessage = 'Wenn du Einstellungen oder Plugins geändert hast, starte die Prüfung erneut.';
            $nextType = 'info';
        }

        $nextBoxStyle = 'padding:10px 12px; border:1px solid #dcdcde; border-left-width:4px; background:#fff;';
        if ($nextType === 'error') {
            $nextBoxStyle = 'padding:10px 12px; border:1px solid #e5b3b3; border-left-width:4px; border-left-color:#d63638; background:#fff;';
        } elseif ($nextType === 'warning') {
            $nextBoxStyle = 'padding:10px 12px; border:1px solid #ead9a8; border-left-width:4px; border-left-color:#dba617; background:#fff;';
        } elseif ($nextType === 'success') {
            $nextBoxStyle = 'padding:10px 12px; border:1px solid #b8d8b8; border-left-width:4px; border-left-color:#00a32a; background:#fff;';
        }
        ?>
        <div class="wrap">
            <h1>Planning</h1>
            <p>Gated Planning Mode fuer die Erstkonfiguration.</p>
            <style>
                .agent-field-hint {
                    display: block;
                    width: fit-content;
                    margin-top: 6px;
                    padding: 3px 8px;
                    border-radius: 4px;
                    border: 1px solid #dcdcde;
                    background: #f6f7f7;
                    color: #1d2327;
                    font-weight: 500;
                }
                .agent-field-hint--warning {
                    border-color: #ead9a8;
                    background: #fff8e5;
                    color: #6b4f00;
                }
                .agent-field-hint--blocker {
                    border-color: #e5b3b3;
                    background: #fff1f1;
                    color: #8a1f11;
                }
                .agent-field-hint--ok {
                    border-color: #b8d8b8;
                    background: #f0fff4;
                    color: #0f5132;
                }
            </style>
            <h2>[USER UI] Nutzerfuehrung</h2>
            <?php
            $currentState = (string) ($run['current_state'] ?? 'IDLE');
            $breadcrumbItems = [
                ['id' => 'GATE_1_DEPENDENCIES', 'label' => '1. Abhaengigkeiten'],
                ['id' => 'GATE_2_CORE_BASELINE', 'label' => '2. WordPress-Basis'],
                ['id' => 'GATE_3_OPENROUTER', 'label' => '3. OpenRouter'],
                ['id' => 'GATE_4_USER_INPUT', 'label' => '4. Nutzerangaben'],
            ];
            ?>
            <p style="margin-top:12px; margin-bottom:14px;">
                <?php foreach ($breadcrumbItems as $idx => $item) : ?>
                    <?php
                    $isActive = $currentState === (string) ($item['id'] ?? '');
                    $style = $isActive ? 'color:#15803d;font-weight:700;' : 'color:#4b5563;';
                    ?>
                    <span style="<?php echo esc_attr($style); ?>"><?php echo esc_html((string) ($item['label'] ?? '')); ?></span>
                    <?php if ($idx < count($breadcrumbItems) - 1) : ?>
                        <span style="color:#9ca3af;"> &gt; </span>
                    <?php endif; ?>
                <?php endforeach; ?>
            </p>

            <h2>WordPress-Basisdaten (Pflicht)</h2>
            <p>Diese Einstellungen müssen für den Planungsmodus korrekt gesetzt sein. Du kannst sie direkt hier bearbeiten.</p>
            <?php $coreRows = $this->planningCoreRows($run); ?>
            <?php
            $coreByLabel = [];
            foreach ($coreRows as $row) {
                if (! is_array($row)) {
                    continue;
                }
                $label = (string) ($row['label'] ?? '');
                if ($label !== '') {
                    $coreByLabel[$label] = $row;
                }
            }
            $blognameCurrent = (string) get_option('blogname', '');
            $taglineCurrent = (string) get_option('blogdescription', '');
            $timezoneCurrent = (string) get_option('timezone_string', '');
            $privacyCurrent = (int) get_option('wp_page_for_privacy_policy', 0);
            $permalinkCurrent = (string) get_option('permalink_structure', '');
            $permalinkStrategy = (string) $this->options->get('planning.permalink_strategy', '');
            $clusterEnabledRaw = $this->options->get('planning.cluster_enabled', null);
            $clusterEnabled = null;
            if (is_bool($clusterEnabledRaw)) {
                $clusterEnabled = $clusterEnabledRaw;
            }
            if (! in_array($permalinkStrategy, ['standard', 'cluster'], true) && is_bool($clusterEnabled)) {
                $permalinkStrategy = $clusterEnabled ? 'cluster' : 'standard';
            }
            // Backfill strategy from current permalink structure if strategy is not set yet.
            if (! in_array($permalinkStrategy, ['standard', 'cluster'], true)) {
                if ($permalinkCurrent === '/%postname%/') {
                    $permalinkStrategy = 'standard';
                    $this->options->set('planning.permalink_strategy', 'standard');
                    $this->options->set('planning.cluster_enabled', false);
                } elseif ($permalinkCurrent === '/%category%/%postname%/') {
                    $permalinkStrategy = 'cluster';
                    $this->options->set('planning.permalink_strategy', 'cluster');
                    $this->options->set('planning.cluster_enabled', true);
                }
            }
            $permalinkStrategyKnown = in_array($permalinkStrategy, ['standard', 'cluster'], true);
            ?>
            <form method="post">
                <?php wp_nonce_field('agent_planning_action'); ?>
                <input type="hidden" name="agent_planning_action" value="save_core_baseline" />
                <table class="form-table" role="presentation">
                    <tbody>
                        <tr>
                            <th scope="row"><label for="agent-core-blogname">Website-Name</label></th>
                            <td>
                                <input id="agent-core-blogname" type="text" class="regular-text" name="agent_core[blogname]" value="<?php echo esc_attr($blognameCurrent); ?>" />
                                <?php if (is_array($coreByLabel['Website-Name'] ?? null)) : ?>
                                    <p class="description agent-field-hint <?php echo esc_attr($this->planningFieldHintClass((array) $coreByLabel['Website-Name'])); ?>"><?php echo esc_html($this->planningFieldHintText((array) $coreByLabel['Website-Name'])); ?></p>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="agent-core-tagline">Tagline</label></th>
                            <td>
                                <input id="agent-core-tagline" type="text" class="regular-text" name="agent_core[blogdescription]" value="<?php echo esc_attr($taglineCurrent); ?>" />
                                <?php if (is_array($coreByLabel['Tagline'] ?? null)) : ?>
                                    <p class="description agent-field-hint <?php echo esc_attr($this->planningFieldHintClass((array) $coreByLabel['Tagline'])); ?>"><?php echo esc_html($this->planningFieldHintText((array) $coreByLabel['Tagline'])); ?></p>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="agent-core-timezone">Zeitzone</label></th>
                            <td>
                                <select id="agent-core-timezone" name="agent_core[timezone_string]">
                                    <?php
                                    if (function_exists('wp_timezone_choice')) {
                                        echo wp_timezone_choice($timezoneCurrent !== '' ? $timezoneCurrent : 'UTC');
                                    } else {
                                        echo '<option value="' . esc_attr($timezoneCurrent) . '">' . esc_html($timezoneCurrent !== '' ? $timezoneCurrent : 'UTC') . '</option>';
                                    }
                                    ?>
                                </select>
                                <?php if (is_array($coreByLabel['Zeitzone'] ?? null)) : ?>
                                    <p class="description agent-field-hint <?php echo esc_attr($this->planningFieldHintClass((array) $coreByLabel['Zeitzone'])); ?>"><?php echo esc_html($this->planningFieldHintText((array) $coreByLabel['Zeitzone'])); ?></p>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="agent-core-privacy">Datenschutzseite</label></th>
                            <td>
                                <?php
                                if (function_exists('wp_dropdown_pages')) {
                                    wp_dropdown_pages([
                                        'name' => 'agent_core[privacy_policy_page]',
                                        'id' => 'agent-core-privacy',
                                        'show_option_none' => 'Nicht gesetzt',
                                        'option_none_value' => '0',
                                        'selected' => $privacyCurrent,
                                        'post_status' => ['publish', 'draft', 'private', 'pending'],
                                    ]);
                                } else {
                                    ?>
                                    <input id="agent-core-privacy" type="number" min="0" name="agent_core[privacy_policy_page]" value="<?php echo esc_attr((string) $privacyCurrent); ?>" />
                                    <?php
                                }
                                ?>
                                <?php if (is_array($coreByLabel['Datenschutzseite'] ?? null)) : ?>
                                    <p class="description agent-field-hint <?php echo esc_attr($this->planningFieldHintClass((array) $coreByLabel['Datenschutzseite'])); ?>"><?php echo esc_html($this->planningFieldHintText((array) $coreByLabel['Datenschutzseite'])); ?></p>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Permalink-Struktur</th>
                            <td>
                                <input id="agent-core-permalink" type="text" class="regular-text code" name="agent_core[permalink_structure]" value="<?php echo esc_attr($permalinkCurrent); ?>" placeholder="/%postname%/" />
                                <?php
                                $permalinkRow = is_array($coreByLabel['Permalink-Struktur'] ?? null) ? (array) $coreByLabel['Permalink-Struktur'] : null;
                                $showPermalinkCoreHint = true;
                                if ($permalinkRow !== null && $permalinkStrategyKnown && (string) ($permalinkRow['status'] ?? '') === 'ok') {
                                    $showPermalinkCoreHint = false;
                                }
                                ?>
                                <?php if ($permalinkRow !== null && $showPermalinkCoreHint) : ?>
                                    <p class="description agent-field-hint <?php echo esc_attr($this->planningFieldHintClass((array) $coreByLabel['Permalink-Struktur'])); ?>"><?php echo esc_html($this->planningFieldHintText((array) $coreByLabel['Permalink-Struktur'])); ?></p>
                                <?php endif; ?>
                                <?php if ($permalinkStrategyKnown) : ?>
                                    <p class="description agent-field-hint agent-field-hint--ok">
                                        <?php echo esc_html('wpAI URL-Strategie aktiv: ' . $this->planningPermalinkStrategyLabel($permalinkStrategy)); ?>
                                    </p>
                                <?php else : ?>
                                    <p class="description agent-field-hint agent-field-hint--warning">Hinweis: URL-Strategie noch nicht festgelegt. Bitte eine Struktur per Button setzen. Ohne Auswahl wird die Entscheidung später in der wpAI-Basiskonfiguration abgefragt.</p>
                                <?php endif; ?>
                                <div style="margin-top:8px; display:flex; gap:8px; flex-wrap:wrap;">
                                    <form method="post" style="margin:0;">
                                        <?php wp_nonce_field('agent_planning_action'); ?>
                                        <input type="hidden" name="agent_planning_action" value="set_permalink_standard" />
                                        <button type="submit" class="button button-secondary">Standard-Struktur setzen</button>
                                    </form>
                                    <form method="post" style="margin:0;">
                                        <?php wp_nonce_field('agent_planning_action'); ?>
                                        <input type="hidden" name="agent_planning_action" value="set_permalink_cluster" />
                                        <button type="submit" class="button button-secondary">Cluster-Struktur (mit Kategorie) setzen</button>
                                    </form>
                                </div>
                            </td>
                        </tr>
                    </tbody>
                </table>
                <?php submit_button('Core-Einstellungen speichern & erneut prüfen', 'secondary', 'submit', false); ?>
            </form>

            <h2>Naechster Schritt</h2>
            <div style="<?php echo esc_attr($nextBoxStyle); ?>">
                <?php if ($blockingError !== null) : ?>
                    <?php
                    $errorCode = (string) ($blockingError['code'] ?? 'unknown');
                    $errorMessage = (string) ($blockingError['message'] ?? $nextMessage);
                    $errorHeading = $this->planningErrorHeading($errorCode, (string) ($blockingError['label'] ?? ''));
                    ?>
                    <h3><?php echo esc_html($errorHeading); ?></h3>
                    <p><?php echo esc_html($errorMessage); ?></p>
                <?php else : ?>
                    <p><strong><?php echo esc_html($nextTitle); ?></strong></p>
                    <p><?php echo esc_html($nextMessage); ?></p>
                <?php endif; ?>
                <p>
                    <?php if ($nextActionType === 'run_gates') : ?>
                        <form method="post" style="display:inline;">
                            <?php wp_nonce_field('agent_planning_action'); ?>
                            <input type="hidden" name="agent_planning_action" value="run_gates" />
                            <button type="submit" class="button button-primary"><?php echo esc_html($nextActionLabel); ?></button>
                        </form>
                    <?php elseif ($nextActionType === 'link' && $nextActionUrl !== '') : ?>
                        <a class="button button-primary" href="<?php echo esc_url($nextActionUrl); ?>"><?php echo esc_html($nextActionLabel); ?></a>
                    <?php endif; ?>
                </p>
                <?php if ($seoInstallButtons !== []) : ?>
                    <p>
                        <?php foreach ($seoInstallButtons as $button) : ?>
                            <?php if ((string) ($button['type'] ?? '') === 'post' && (string) ($button['action'] ?? '') !== '') : ?>
                                <form method="post" style="display:inline-block; margin-right:8px;">
                                    <?php wp_nonce_field('agent_planning_action'); ?>
                                    <input type="hidden" name="agent_planning_action" value="<?php echo esc_attr((string) $button['action']); ?>" />
                                    <button type="submit" class="button <?php echo esc_attr(((bool) ($button['recommended'] ?? false)) ? 'button-primary' : 'button-secondary'); ?>">
                                        <?php echo esc_html((string) ($button['label'] ?? 'Installieren')); ?>
                                    </button>
                                </form>
                            <?php else : ?>
                                <a
                                    class="button <?php echo esc_attr(((bool) ($button['recommended'] ?? false)) ? 'button-primary' : 'button-secondary'); ?>"
                                    href="<?php echo esc_url((string) ($button['url'] ?? '#')); ?>"
                                    style="margin-right:8px;"
                                >
                                    <?php echo esc_html((string) ($button['label'] ?? 'Installieren')); ?>
                                </a>
                            <?php endif; ?>
                        <?php endforeach; ?>
                    </p>
                <?php endif; ?>
                <?php if ($seoConflictButtons !== []) : ?>
                    <p>
                        <?php foreach ($seoConflictButtons as $button) : ?>
                            <form method="post" style="display:inline-block; margin-right:8px;">
                                <?php wp_nonce_field('agent_planning_action'); ?>
                                <input type="hidden" name="agent_planning_action" value="<?php echo esc_attr((string) ($button['action'] ?? '')); ?>" />
                                <button type="submit" class="button <?php echo esc_attr(((bool) ($button['recommended'] ?? false)) ? 'button-primary' : 'button-secondary'); ?>">
                                    <?php echo esc_html((string) ($button['label'] ?? 'Konflikt loesen')); ?>
                                </button>
                            </form>
                        <?php endforeach; ?>
                    </p>
                <?php endif; ?>
            </div>

            <h2 id="agent-planning-input">Gate 4: User Input v1</h2>
            <?php
            $coreBlockingRows = array_values(array_filter(
                $coreRows,
                static fn (array $row): bool => (string) ($row['status'] ?? '') === 'blocker'
            ));
            $disabledAttr = $canCollectInput ? '' : 'disabled';
            ?>
            <?php if (! $canCollectInput) : ?>
                <p>Gate 4 ist derzeit gesperrt. Die Felder sind sichtbar, werden aber erst nach Behebung der Pflicht-Blocker freigeschaltet.</p>
                <?php if ($coreBlockingRows !== []) : ?>
                    <ul>
                        <?php foreach ($coreBlockingRows as $coreIssue) : ?>
                            <li>
                                <?php echo esc_html((string) ($coreIssue['message'] ?? '')); ?>
                                <?php if ((string) ($coreIssue['action_url'] ?? '') !== '') : ?>
                                    <a href="<?php echo esc_url((string) ($coreIssue['action_url'] ?? '')); ?>"><?php echo esc_html((string) ($coreIssue['action_label'] ?? 'Jetzt beheben')); ?></a>
                                <?php endif; ?>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                <?php endif; ?>
            <?php else : ?>
                <p>Die technischen Voraussetzungen sind erfüllt. Du kannst die Eingaben jetzt speichern.</p>
            <?php endif; ?>
            <form method="post">
                <?php wp_nonce_field('agent_planning_action'); ?>
                <input type="hidden" name="agent_planning_action" value="save_input" />
                <table class="form-table" role="presentation">
                    <tbody>
                        <tr>
                            <th scope="row"><label for="agent-planning-project-goal">Projektziel</label></th>
                            <td><input id="agent-planning-project-goal" type="text" class="regular-text" name="agent_planning_input[project_goal]" value="<?php echo esc_attr((string) ($input['project_goal'] ?? '')); ?>" <?php echo esc_attr($disabledAttr); ?> /></td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="agent-planning-primary-audience">Primaere Zielgruppe</label></th>
                            <td><input id="agent-planning-primary-audience" type="text" class="regular-text" name="agent_planning_input[primary_audience]" value="<?php echo esc_attr((string) ($input['primary_audience'] ?? '')); ?>" <?php echo esc_attr($disabledAttr); ?> /></td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="agent-planning-geo-focus">Geo-Fokus</label></th>
                            <td><input id="agent-planning-geo-focus" type="text" class="regular-text" name="agent_planning_input[geo_focus]" value="<?php echo esc_attr((string) ($input['geo_focus'] ?? '')); ?>" <?php echo esc_attr($disabledAttr); ?> /></td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="agent-planning-topics">Hauptthemen/Leistungen</label></th>
                            <td><input id="agent-planning-topics" type="text" class="regular-text" name="agent_planning_input[main_services_or_topics]" value="<?php echo esc_attr((string) ($input['main_services_or_topics'] ?? '')); ?>" <?php echo esc_attr($disabledAttr); ?> /></td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="agent-planning-brand-tone">Brand-Ton</label></th>
                            <td><input id="agent-planning-brand-tone" type="text" class="regular-text" name="agent_planning_input[brand_tone]" value="<?php echo esc_attr((string) ($input['brand_tone'] ?? '')); ?>" <?php echo esc_attr($disabledAttr); ?> /></td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="agent-planning-conversion-goal">Conversion-Ziel</label></th>
                            <td><input id="agent-planning-conversion-goal" type="text" class="regular-text" name="agent_planning_input[conversion_goal]" value="<?php echo esc_attr((string) ($input['conversion_goal'] ?? '')); ?>" <?php echo esc_attr($disabledAttr); ?> /></td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="agent-planning-content-depth">Content-Tiefe</label></th>
                            <td><input id="agent-planning-content-depth" type="text" class="regular-text" name="agent_planning_input[content_depth]" value="<?php echo esc_attr((string) ($input['content_depth'] ?? '')); ?>" <?php echo esc_attr($disabledAttr); ?> /></td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="agent-planning-constraints">Constraints</label></th>
                            <td><textarea id="agent-planning-constraints" class="large-text" rows="4" name="agent_planning_input[constraints]" <?php echo esc_attr($disabledAttr); ?>><?php echo esc_textarea((string) ($input['constraints'] ?? '')); ?></textarea></td>
                        </tr>
                    </tbody>
                </table>

                <?php submit_button('Input speichern', 'secondary', 'submit', false, $canCollectInput ? [] : ['disabled' => 'disabled']); ?>
            </form>

            <h2>[DEBUG] Debugging (Technische Details)</h2>
            <p><em>Die folgenden Bereiche sind fuer detaillierte Diagnose und technische Pruefung gedacht.</em></p>
            <?php if ($notice !== '') : ?>
                <div style="margin:12px 0; padding:10px 12px; border:1px solid #dcdcde; background:#f6f7f7;">
                    <p><strong>Hinweis:</strong> <?php echo esc_html($notice); ?></p>
                </div>
            <?php endif; ?>

            <h2>Status</h2>
            <table class="widefat striped">
                <tbody>
                    <tr><td>Run ID</td><td><code><?php echo esc_html((string) ($run['run_id'] ?? '')); ?></code></td></tr>
                    <tr><td>Status</td><td><?php echo esc_html((string) ($run['status'] ?? 'idle')); ?></td></tr>
                    <tr><td>Aktueller State</td><td><?php echo esc_html((string) ($run['current_state'] ?? 'IDLE')); ?></td></tr>
                    <tr><td>Blocker</td><td><?php echo esc_html((string) ((int) ($run['summary']['blocker_count'] ?? 0))); ?></td></tr>
                    <tr><td>Warnings</td><td><?php echo esc_html((string) ((int) ($run['summary']['warning_count'] ?? 0))); ?></td></tr>
                    <tr><td>Started (UTC)</td><td><?php echo esc_html((string) ($run['started_at'] ?? '')); ?></td></tr>
                    <tr><td>Updated (UTC)</td><td><?php echo esc_html((string) ($run['updated_at'] ?? '')); ?></td></tr>
                </tbody>
            </table>

            <form method="post">
                <?php wp_nonce_field('agent_planning_action'); ?>
                <input type="hidden" name="agent_planning_action" value="run_gates" />
                <?php submit_button('Gates pruefen / erneut pruefen', 'primary', 'submit', false); ?>
            </form>

            <h2>Gate 1-3 Ergebnisse</h2>
            <?php foreach (['dependencies' => 'Gate 1: Abhaengigkeiten', 'core_baseline' => 'Gate 2: WordPress-Basis', 'openrouter' => 'Gate 3: OpenRouter'] as $gateKey => $gateLabel) : ?>
                <?php $gate = is_array($run['gates'][$gateKey] ?? null) ? (array) $run['gates'][$gateKey] : []; ?>
                <h3><?php echo esc_html($gateLabel); ?> - <?php echo esc_html((string) ($gate['status'] ?? 'pending')); ?></h3>
                <p><em>Geprueft: <?php echo esc_html((string) ($gate['checked_at'] ?? '')); ?></em></p>
                <?php $findings = is_array($gate['findings'] ?? null) ? (array) $gate['findings'] : []; ?>
                <?php if ($findings === []) : ?>
                    <p><em>Keine Findings.</em></p>
                <?php else : ?>
                    <table class="widefat striped">
                        <thead>
                            <tr>
                                <th>Severity</th>
                                <th>Check</th>
                                <th>Message</th>
                                <th>Value</th>
                                <th>Aktion</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($findings as $finding) : ?>
                                <?php $action = is_array($finding['action'] ?? null) ? $finding['action'] : []; ?>
                                <tr>
                                    <td><?php echo esc_html((string) ($finding['severity'] ?? '')); ?></td>
                                    <td><?php echo esc_html((string) ($finding['label'] ?? '')); ?></td>
                                    <td><?php echo esc_html((string) ($finding['message'] ?? '')); ?></td>
                                    <td><?php echo esc_html((string) ($finding['value'] ?? '')); ?></td>
                                    <td>
                                        <?php if ((string) ($action['url'] ?? '') !== '') : ?>
                                            <a href="<?php echo esc_url((string) $action['url']); ?>"><?php echo esc_html((string) ($action['label'] ?? 'Open')); ?></a>
                                        <?php else : ?>
                                            -
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            <?php endforeach; ?>

            <h2>Fortschritt</h2>
            <table class="widefat striped">
                <thead>
                    <tr>
                        <th>Schritt</th>
                        <th>Status</th>
                        <th>Was jetzt zu tun ist</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($gateOrder as $gateKey) : ?>
                        <?php
                        $state = (string) ($run['gates'][$gateKey]['status'] ?? 'pending');
                        $todo = 'Warten auf vorherige Schritte.';
                        if ($gateKey === 'user_input_v1') {
                            if ($canCollectInput && ! $userInputDone) {
                                $todo = 'Nutzerangaben ausfuellen und speichern.';
                            } elseif ($userInputDone) {
                                $todo = 'Erledigt.';
                            }
                        } else {
                            if ($state === 'blocked') {
                                $todo = 'Blocker beheben und erneut pruefen.';
                            } elseif ($state === 'warning') {
                                $todo = 'Kein Stopper. Empfehlung prüfen und bei Bedarf später verbessern.';
                            } elseif ($state === 'passed') {
                                $todo = 'Erledigt.';
                            } elseif ((string) ($run['status'] ?? 'idle') === 'idle') {
                                $todo = 'Gate-Pruefung starten.';
                            } else {
                                $todo = 'Gate pruefen.';
                            }
                        }
                        ?>
                        <tr>
                            <td><?php echo esc_html((string) ($gateLabels[$gateKey] ?? $gateKey)); ?></td>
                            <td><?php echo esc_html($state); ?></td>
                            <td><?php echo esc_html($todo); ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php
    }

    private function openRouterConnectionStatus(): array
    {
        $cfg = $this->openRouter->config();
        $apiKey = (string) ($cfg['api_key'] ?? '');

        if ($apiKey === '') {
            return ['state' => 'no_key', 'label' => 'Kein API-Key', 'details' => 'Es ist kein OpenRouter-API-Key hinterlegt.'];
        }

        if (! $this->openRouter->isEnabled()) {
            return ['state' => 'disabled', 'label' => 'Deaktiviert', 'details' => 'Die OpenRouter-Integration ist in den Einstellungen deaktiviert.'];
        }

        $result = $this->openRouter->models();
        if ($result instanceof \WP_Error) {
            $message = $result->get_error_message();
            if ($message === '') {
                $message = 'Der Verbindungstest ist fehlgeschlagen.';
            }

            return ['state' => 'disconnected', 'label' => 'Nicht verbunden', 'details' => $message];
        }

        $normalizedModels = $this->normalizeModels($result);
        $filteredModels = $this->applyConfiguredModelFilter($normalizedModels);
        $totalModels = $this->extractModelsTotal($result, count($normalizedModels));
        $filterInfo = $this->buildModelFilterInfo(count($normalizedModels), count($filteredModels));

        return [
            'state' => 'connected',
            'label' => 'Verbunden',
            'details' => 'Verbindungstest zu OpenRouter erfolgreich.',
            'models' => array_slice($filteredModels, 0, 100),
            'models_total' => $totalModels,
            'model_filter_info' => $filterInfo,
        ];
    }

    private function planningSeoInstallButtons(): array
    {
        $candidates = [
            [
                'slug' => 'wpzeo-zero-seo',
                'label' => 'wpZeo installieren (empfohlen)',
                'search' => 'wpzeo',
                'recommended' => true,
                'auto_install' => true,
            ],
            [
                'slug' => 'wordpress-seo',
                'label' => 'Yoast installieren',
                'search' => 'wordpress-seo',
                'recommended' => false,
                'auto_install' => false,
            ],
            [
                'slug' => 'seo-by-rank-math',
                'label' => 'Rank Math installieren',
                'search' => 'rank-math',
                'recommended' => false,
                'auto_install' => false,
            ],
        ];

        $buttons = [];
        $canInstall = current_user_can('install_plugins');

        foreach ($candidates as $plugin) {
            $slug = (string) ($plugin['slug'] ?? '');
            if ($slug === '') {
                continue;
            }

            $url = admin_url('plugin-install.php?s=' . rawurlencode((string) ($plugin['search'] ?? $slug)) . '&tab=search&type=term');
            $type = 'link';
            $action = '';
            if ($canInstall) {
                if ((bool) ($plugin['auto_install'] ?? false)) {
                    $type = 'post';
                    $action = 'install_wpzeo_auto';
                } else {
                    $installUrl = admin_url('update.php?action=install-plugin&plugin=' . rawurlencode($slug));
                    if (function_exists('wp_nonce_url')) {
                        $installUrl = wp_nonce_url($installUrl, 'install-plugin_' . $slug);
                    }
                    $url = $installUrl;
                }
            }

            $buttons[] = [
                'label' => (string) ($plugin['label'] ?? 'Plugin installieren'),
                'url' => $url,
                'recommended' => (bool) ($plugin['recommended'] ?? false),
                'type' => $type,
                'action' => $action,
            ];
        }

        return $buttons;
    }

    private function planningErrorHeading(string $code, string $label): string
    {
        return match ($code) {
            'dependency_missing' => 'Pflicht-Plugin fehlt',
            'dependency_inactive' => 'Pflicht-Plugin ist deaktiviert',
            'dependency_version_low' => 'Plugin-Version zu alt',
            'seo_plugin_missing' => 'SEO-Plugin fehlt',
            'seo_multiple_installed' => 'Mehrere SEO-Plugins installiert',
            'seo_multiple_active' => 'Mehrere SEO-Plugins aktiv',
            'blogname_missing' => 'Website-Name fehlt',
            'blogname_generic' => 'Website-Name zu allgemein',
            'tagline_missing' => 'Tagline fehlt',
            'tagline_generic' => 'Tagline zu allgemein',
            'permalinks_plain' => 'Permalink-Struktur ungeeignet',
            'url_mismatch' => 'WordPress-URL-Konflikt',
            'rest_api_unavailable' => 'REST API nicht erreichbar',
            'openrouter_disabled' => 'OpenRouter deaktiviert',
            'openrouter_missing_key' => 'OpenRouter API-Key fehlt',
            'openrouter_connectivity_failed' => 'OpenRouter-Verbindung fehlgeschlagen',
            'openrouter_rate_limited' => 'OpenRouter temporär limitiert',
            default => ($label !== '' ? $label : 'Planungs-Blocker erkannt'),
        };
    }

    private function planningFieldHintClass(array $row): string
    {
        $status = (string) ($row['status'] ?? 'ok');
        return match ($status) {
            'blocker' => 'agent-field-hint--blocker',
            'warning' => 'agent-field-hint--warning',
            'ok' => 'agent-field-hint--ok',
            default => 'agent-field-hint--ok',
        };
    }

    private function planningFieldHintText(array $row): string
    {
        $status = (string) ($row['status'] ?? 'ok');
        $message = trim((string) ($row['message'] ?? ''));

        if ($status === 'ok') {
            return 'Korrekt eingerichtet';
        }

        if ($status === 'warning') {
            return $message !== '' ? ('Hinweis: ' . $message) : 'Hinweis';
        }

        if ($status === 'blocker') {
            return $message !== '' ? ('Fehler: ' . $message) : 'Fehler';
        }

        return $message !== '' ? $message : 'Korrekt eingerichtet';
    }

    private function sanitizePermalinkStructure(string $raw): string
    {
        $value = trim($raw);
        if ($value === '') {
            return '';
        }

        if (! str_starts_with($value, '/')) {
            $value = '/' . $value;
        }
        if (! str_ends_with($value, '/')) {
            $value .= '/';
        }

        return $value;
    }

    private function planningPermalinkStrategyLabel(string $strategy): string
    {
        return match ($strategy) {
            'standard' => 'Standard (ohne Kategorie)',
            'cluster' => 'Themen-Cluster (mit Kategorie)',
            default => 'Nicht festgelegt',
        };
    }

    private function planningCoreRows(array $run): array
    {
        $coreGate = is_array($run['gates']['core_baseline'] ?? null) ? (array) $run['gates']['core_baseline'] : [];
        $findings = is_array($coreGate['findings'] ?? null) ? (array) $coreGate['findings'] : [];

        $siteUrl = (string) get_option('siteurl', '');
        $homeUrl = (string) get_option('home', '');
        $privacyId = (int) get_option('wp_page_for_privacy_policy', 0);
        $privacyStatus = $privacyId > 0 && function_exists('get_post_status') ? (string) get_post_status($privacyId) : '';
        $privacyValue = $privacyId > 0 ? ('ID ' . (string) $privacyId . ' (' . ($privacyStatus !== '' ? $privacyStatus : 'unbekannt') . ')') : 'Nicht gesetzt';

        $rows = [
            [
                'label' => 'Website-Name',
                'value' => (string) get_option('blogname', ''),
                'blocker_codes' => ['blogname_missing'],
                'warning_codes' => ['blogname_generic'],
                'action_url' => admin_url('options-general.php'),
                'action_label' => 'Einstellungen > Allgemein oeffnen',
            ],
            [
                'label' => 'Tagline',
                'value' => (string) get_option('blogdescription', ''),
                'blocker_codes' => ['tagline_missing'],
                'warning_codes' => ['tagline_generic'],
                'action_url' => admin_url('options-general.php'),
                'action_label' => 'Einstellungen > Allgemein oeffnen',
            ],
            [
                'label' => 'Permalink-Struktur',
                'value' => (string) get_option('permalink_structure', ''),
                'blocker_codes' => ['permalinks_plain'],
                'warning_codes' => [],
                'action_url' => admin_url('options-permalink.php'),
                'action_label' => 'Permalinks öffnen',
            ],
            [
                'label' => 'Site/Home URL',
                'value' => $siteUrl . ' | ' . $homeUrl,
                'blocker_codes' => ['url_mismatch'],
                'warning_codes' => [],
                'action_url' => admin_url('options-general.php'),
                'action_label' => 'Einstellungen > Allgemein oeffnen',
            ],
            [
                'label' => 'Sprache',
                'value' => function_exists('get_locale') ? (string) get_locale() : '',
                'blocker_codes' => [],
                'warning_codes' => ['locale_missing'],
                'action_url' => admin_url('options-general.php'),
                'action_label' => 'Allgemein öffnen',
            ],
            [
                'label' => 'Zeitzone',
                'value' => (string) get_option('timezone_string', ''),
                'blocker_codes' => [],
                'warning_codes' => ['timezone_missing'],
                'action_url' => admin_url('options-general.php'),
                'action_label' => 'Allgemein öffnen',
            ],
            [
                'label' => 'Datenschutzseite',
                'value' => $privacyValue,
                'blocker_codes' => [],
                'warning_codes' => ['privacy_policy_missing'],
                'action_url' => admin_url('options-privacy.php'),
                'action_label' => 'Datenschutz öffnen',
            ],
            [
                'label' => 'REST API',
                'value' => function_exists('rest_url') ? (string) rest_url() : 'Nicht verfügbar',
                'blocker_codes' => ['rest_api_unavailable'],
                'warning_codes' => [],
                'action_url' => '',
                'action_label' => '',
            ],
        ];

        $resolved = [];
        foreach ($rows as $row) {
            $status = 'ok';
            $message = 'OK';
            $actionUrl = (string) ($row['action_url'] ?? '');
            $actionLabel = (string) ($row['action_label'] ?? '');

            $blockerFinding = $this->findPlanningFindingByCodes($findings, (array) ($row['blocker_codes'] ?? []));
            $warningFinding = $this->findPlanningFindingByCodes($findings, (array) ($row['warning_codes'] ?? []));

            if ($blockerFinding !== null) {
                $status = 'blocker';
                $message = (string) ($blockerFinding['message'] ?? 'Blocker erkannt');
                $action = is_array($blockerFinding['action'] ?? null) ? (array) $blockerFinding['action'] : [];
                if ((string) ($action['url'] ?? '') !== '') {
                    $actionUrl = (string) ($action['url'] ?? '');
                    $actionLabel = (string) ($action['label'] ?? $actionLabel);
                }
            } elseif ($warningFinding !== null) {
                $status = 'warning';
                $message = (string) ($warningFinding['message'] ?? 'Hinweis');
                $action = is_array($warningFinding['action'] ?? null) ? (array) $warningFinding['action'] : [];
                if ((string) ($action['url'] ?? '') !== '') {
                    $actionUrl = (string) ($action['url'] ?? '');
                    $actionLabel = (string) ($action['label'] ?? $actionLabel);
                }
            }

            $value = (string) ($row['value'] ?? '');
            if ($value === '') {
                $value = 'Nicht gesetzt';
            }

            $resolved[] = [
                'label' => (string) ($row['label'] ?? ''),
                'value' => $value,
                'status' => $status,
                'message' => $message,
                'action_url' => $actionUrl,
                'action_label' => $actionLabel,
            ];
        }

        return $resolved;
    }

    private function findPlanningFindingByCodes(array $findings, array $codes): ?array
    {
        if ($codes === []) {
            return null;
        }

        foreach ($findings as $finding) {
            if (! is_array($finding)) {
                continue;
            }
            $code = (string) ($finding['code'] ?? '');
            if ($code !== '' && in_array($code, $codes, true)) {
                return $finding;
            }
        }

        return null;
    }

    private function planningSeoConflictButtons(): array
    {
        return [
            [
                'label' => 'Nur wpZeo aktivieren (empfohlen)',
                'action' => 'fix_seo_single_wpzeo',
                'recommended' => true,
            ],
            [
                'label' => 'Nur Yoast aktivieren',
                'action' => 'fix_seo_single_yoast',
                'recommended' => false,
            ],
            [
                'label' => 'Nur Rank Math aktivieren',
                'action' => 'fix_seo_single_rankmath',
                'recommended' => false,
            ],
        ];
    }

    private function enforceSingleSeoPlugin(string $target): array|\WP_Error
    {
        if (! current_user_can('activate_plugins')) {
            return new \WP_Error('agent_seo_conflict_forbidden', 'Du darfst Plugin-Status nicht aendern.', ['status' => 403]);
        }

        if (! function_exists('activate_plugin') || ! function_exists('deactivate_plugins') || ! function_exists('is_plugin_active')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        if (! function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        $allPlugins = get_plugins();
        if (! is_array($allPlugins) || $allPlugins === []) {
            return new \WP_Error('agent_seo_conflict_no_plugins', 'Es konnten keine Plugins gelesen werden.', ['status' => 500]);
        }

        $targets = [
            'wpzeo' => ['wpzeo-zero-seo/wpzeo-zero-seo.php'],
            'yoast' => ['wordpress-seo/wp-seo.php'],
            'rankmath' => ['seo-by-rank-math/rank-math.php', 'rank-math/rank-math.php'],
        ];
        $candidateFiles = $targets[$target] ?? [];
        $targetFile = '';
        foreach ($candidateFiles as $candidate) {
            if (isset($allPlugins[$candidate])) {
                $targetFile = $candidate;
                break;
            }
        }
        if ($targetFile === '') {
            return new \WP_Error('agent_seo_target_not_installed', 'Das Ziel-SEO-Plugin ist nicht installiert.', ['status' => 400, 'target' => $target]);
        }

        $seoPlugins = $this->activeSeoPluginsForAdmin();
        $toDeactivate = [];
        foreach ($seoPlugins as $plugin) {
            $file = (string) ($plugin['plugin'] ?? '');
            if ($file === '' || $file === $targetFile) {
                continue;
            }
            if (is_plugin_active($file)) {
                $toDeactivate[] = $file;
            }
        }

        if ($toDeactivate !== []) {
            deactivate_plugins($toDeactivate, false, false);
        }

        if (! is_plugin_active($targetFile)) {
            $activation = activate_plugin($targetFile);
            if ($activation instanceof \WP_Error) {
                return new \WP_Error('agent_seo_target_activate_failed', 'Ziel-SEO-Plugin konnte nicht aktiviert werden: ' . $activation->get_error_message(), ['status' => 500]);
            }
        }

        update_option(\Agent\Modules\Planning\PlanningService::OPTION_SELECTED_SEO_PLUGIN, $targetFile, false);

        return [
            'active_plugin' => $targetFile,
            'active_label' => (string) ($allPlugins[$targetFile]['Name'] ?? $targetFile),
            'deactivated_plugins' => $toDeactivate,
        ];
    }

    private function activeSeoPluginsForAdmin(): array
    {
        if (! function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        if (! function_exists('is_plugin_active')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        $plugins = get_plugins();
        if (! is_array($plugins) || $plugins === []) {
            return [];
        }

        $matches = [];
        foreach ($plugins as $file => $meta) {
            $candidate = strtolower((string) $file);
            if (! (str_contains($candidate, 'wordpress-seo')
                || str_contains($candidate, 'wp-seo')
                || str_contains($candidate, 'rank-math')
                || str_contains($candidate, 'wpzeo'))
            ) {
                continue;
            }
            if (! is_plugin_active((string) $file)) {
                continue;
            }
            $matches[] = [
                'plugin' => (string) $file,
                'label' => (string) ($meta['Name'] ?? (string) $file),
            ];
        }

        return $matches;
    }

    private function normalizeModels(array $payload): array
    {
        $rawModels = [];
        if (isset($payload['data']) && is_array($payload['data'])) {
            $rawModels = $payload['data'];
        } elseif (array_is_list($payload)) {
            $rawModels = $payload;
        }

        $models = [];
        foreach ($rawModels as $entry) {
            if (! is_array($entry)) {
                continue;
            }

            $id = (string) ($entry['id'] ?? '');
            if ($id === '') {
                continue;
            }

            $name = (string) ($entry['name'] ?? $id);
            $contextLength = '';
            if (isset($entry['context_length'])) {
                $contextLength = (string) (int) $entry['context_length'];
            } elseif (isset($entry['top_provider']['context_length'])) {
                $contextLength = (string) (int) $entry['top_provider']['context_length'];
            }

            $models[] = [
                'id' => $id,
                'name' => $name,
                'context_length' => $contextLength,
                'pricing' => $this->formatPricing($entry),
            ];
        }

        return $models;
    }

    private function extractModelsTotal(array $payload, int $fallback): int
    {
        if (isset($payload['meta']['total']) && is_numeric($payload['meta']['total'])) {
            return max(0, (int) $payload['meta']['total']);
        }

        return $fallback;
    }

    private function formatPricing(array $entry): string
    {
        $pricing = $entry['pricing'] ?? null;
        if (! is_array($pricing)) {
            return '';
        }

        $prompt = isset($pricing['prompt']) ? (string) $pricing['prompt'] : '';
        $completion = isset($pricing['completion']) ? (string) $pricing['completion'] : '';
        $promptValue = $prompt === '' ? '-' : $this->formatPriceValue($prompt);
        $completionValue = $completion === '' ? '-' : $this->formatPriceValue($completion);
        return $promptValue . ' | ' . $completionValue;
    }

    private function formatPriceValue(string $value): string
    {
        if (! is_numeric($value)) {
            return $value;
        }

        $number = (float) $value;
        if ($number === 0.0) {
            return '0 mct';
        }

        // 1 mct = 0.001 cent = $0.00001, 1 µct = $0.00000001
        $mct = $number * 100000.0;
        $uct = $number * 100000000.0;

        if (abs($mct) >= 1.0) {
            return $this->trimDecimal($mct, 3) . ' mct';
        }

        return $this->trimDecimal($uct, 2) . ' µct';
    }

    private function trimDecimal(float $number, int $precision): string
    {
        $formatted = number_format($number, $precision, '.', '');
        $formatted = rtrim($formatted, '0');
        $formatted = rtrim($formatted, '.');
        return $formatted === '' ? '0' : $formatted;
    }

    private function applyConfiguredModelFilter(array $models): array
    {
        $cfg = $this->openRouter->config();
        if (! (bool) ($cfg['model_filter_enabled'] ?? true)) {
            return $models;
        }

        $terms = $this->parseModelFilterTerms((string) ($cfg['model_filter'] ?? ''));
        if ($terms === []) {
            return $models;
        }

        $filtered = [];
        foreach ($models as $model) {
            $haystack = strtolower((string) ($model['id'] ?? '') . ' ' . (string) ($model['name'] ?? ''));
            foreach ($terms as $term) {
                if (str_contains($haystack, $term)) {
                    $filtered[] = $model;
                    break;
                }
            }
        }

        return $filtered;
    }

    private function parseModelFilterTerms(string $raw): array
    {
        $parts = explode('|', strtolower($raw));
        $terms = [];
        foreach ($parts as $part) {
            $term = trim($part);
            if ($term === '') {
                continue;
            }
            $terms[] = $term;
        }

        return array_values(array_unique($terms));
    }

    private function buildModelFilterInfo(int $total, int $filtered): string
    {
        $cfg = $this->openRouter->config();
        $enabled = (bool) ($cfg['model_filter_enabled'] ?? true);
        $rawFilter = (string) ($cfg['model_filter'] ?? '');

        if (! $enabled) {
            return 'Modellfilter ist deaktiviert.';
        }

        $terms = $this->parseModelFilterTerms($rawFilter);
        if ($terms === []) {
            return 'Modellfilter ist aktiv, aber leer.';
        }

        return 'Modellfilter aktiv: ' . implode(' | ', $terms) . ' (' . (string) $filtered . '/' . (string) $total . ' Modelle).';
    }

    private function pipelineStateRows(int $limit = 100): array
    {
        $limit = max(1, min(500, $limit));
        $state = get_option('agent_chunk_pipeline_state', []);
        if (! is_array($state)) {
            return [];
        }

        $rows = [];
        foreach ($state as $entry) {
            if (! is_array($entry)) {
                continue;
            }

            $steps = [];
            foreach ((array) ($entry['steps'] ?? []) as $stepName => $stepData) {
                $stepStatus = is_array($stepData) ? (string) ($stepData['status'] ?? 'unknown') : 'unknown';
                $steps[] = (string) $stepName . ':' . $stepStatus;
            }

            $rows[] = [
                'post_id' => (int) ($entry['post_id'] ?? 0),
                'status' => (string) ($entry['status'] ?? ''),
                'content_hash' => (string) ($entry['content_hash'] ?? ''),
                'updated_at' => (string) ($entry['updated_at'] ?? ''),
                'steps_label' => implode(' | ', $steps),
            ];
        }

        usort($rows, static function (array $a, array $b): int {
            return strcmp((string) ($b['updated_at'] ?? ''), (string) ($a['updated_at'] ?? ''));
        });

        return array_slice($rows, 0, $limit);
    }

    private function chunkRows(?int $postId = null, int $limit = 200): array
    {
        $limit = max(1, min(500, $limit));
        if ($this->canUseDbForChunkTables()) {
            global $wpdb;
            $table = $wpdb->prefix . 'ai_chunks';
            if ($postId !== null && $postId > 0) {
                $sql = $wpdb->prepare(
                    'SELECT post_id, chunk_no, text, embedding, updated_at FROM ' . $table . ' WHERE post_id = %d ORDER BY post_id DESC, chunk_no ASC LIMIT %d',
                    $postId,
                    $limit
                );
            } else {
                $sql = $wpdb->prepare(
                    'SELECT post_id, chunk_no, text, embedding, updated_at FROM ' . $table . ' ORDER BY post_id DESC, chunk_no ASC LIMIT %d',
                    $limit
                );
            }
            $rows = $wpdb->get_results($sql, ARRAY_A);
            if (! is_array($rows)) {
                return [];
            }

            $out = [];
            foreach ($rows as $row) {
                if (! is_array($row)) {
                    continue;
                }
                $text = trim((string) ($row['text'] ?? ''));
                $out[] = [
                    'post_id' => (int) ($row['post_id'] ?? 0),
                    'chunk_no' => (int) ($row['chunk_no'] ?? 0),
                    'text_preview' => function_exists('mb_substr') ? mb_substr($text, 0, 140) : substr($text, 0, 140),
                    'embedding_state' => $this->hasEmbedding((string) ($row['embedding'] ?? '')) ? 'present' : 'none',
                    'updated_at' => (string) ($row['updated_at'] ?? ''),
                ];
            }

            return $out;
        }

        $all = get_option('agent_chunk_pipeline_chunks', []);
        if (! is_array($all)) {
            return [];
        }

        $rows = [];
        foreach ($all as $pid => $chunks) {
            $pidInt = (int) $pid;
            if ($postId !== null && $postId > 0 && $pidInt !== $postId) {
                continue;
            }
            if (! is_array($chunks)) {
                continue;
            }

            foreach ($chunks as $chunk) {
                if (! is_array($chunk)) {
                    continue;
                }
                $text = trim((string) ($chunk['text'] ?? ''));
                $rows[] = [
                    'post_id' => $pidInt,
                    'chunk_no' => (int) ($chunk['chunk_no'] ?? 0),
                    'text_preview' => function_exists('mb_substr') ? mb_substr($text, 0, 140) : substr($text, 0, 140),
                    'embedding_state' => is_array($chunk['embedding'] ?? null) && $chunk['embedding'] !== [] ? 'present' : 'none',
                    'updated_at' => '',
                ];
            }
        }

        usort($rows, static function (array $a, array $b): int {
            if ((int) ($a['post_id'] ?? 0) === (int) ($b['post_id'] ?? 0)) {
                return ((int) ($a['chunk_no'] ?? 0)) <=> ((int) ($b['chunk_no'] ?? 0));
            }
            return ((int) ($b['post_id'] ?? 0)) <=> ((int) ($a['post_id'] ?? 0));
        });

        return array_slice($rows, 0, $limit);
    }

    private function chunkSummary(?int $postId = null): array
    {
        $rows = $this->chunkRows($postId, 500);
        $postSet = [];
        $withEmbeddings = 0;
        foreach ($rows as $row) {
            $postSet[(string) ((int) ($row['post_id'] ?? 0))] = true;
            if ((string) ($row['embedding_state'] ?? 'none') === 'present') {
                $withEmbeddings++;
            }
        }

        return [
            'chunks_total' => count($rows),
            'distinct_posts' => count($postSet),
            'with_embeddings' => $withEmbeddings,
        ];
    }

    private function hasEmbedding(string $raw): bool
    {
        $decoded = json_decode($raw, true);
        return is_array($decoded) && $decoded !== [];
    }

    private function canUseDbForChunkTables(): bool
    {
        global $wpdb;
        if (! isset($wpdb) || ! is_object($wpdb)) {
            return false;
        }
        if (! isset($wpdb->prefix) || ! is_string($wpdb->prefix) || $wpdb->prefix === '') {
            return false;
        }

        return method_exists($wpdb, 'get_results') && method_exists($wpdb, 'prepare');
    }

    private function jobsWorkerStatus(): array
    {
        $status = get_option(JobWorkerScheduler::OPTION_STATUS, []);
        if (! is_array($status)) {
            $status = [];
        }

        return [
            'last_tick_at' => (string) ($status['last_tick_at'] ?? ''),
            'state' => (string) ($status['state'] ?? 'unknown'),
            'reason' => (string) ($status['reason'] ?? ''),
            'processed' => (int) ($status['processed'] ?? 0),
            'completed' => (int) ($status['completed'] ?? 0),
            'failed' => (int) ($status['failed'] ?? 0),
            'max_per_tick' => (int) ($status['max_per_tick'] ?? 0),
            'error_code' => (string) ($status['error_code'] ?? ''),
            'error_message' => (string) ($status['error_message'] ?? ''),
        ];
    }
}
