<?php

declare(strict_types=1);

namespace Agent\Support;

final class Logger
{
    private string $optionName;

    public function __construct(string $optionName)
    {
        $this->optionName = $optionName;
    }

    public function log(string $action, string $severity = 'info', array $context = []): void
    {
        $logs = get_option($this->optionName, []);
        if (! is_array($logs)) {
            $logs = [];
        }

        $logs[] = [
            'time' => gmdate('c'),
            'action' => $action,
            'severity' => $severity,
            'user_id' => get_current_user_id(),
            'context' => $context,
        ];

        if (count($logs) > 200) {
            $logs = array_slice($logs, -200);
        }

        update_option($this->optionName, $logs, false);
    }

    public function latest(int $limit = 25): array
    {
        $logs = get_option($this->optionName, []);
        if (! is_array($logs)) {
            return [];
        }

        return array_reverse(array_slice($logs, -$limit));
    }

    public function query(int $limit = 50, ?string $severity = null, ?string $action = null): array
    {
        $logs = $this->latest($limit * 4);
        $filtered = [];

        foreach ($logs as $entry) {
            if (! is_array($entry)) {
                continue;
            }

            if ($severity !== null && (string) ($entry['severity'] ?? '') !== $severity) {
                continue;
            }

            if ($action !== null && (string) ($entry['action'] ?? '') !== $action) {
                continue;
            }

            $filtered[] = $entry;
            if (count($filtered) >= $limit) {
                break;
            }
        }

        return $filtered;
    }

    public function errors(int $limit = 50): array
    {
        $logs = $this->latest($limit * 4);
        $filtered = [];
        $errorSeverities = ['warning', 'error', 'critical'];

        foreach ($logs as $entry) {
            if (! is_array($entry)) {
                continue;
            }

            $severity = (string) ($entry['severity'] ?? '');
            if (! in_array($severity, $errorSeverities, true)) {
                continue;
            }

            $filtered[] = $entry;
            if (count($filtered) >= $limit) {
                break;
            }
        }

        return $filtered;
    }
}
