<?php

declare(strict_types=1);

namespace Agent\Security;

use Agent\Support\Logger;
use Agent\Support\Options;
use WP_Error;

final class RequestGuard
{
    private Options $options;
    private Logger $logger;

    public function __construct(Options $options, Logger $logger)
    {
        $this->options = $options;
        $this->logger = $logger;
    }

    public function checkReadAccess(): bool
    {
        if (! is_user_logged_in()) {
            return false;
        }

        if ($this->isBasicAuthenticatedRequest()) {
            return true;
        }

        if ($this->isEmergencyLocked()) {
            return false;
        }

        return current_user_can('manage_options') || current_user_can('agent_read');
    }

    public function checkWriteAccess(): bool
    {
        if ($this->isEmergencyLocked()) {
            return false;
        }

        if ($this->options->get('full_write_access', false) !== true) {
            return false;
        }

        return current_user_can('manage_options');
    }

    public function validateStateChangeNonce(?string $nonce): bool
    {
        if (! is_string($nonce) || $nonce === '') {
            return false;
        }

        return wp_verify_nonce($nonce, 'wp_rest') === 1;
    }

    public function isEmergencyLocked(): bool
    {
        return (bool) $this->options->get('emergency_lock', false);
    }

    public function isBasicAuthenticatedRequest(): bool
    {
        if (! is_user_logged_in()) {
            return false;
        }

        $authHeader = (string) ($_SERVER['HTTP_AUTHORIZATION'] ?? '');
        if (stripos($authHeader, 'basic ') === 0) {
            return true;
        }

        return isset($_SERVER['PHP_AUTH_USER']) && (string) $_SERVER['PHP_AUTH_USER'] !== '';
    }

    public function assertRateLimit(string $actionKey): bool|WP_Error
    {
        $limit = (int) $this->options->get('rate_limit_per_minute', 60);
        if ($limit < 1) {
            $limit = 1;
        }

        $actor = (string) get_current_user_id();
        if ($actor === '0') {
            $actor = sanitize_text_field((string) ($_SERVER['REMOTE_ADDR'] ?? 'unknown'));
        }

        $bucketKey = sprintf('agent_rl_%s_%s', md5($actionKey), md5($actor));
        $hits = (int) get_transient($bucketKey);

        if ($hits >= $limit) {
            $this->logger->log('rate_limit_blocked', 'warning', ['action' => $actionKey, 'actor' => $actor]);
            return new WP_Error('agent_rate_limited', 'Rate limit exceeded.', ['status' => 429]);
        }

        set_transient($bucketKey, $hits + 1, MINUTE_IN_SECONDS);

        return true;
    }
}
