<?php

declare(strict_types=1);

namespace Agent;

use Agent\Admin\AdminPage;
use Agent\Modules\CoreAnalyzer\CoreAnalyzerController;
use Agent\Modules\CoreAnalyzer\CoreAnalyzerService;
use Agent\Modules\Editor\EditorSidebarModule;
use Agent\Modules\Indexing\ChunkPipelineService;
use Agent\Modules\Indexing\ContentChangeIndexer;
use Agent\Modules\Indexing\IndexBackfillScheduler;
use Agent\Modules\Indexing\IndexTablesInstaller;
use Agent\Modules\Indexing\IndexingController;
use Agent\Modules\Jobs\JobsController;
use Agent\Modules\Jobs\JobExecutionService;
use Agent\Modules\Jobs\JobQueueService;
use Agent\Modules\Jobs\JobWorkerScheduler;
use Agent\Modules\Planning\PlanningController;
use Agent\Modules\Planning\PlanningService;
use Agent\Modules\Planning\WpZeoInstaller;
use Agent\Modules\PromptRecipe\PromptRecipeModule;
use Agent\Modules\System\CapabilityMapService;
use Agent\Modules\System\ChangesetService;
use Agent\Modules\OpenRouter\OpenRouterController;
use Agent\Modules\OpenRouter\OpenRouterService;
use Agent\Modules\Seo\SeoMetaOrchestrator;
use Agent\Modules\StyleGuide\StyleGuideModule;
use Agent\Modules\System\SnapshotService;
use Agent\Modules\System\SystemController;
use Agent\Modules\System\UpdateService;
use Agent\Security\RequestGuard;
use Agent\Support\Logger;
use Agent\Support\Options;

final class Plugin
{
    public const OPTION_SETTINGS = 'agent_settings';
    public const OPTION_LOGS = 'agent_logs';
    public const OPTION_ROLLBACKS = 'agent_rollbacks';
    public const OPTION_RESTORE_POINTS = 'agent_restore_points';
    public const OPTION_SETUP_STATUS = 'agent_setup_status';
    public const OPTION_SCHEMA_VERSION = 'agent_schema_version';
    public const SCHEMA_VERSION = '2026-02-21-2';

    private string $pluginFile;

    public function __construct(string $pluginFile)
    {
        $this->pluginFile = $pluginFile;
    }

    public function boot(): void
    {
        $options = new Options(self::OPTION_SETTINGS, self::defaultSettings());
        $logger = new Logger(self::OPTION_LOGS);
        $guard = new RequestGuard($options, $logger);
        $analyzer = new CoreAnalyzerService();
        $capabilityMap = new CapabilityMapService($options);
        $changeset = new ChangesetService($options);
        $snapshot = new SnapshotService();
        $updates = new UpdateService();
        $openRouter = new OpenRouterService($options);
        $planning = new PlanningService($options, $openRouter, $logger);
        $wpZeoInstaller = new WpZeoInstaller();
        $chunkPipeline = new ChunkPipelineService($openRouter, $logger);
        $jobs = new JobQueueService();
        $contentIndexer = new ContentChangeIndexer($jobs, $options, $logger);
        $indexBackfill = new IndexBackfillScheduler($jobs, $options, $logger);
        $seoOrchestrator = new SeoMetaOrchestrator();
        $jobExecutor = new JobExecutionService($openRouter, $seoOrchestrator, $chunkPipeline, $options, $logger);
        $jobsWorker = new JobWorkerScheduler($jobs, $jobExecutor, $options, $logger);
        $styleGuide = new StyleGuideModule();
        $promptRecipe = new PromptRecipeModule();
        $editorSidebar = new EditorSidebarModule($this->pluginFile);

        $coreController = new CoreAnalyzerController($analyzer, $guard, $logger, $options);
        $systemController = new SystemController($capabilityMap, $changeset, $snapshot, $updates, $guard, $logger, $options);
        $openRouterController = new OpenRouterController($openRouter, $seoOrchestrator, $guard, $logger, $options);
        $jobsController = new JobsController($jobs, $jobExecutor, $guard, $logger);
        $indexingController = new IndexingController($chunkPipeline, $guard, $logger);
        $planningController = new PlanningController($planning, $guard, $logger);
        add_action('rest_api_init', [$coreController, 'registerRoutes']);
        add_action('rest_api_init', [$systemController, 'registerRoutes']);
        add_action('rest_api_init', [$openRouterController, 'registerRoutes']);
        add_action('rest_api_init', [$jobsController, 'registerRoutes']);
        add_action('rest_api_init', [$indexingController, 'registerRoutes']);
        add_action('rest_api_init', [$planningController, 'registerRoutes']);
        add_action('save_post', [$contentIndexer, 'onSavePost'], 20, 3);
        $indexBackfill->register();
        $jobsWorker->register();
        add_action('init', [$styleGuide, 'register']);
        add_action('init', [$promptRecipe, 'register']);
        $editorSidebar->register();

        if (is_admin()) {
            $this->ensureSchema();
            $this->runAutomatedSetupMaintenance();
            $adminPage = new AdminPage($analyzer, $guard, $logger, $options, $openRouter, $planning, $wpZeoInstaller);
            add_action('admin_menu', [$adminPage, 'registerPage']);
            add_action('admin_init', [$adminPage, 'registerSettings']);
            add_action('admin_notices', [$adminPage, 'renderPlanningNotice']);
        }
    }

    public static function activate(): void
    {
        if (class_exists(IndexTablesInstaller::class)) {
            IndexTablesInstaller::install();
        }

        if (class_exists(StyleGuideModule::class)) {
            StyleGuideModule::seedDefaultRulePosts();
        }
        if (class_exists(PromptRecipeModule::class)) {
            PromptRecipeModule::seedDefaultRecipePosts();
        }

        if (class_exists(IndexBackfillScheduler::class) && function_exists('wp_next_scheduled') && function_exists('wp_schedule_event')) {
            if (wp_next_scheduled(IndexBackfillScheduler::CRON_HOOK) === false) {
                wp_schedule_event(time() + 60, IndexBackfillScheduler::SCHEDULE_KEY, IndexBackfillScheduler::CRON_HOOK);
            }
        }
        if (class_exists(JobWorkerScheduler::class) && function_exists('wp_next_scheduled') && function_exists('wp_schedule_event')) {
            if (wp_next_scheduled(JobWorkerScheduler::CRON_HOOK) === false) {
                wp_schedule_event(time() + 30, JobWorkerScheduler::SCHEDULE_KEY, JobWorkerScheduler::CRON_HOOK);
            }
        }

        update_option(self::OPTION_SCHEMA_VERSION, self::SCHEMA_VERSION, false);

        $role = get_role('agent_operator');
        if ($role === null) {
            add_role(
                'agent_operator',
                'Agent Operator',
                [
                    'read' => true,
                    'agent_read' => true,
                ]
            );
        } elseif (! $role->has_cap('agent_read')) {
            $role->add_cap('agent_read');
        }

        if (get_option(self::OPTION_SETTINGS) === false) {
            add_option(self::OPTION_SETTINGS, self::defaultSettings(), '', false);
        }

        if (get_option(self::OPTION_LOGS) === false) {
            add_option(self::OPTION_LOGS, [], '', false);
        }

        if (get_option(self::OPTION_ROLLBACKS) === false) {
            add_option(self::OPTION_ROLLBACKS, [], '', false);
        }

        if (get_option(self::OPTION_RESTORE_POINTS) === false) {
            add_option(self::OPTION_RESTORE_POINTS, [], '', false);
        }
    }

    public static function deactivate(): void
    {
        if (class_exists(IndexBackfillScheduler::class)) {
            IndexBackfillScheduler::unschedule();
        }
        if (class_exists(JobWorkerScheduler::class)) {
            JobWorkerScheduler::unschedule();
        }
        // Keep data for diagnostics after deactivation.
    }

    public static function defaultSettings(): array
    {
        return [
            'modules' => [
                'core_analyzer' => true,
                'content' => false,
                'system' => false,
                'security' => false,
                'performance' => false,
            ],
            'readonly_mode' => true,
            'full_write_access' => false,
            'emergency_lock' => false,
            'rate_limit_per_minute' => 60,
            'api_key' => '',
            'openrouter' => [
                'enabled' => false,
                'api_key' => '',
                'base_url' => 'https://openrouter.ai/api/v1',
                'site_url' => '',
                'app_name' => 'Agent Bridge',
                'timeout' => 60,
                'planner_model' => 'openrouter/free',
                'writer_model' => 'openrouter/free',
                'image_model' => 'openrouter/free',
                'model_filter_enabled' => true,
                'model_filter' => 'openai|x-ai|google|anthropic|meta-llama|free',
            ],
            'indexing' => [
                'backfill_limit' => 10,
                'worker_max_per_tick' => 5,
            ],
            'planning' => [
                'permalink_strategy' => '',
                'cluster_enabled' => null,
            ],
        ];
    }

    private function ensureSchema(): void
    {
        $current = (string) get_option(self::OPTION_SCHEMA_VERSION, '');
        if ($current === self::SCHEMA_VERSION) {
            return;
        }

        if (class_exists(IndexTablesInstaller::class)) {
            IndexTablesInstaller::install();
        }

        update_option(self::OPTION_SCHEMA_VERSION, self::SCHEMA_VERSION, false);
    }

    private function runAutomatedSetupMaintenance(): void
    {
        $state = get_option(self::OPTION_SETUP_STATUS, []);
        if (! is_array($state)) {
            $state = [];
        }

        $lastRunTs = isset($state['last_run']) ? strtotime((string) $state['last_run']) : false;
        $needsRun = (($state['schema_version'] ?? '') !== self::SCHEMA_VERSION)
            || ($lastRunTs === false)
            || (time() - (int) $lastRunTs >= 21600);

        if (! $needsRun) {
            return;
        }

        $steps = [];
        $this->runSetupStep($steps, 'settings', 'Einstellungen', function (): string {
            $existing = get_option(self::OPTION_SETTINGS, null);
            if (! is_array($existing)) {
                add_option(self::OPTION_SETTINGS, self::defaultSettings(), '', false);
                return 'created';
            }

            $merged = array_replace_recursive(self::defaultSettings(), $existing);
            if ($merged === $existing) {
                return 'ok';
            }

            update_option(self::OPTION_SETTINGS, $merged, false);
            return 'updated';
        });

        $this->runSetupStep($steps, 'logs', 'Logspeicher', function (): string {
            if (get_option(self::OPTION_LOGS) !== false) {
                return 'ok';
            }

            add_option(self::OPTION_LOGS, [], '', false);
            return 'created';
        });

        $this->runSetupStep($steps, 'rollback', 'Rollbackspeicher', function (): string {
            if (get_option(self::OPTION_ROLLBACKS) !== false) {
                return 'ok';
            }

            add_option(self::OPTION_ROLLBACKS, [], '', false);
            return 'created';
        });

        $this->runSetupStep($steps, 'restore_points', 'Restore-Points', function (): string {
            if (get_option(self::OPTION_RESTORE_POINTS) !== false) {
                return 'ok';
            }

            add_option(self::OPTION_RESTORE_POINTS, [], '', false);
            return 'created';
        });

        $this->runSetupStep($steps, 'index_tables', 'Index-Tabellen', function (): string {
            if (class_exists(IndexTablesInstaller::class)) {
                IndexTablesInstaller::install();
                return 'ok';
            }

            return 'skipped';
        });

        $this->runSetupStep($steps, 'operator_role', 'Rolle agent_operator', function (): string {
            $role = get_role('agent_operator');
            if ($role === null) {
                add_role(
                    'agent_operator',
                    'Agent Operator',
                    [
                        'read' => true,
                        'agent_read' => true,
                    ]
                );
                return 'created';
            }

            if (! $role->has_cap('agent_read')) {
                $role->add_cap('agent_read');
                return 'updated';
            }

            return 'ok';
        });

        $this->runSetupStep($steps, 'style_guides', 'Default Style Guides', function (): string {
            if (! class_exists(StyleGuideModule::class)) {
                return 'skipped';
            }

            StyleGuideModule::seedDefaultRulePosts();
            return 'ok';
        });

        $this->runSetupStep($steps, 'prompt_recipes', 'Default Prompt Recipes', function (): string {
            if (! class_exists(PromptRecipeModule::class)) {
                return 'skipped';
            }

            PromptRecipeModule::seedDefaultRecipePosts();
            return 'ok';
        });

        update_option(
            self::OPTION_SETUP_STATUS,
            [
                'automated' => true,
                'schema_version' => self::SCHEMA_VERSION,
                'last_run' => gmdate('c'),
                'steps' => $steps,
            ],
            false
        );
    }

    private function runSetupStep(array &$steps, string $key, string $label, callable $callback): void
    {
        try {
            $result = (string) $callback();
            $normalized = $result === '' ? 'ok' : $result;
            $steps[] = [
                'key' => $key,
                'label' => $label,
                'status' => $normalized,
                'message' => '',
            ];
        } catch (\Throwable $e) {
            $steps[] = [
                'key' => $key,
                'label' => $label,
                'status' => 'failed',
                'message' => $e->getMessage(),
            ];
        }
    }
}
