<?php

declare(strict_types=1);

namespace Agent\Modules\System;

use WP_Error;

final class UpdateService
{
    public function preflight(array $targets = []): array
    {
        $environment = $this->environment();

        $updates = [
            'core' => $this->coreUpdateCandidate(),
            'plugins' => $this->pluginUpdateCandidates($targets['plugins'] ?? null),
            'themes' => $this->themeUpdateCandidates($targets['themes'] ?? null),
        ];

        $compatibility = [
            'core' => $this->evaluateCoreCompatibility($updates['core'], $environment, (bool) ($targets['core'] ?? false)),
            'plugins' => $this->evaluateComponentCompatibility($updates['plugins'], $environment),
            'themes' => $this->evaluateComponentCompatibility($updates['themes'], $environment),
        ];

        return [
            'generated_at' => gmdate('c'),
            'environment' => $environment,
            'requested' => [
                'core' => (bool) ($targets['core'] ?? false),
                'plugins' => array_values(is_array($targets['plugins'] ?? null) ? $targets['plugins'] : []),
                'themes' => array_values(is_array($targets['themes'] ?? null) ? $targets['themes'] : []),
            ],
            'available_updates' => $updates,
            'compatibility' => $compatibility,
            'summary' => [
                'core_installable' => (bool) ($compatibility['core']['installable'] ?? false),
                'plugin_installable_count' => $this->countInstallable($compatibility['plugins']),
                'theme_installable_count' => $this->countInstallable($compatibility['themes']),
                'plugin_blocked_count' => $this->countBlocked($compatibility['plugins']),
                'theme_blocked_count' => $this->countBlocked($compatibility['themes']),
            ],
        ];
    }

    public function apply(array $preflight): array
    {
        $report = [
            'generated_at' => gmdate('c'),
            'core' => ['status' => 'skipped', 'reason' => 'not_requested_or_not_installable'],
            'plugins' => [],
            'themes' => [],
        ];

        $this->includeUpgraderDependencies();

        $coreComp = $preflight['compatibility']['core'] ?? [];
        if (($coreComp['requested'] ?? false) === true && ($coreComp['installable'] ?? false) === true) {
            $report['core'] = $this->applyCoreUpdate($preflight['available_updates']['core'] ?? null);
        }

        $pluginCompat = is_array($preflight['compatibility']['plugins'] ?? null) ? $preflight['compatibility']['plugins'] : [];
        $report['plugins'] = $this->applyPluginUpdates($pluginCompat);

        $themeCompat = is_array($preflight['compatibility']['themes'] ?? null) ? $preflight['compatibility']['themes'] : [];
        $report['themes'] = $this->applyThemeUpdates($themeCompat);

        return $report;
    }

    private function environment(): array
    {
        global $wp_version, $wpdb;

        $dbVersion = null;
        if (isset($wpdb) && is_object($wpdb) && method_exists($wpdb, 'db_version')) {
            $dbVersion = (string) $wpdb->db_version();
        }

        return [
            'wordpress_version' => isset($wp_version) ? (string) $wp_version : (function_exists('get_bloginfo') ? (string) get_bloginfo('version') : 'unknown'),
            'php_version' => PHP_VERSION,
            'db_version' => $dbVersion,
        ];
    }

    private function coreUpdateCandidate(): ?array
    {
        if (! function_exists('get_core_updates')) {
            return null;
        }

        $offers = get_core_updates(['dismissed' => false]);
        if (! is_array($offers) || $offers === []) {
            return null;
        }

        $offer = $offers[0];

        return [
            'version' => (string) ($offer->current ?? ''),
            'php_version' => (string) ($offer->php_version ?? ''),
            'mysql_version' => (string) ($offer->mysql_version ?? ''),
            'response' => (string) ($offer->response ?? ''),
            'raw' => $offer,
        ];
    }

    private function pluginUpdateCandidates(mixed $requestedPlugins): array
    {
        if (function_exists('wp_update_plugins')) {
            wp_update_plugins();
        }

        $transient = function_exists('get_site_transient') ? get_site_transient('update_plugins') : null;
        $response = is_object($transient) && is_object($transient->response ?? null) ? (array) $transient->response : [];

        $requested = is_array($requestedPlugins) ? array_fill_keys(array_map('strval', $requestedPlugins), true) : null;

        $out = [];
        foreach ($response as $pluginFile => $item) {
            if ($requested !== null && ! isset($requested[(string) $pluginFile])) {
                continue;
            }

            $out[] = [
                'id' => (string) $pluginFile,
                'type' => 'plugin',
                'new_version' => (string) ($item->new_version ?? ''),
                'requires_php' => (string) ($item->requires_php ?? ''),
                'requires_wp' => (string) ($item->requires ?? ''),
                'raw' => $item,
            ];
        }

        return $out;
    }

    private function themeUpdateCandidates(mixed $requestedThemes): array
    {
        if (function_exists('wp_update_themes')) {
            wp_update_themes();
        }

        $transient = function_exists('get_site_transient') ? get_site_transient('update_themes') : null;
        $response = is_object($transient) && is_array($transient->response ?? null) ? (array) $transient->response : [];

        $requested = is_array($requestedThemes) ? array_fill_keys(array_map('strval', $requestedThemes), true) : null;

        $out = [];
        foreach ($response as $stylesheet => $item) {
            if ($requested !== null && ! isset($requested[(string) $stylesheet])) {
                continue;
            }

            $requiresPhp = is_array($item) ? (string) ($item['requires_php'] ?? '') : '';
            $requiresWp = is_array($item) ? (string) ($item['requires'] ?? '') : '';
            $newVersion = is_array($item) ? (string) ($item['new_version'] ?? '') : '';

            $out[] = [
                'id' => (string) $stylesheet,
                'type' => 'theme',
                'new_version' => $newVersion,
                'requires_php' => $requiresPhp,
                'requires_wp' => $requiresWp,
                'raw' => $item,
            ];
        }

        return $out;
    }

    private function evaluateCoreCompatibility(?array $core, array $environment, bool $requested): array
    {
        if (! $requested) {
            return ['requested' => false, 'installable' => false, 'issues' => [], 'warnings' => []];
        }

        if ($core === null) {
            return ['requested' => true, 'installable' => false, 'issues' => [['code' => 'no_core_update', 'message' => 'No core update candidate found.']], 'warnings' => []];
        }

        $issues = [];

        if (! $this->meetsVersion((string) $environment['php_version'], (string) ($core['php_version'] ?? ''))) {
            $issues[] = ['code' => 'php_incompatible', 'message' => 'Current PHP version does not satisfy core update requirement.'];
        }

        $dbVersion = (string) ($environment['db_version'] ?? '');
        if ($dbVersion !== '' && ! $this->meetsVersion($dbVersion, (string) ($core['mysql_version'] ?? ''))) {
            $issues[] = ['code' => 'db_incompatible', 'message' => 'Current DB version does not satisfy core update requirement.'];
        }

        return [
            'requested' => true,
            'installable' => $issues === [],
            'issues' => $issues,
            'warnings' => [],
            'candidate' => [
                'version' => (string) ($core['version'] ?? ''),
                'php_version' => (string) ($core['php_version'] ?? ''),
                'mysql_version' => (string) ($core['mysql_version'] ?? ''),
            ],
        ];
    }

    private function evaluateComponentCompatibility(array $candidates, array $environment): array
    {
        $out = [];

        foreach ($candidates as $candidate) {
            $issues = [];
            $warnings = [];

            $requiresPhp = (string) ($candidate['requires_php'] ?? '');
            $requiresWp = (string) ($candidate['requires_wp'] ?? '');

            if (! $this->meetsVersion((string) $environment['php_version'], $requiresPhp)) {
                $issues[] = ['code' => 'php_incompatible', 'message' => 'PHP version does not meet requirement.'];
            }

            if (! $this->meetsVersion((string) $environment['wordpress_version'], $requiresWp)) {
                $issues[] = ['code' => 'wp_incompatible', 'message' => 'WordPress version does not meet requirement.'];
            }

            if ($requiresPhp === '' && $requiresWp === '') {
                $warnings[] = ['code' => 'missing_requirements_metadata', 'message' => 'No explicit compatibility metadata available.'];
            }

            $out[] = [
                'id' => (string) ($candidate['id'] ?? ''),
                'type' => (string) ($candidate['type'] ?? ''),
                'new_version' => (string) ($candidate['new_version'] ?? ''),
                'requires_php' => $requiresPhp,
                'requires_wp' => $requiresWp,
                'installable' => $issues === [],
                'issues' => $issues,
                'warnings' => $warnings,
                'raw' => $candidate['raw'] ?? null,
            ];
        }

        return $out;
    }

    private function applyCoreUpdate(mixed $coreCandidate): array
    {
        if (! class_exists('Core_Upgrader') || ! is_array($coreCandidate) || ! isset($coreCandidate['raw'])) {
            return ['status' => 'failed', 'reason' => 'core_upgrader_unavailable'];
        }

        $upgrader = new \Core_Upgrader(new \Automatic_Upgrader_Skin());
        $result = $upgrader->upgrade($coreCandidate['raw']);

        if ($result instanceof WP_Error) {
            return ['status' => 'failed', 'reason' => $result->get_error_message()];
        }

        if ($result === false) {
            return ['status' => 'failed', 'reason' => 'core_upgrade_returned_false'];
        }

        return ['status' => 'updated', 'to_version' => (string) ($coreCandidate['version'] ?? '')];
    }

    private function applyPluginUpdates(array $compatibility): array
    {
        $installable = [];
        $report = [];

        foreach ($compatibility as $item) {
            $id = (string) ($item['id'] ?? '');
            if ($id === '') {
                continue;
            }

            if (($item['installable'] ?? false) !== true) {
                $report[] = ['id' => $id, 'status' => 'skipped', 'reason' => 'compatibility_blocked', 'issues' => $item['issues'] ?? []];
                continue;
            }

            $installable[] = $id;
        }

        if ($installable === []) {
            return $report;
        }

        if (! class_exists('Plugin_Upgrader')) {
            foreach ($installable as $id) {
                $report[] = ['id' => $id, 'status' => 'failed', 'reason' => 'plugin_upgrader_unavailable'];
            }
            return $report;
        }

        $upgrader = new \Plugin_Upgrader(new \Automatic_Upgrader_Skin());
        $results = $upgrader->bulk_upgrade($installable);
        $results = is_array($results) ? $results : [];

        foreach ($installable as $id) {
            $result = $results[$id] ?? null;
            if ($result instanceof WP_Error) {
                $report[] = ['id' => $id, 'status' => 'failed', 'reason' => $result->get_error_message()];
                continue;
            }
            if ($result === false || $result === null) {
                $report[] = ['id' => $id, 'status' => 'failed', 'reason' => 'plugin_update_failed'];
                continue;
            }
            $report[] = ['id' => $id, 'status' => 'updated'];
        }

        return $report;
    }

    private function applyThemeUpdates(array $compatibility): array
    {
        $installable = [];
        $report = [];

        foreach ($compatibility as $item) {
            $id = (string) ($item['id'] ?? '');
            if ($id === '') {
                continue;
            }

            if (($item['installable'] ?? false) !== true) {
                $report[] = ['id' => $id, 'status' => 'skipped', 'reason' => 'compatibility_blocked', 'issues' => $item['issues'] ?? []];
                continue;
            }

            $installable[] = $id;
        }

        if ($installable === []) {
            return $report;
        }

        if (! class_exists('Theme_Upgrader')) {
            foreach ($installable as $id) {
                $report[] = ['id' => $id, 'status' => 'failed', 'reason' => 'theme_upgrader_unavailable'];
            }
            return $report;
        }

        $upgrader = new \Theme_Upgrader(new \Automatic_Upgrader_Skin());
        $results = $upgrader->bulk_upgrade($installable);
        $results = is_array($results) ? $results : [];

        foreach ($installable as $id) {
            $result = $results[$id] ?? null;
            if ($result instanceof WP_Error) {
                $report[] = ['id' => $id, 'status' => 'failed', 'reason' => $result->get_error_message()];
                continue;
            }
            if ($result === false || $result === null) {
                $report[] = ['id' => $id, 'status' => 'failed', 'reason' => 'theme_update_failed'];
                continue;
            }
            $report[] = ['id' => $id, 'status' => 'updated'];
        }

        return $report;
    }

    private function includeUpgraderDependencies(): void
    {
        if (! defined('ABSPATH')) {
            return;
        }

        $paths = [
            ABSPATH . 'wp-admin/includes/update.php',
            ABSPATH . 'wp-admin/includes/admin.php',
            ABSPATH . 'wp-admin/includes/file.php',
            ABSPATH . 'wp-admin/includes/class-wp-upgrader.php',
            ABSPATH . 'wp-admin/includes/class-core-upgrader.php',
            ABSPATH . 'wp-admin/includes/class-plugin-upgrader.php',
            ABSPATH . 'wp-admin/includes/class-theme-upgrader.php',
        ];

        foreach ($paths as $path) {
            if (is_readable($path)) {
                require_once $path;
            }
        }
    }

    private function meetsVersion(string $current, string $required): bool
    {
        if ($required === '') {
            return true;
        }

        if ($current === '') {
            return false;
        }

        return version_compare($current, $required, '>=');
    }

    private function countInstallable(array $items): int
    {
        $count = 0;
        foreach ($items as $item) {
            if (($item['installable'] ?? false) === true) {
                $count++;
            }
        }
        return $count;
    }

    private function countBlocked(array $items): int
    {
        $count = 0;
        foreach ($items as $item) {
            if (($item['installable'] ?? false) !== true) {
                $count++;
            }
        }
        return $count;
    }
}
