<?php

declare(strict_types=1);

namespace Agent\Modules\System;

use Agent\Security\RequestGuard;
use Agent\Support\Logger;
use Agent\Support\Options;
use WP_Error;
use WP_REST_Request;
use WP_REST_Response;

final class SystemController
{
    private const ALLOWED_CHANGESET_OPERATIONS = [
        'set_readonly_mode',
        'set_emergency_lock',
        'set_module',
        'set_rate_limit',
    ];

    private CapabilityMapService $capabilityMap;
    private ChangesetService $changeset;
    private SnapshotService $snapshot;
    private UpdateService $updates;
    private RequestGuard $guard;
    private Logger $logger;
    private Options $options;

    public function __construct(CapabilityMapService $capabilityMap, ChangesetService $changeset, SnapshotService $snapshot, UpdateService $updates, RequestGuard $guard, Logger $logger, Options $options)
    {
        $this->capabilityMap = $capabilityMap;
        $this->changeset = $changeset;
        $this->snapshot = $snapshot;
        $this->updates = $updates;
        $this->guard = $guard;
        $this->logger = $logger;
        $this->options = $options;
    }

    public function registerRoutes(): void
    {
        register_rest_route(
            'wpai/v1',
            '/capabilities/map',
            [
                'methods' => 'GET',
                'callback' => [$this, 'capabilitiesMap'],
                'permission_callback' => [$this, 'canRead'],
            ]
        );

        register_rest_route(
            'wpai/v1',
            '/plan/validate',
            [
                'methods' => 'POST',
                'callback' => [$this, 'validatePlan'],
                'permission_callback' => [$this, 'canRead'],
            ]
        );

        register_rest_route(
            'wpai/v1',
            '/changeset/apply',
            [
                'methods' => 'POST',
                'callback' => [$this, 'applyChangeset'],
                'permission_callback' => static fn (): bool => current_user_can('manage_options'),
            ]
        );

        register_rest_route(
            'wpai/v1',
            '/changeset/rollback',
            [
                'methods' => 'POST',
                'callback' => [$this, 'rollbackChangeset'],
                'permission_callback' => static fn (): bool => current_user_can('manage_options'),
            ]
        );

        register_rest_route(
            'wpai/v1',
            '/logs/audit',
            [
                'methods' => 'GET',
                'callback' => [$this, 'auditLogs'],
                'permission_callback' => [$this, 'canRead'],
            ]
        );

        register_rest_route(
            'wpai/v1',
            '/logs/errors',
            [
                'methods' => 'GET',
                'callback' => [$this, 'errorLogs'],
                'permission_callback' => [$this, 'canRead'],
            ]
        );

        register_rest_route(
            'wpai/v1',
            '/system/snapshot',
            [
                'methods' => 'GET',
                'callback' => [$this, 'systemSnapshot'],
                'permission_callback' => [$this, 'canRead'],
            ]
        );

        register_rest_route(
            'wpai/v1',
            '/system/restore-point',
            [
                'methods' => 'POST',
                'callback' => [$this, 'createRestorePoint'],
                'permission_callback' => static fn (): bool => current_user_can('manage_options'),
            ]
        );

        register_rest_route(
            'wpai/v1',
            '/system/full-write-access',
            [
                'methods' => 'POST',
                'callback' => [$this, 'setFullWriteAccess'],
                'permission_callback' => static fn (): bool => current_user_can('manage_options'),
            ]
        );

        register_rest_route(
            'wpai/v1',
            '/updates/preflight',
            [
                'methods' => 'POST',
                'callback' => [$this, 'updatesPreflight'],
                'permission_callback' => [$this, 'canRead'],
            ]
        );

        register_rest_route(
            'wpai/v1',
            '/updates/apply',
            [
                'methods' => 'POST',
                'callback' => [$this, 'updatesApply'],
                'permission_callback' => static fn (): bool => current_user_can('manage_options'),
            ]
        );
    }

    public function canRead(): bool
    {
        return $this->guard->checkReadAccess();
    }

    public function capabilitiesMap(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        $rate = $this->guard->assertRateLimit('capabilities_map');
        if ($rate instanceof WP_Error) {
            return $rate;
        }

        $payload = $this->capabilityMap->generate();
        $this->logger->log('capabilities_map_requested', 'info');

        return new WP_REST_Response($payload, 200);
    }

    public function validatePlan(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        $rate = $this->guard->assertRateLimit('plan_validate');
        if ($rate instanceof WP_Error) {
            return $rate;
        }

        $actions = $request->get_param('actions');
        if (! is_array($actions)) {
            return new WP_Error('agent_invalid_plan', 'Field "actions" must be an array.', ['status' => 400]);
        }

        $evaluation = $this->evaluateActions($actions);

        $this->logger->log(
            'plan_validated',
            'info',
            ['action_count' => count($actions), 'valid_count' => $evaluation['valid_count']]
        );

        return new WP_REST_Response(
            [
                'generated_at' => gmdate('c'),
                'actor_user_id' => get_current_user_id(),
                'readonly_mode' => $evaluation['readonly_mode'],
                'full_write_access' => $evaluation['full_write_access'],
                'emergency_lock' => $evaluation['emergency_lock'],
                'action_count' => count($actions),
                'valid_count' => $evaluation['valid_count'],
                'overall_valid' => $evaluation['overall_valid'],
                'results' => $evaluation['results'],
            ],
            200
        );
    }

    public function applyChangeset(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        $rate = $this->guard->assertRateLimit('changeset_apply');
        if ($rate instanceof WP_Error) {
            return $rate;
        }

        if (! $this->hasValidOptionalNonce($request)) {
            return new WP_Error('agent_invalid_nonce', 'Invalid nonce.', ['status' => 403]);
        }

        if (! $this->guard->checkWriteAccess()) {
            return new WP_Error('agent_write_disabled', 'Full write access is disabled.', ['status' => 403]);
        }

        $actions = $request->get_param('actions');
        if (! is_array($actions) || $actions === []) {
            return new WP_Error('agent_invalid_changeset', 'Field "actions" must be a non-empty array.', ['status' => 400]);
        }

        $evaluation = $this->evaluateActions($actions, true);
        if (! $evaluation['overall_valid']) {
            return new WP_Error(
                'agent_changeset_blocked',
                'Changeset contains blocking issues.',
                [
                    'status' => 400,
                    'results' => $evaluation['results'],
                    'readonly_mode' => $evaluation['readonly_mode'],
                    'full_write_access' => $evaluation['full_write_access'],
                    'emergency_lock' => $evaluation['emergency_lock'],
                ]
            );
        }

        $auditId = $this->generateAuditId();
        $result = $this->changeset->apply($actions, $auditId);

        $this->logger->log(
            'changeset_applied',
            'warning',
            ['audit_id' => $auditId, 'action_count' => count($actions), 'rollback_token' => $result['rollback_token']]
        );

        return new WP_REST_Response(
            [
                'generated_at' => gmdate('c'),
                'audit_id' => $auditId,
                'rollback_token' => $result['rollback_token'],
                'applied_count' => count($result['applied']),
                'applied' => $result['applied'],
            ],
            200
        );
    }

    public function rollbackChangeset(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        $rate = $this->guard->assertRateLimit('changeset_rollback');
        if ($rate instanceof WP_Error) {
            return $rate;
        }

        if (! $this->hasValidOptionalNonce($request)) {
            return new WP_Error('agent_invalid_nonce', 'Invalid nonce.', ['status' => 403]);
        }

        if (! $this->guard->checkWriteAccess()) {
            return new WP_Error('agent_write_disabled', 'Full write access is disabled.', ['status' => 403]);
        }

        $rollbackToken = sanitize_text_field((string) $request->get_param('rollback_token'));
        if ($rollbackToken === '') {
            return new WP_Error('agent_invalid_rollback_token', 'Field "rollback_token" is required.', ['status' => 400]);
        }

        $result = $this->changeset->rollback($rollbackToken);
        if ($result === null) {
            return new WP_Error('agent_rollback_not_found', 'Rollback token not found.', ['status' => 404]);
        }

        if (($result['status'] ?? '') === 'already_rolled_back') {
            return new WP_Error('agent_rollback_consumed', 'Rollback token already consumed.', ['status' => 409]);
        }

        $this->logger->log('changeset_rolled_back', 'warning', ['rollback_token' => $rollbackToken, 'audit_id' => $result['audit_id'] ?? '']);

        return new WP_REST_Response(
            [
                'generated_at' => gmdate('c'),
                'rollback_token' => $rollbackToken,
                'status' => 'rolled_back',
                'audit_id' => (string) ($result['audit_id'] ?? ''),
                'inverse_count' => (int) ($result['inverse_count'] ?? 0),
            ],
            200
        );
    }

    public function auditLogs(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        $rate = $this->guard->assertRateLimit('logs_audit');
        if ($rate instanceof WP_Error) {
            return $rate;
        }

        $limit = max(1, min(200, (int) $request->get_param('limit')));
        if ($limit === 1 && (string) $request->get_param('limit') === '') {
            $limit = 50;
        }

        $severity = sanitize_key((string) $request->get_param('severity'));
        if ($severity === '') {
            $severity = null;
        }

        $action = sanitize_key((string) $request->get_param('action'));
        if ($action === '') {
            $action = null;
        }

        $logs = $this->logger->query($limit, $severity, $action);
        $this->logger->log('audit_logs_requested', 'info', ['limit' => $limit]);

        return new WP_REST_Response(
            [
                'generated_at' => gmdate('c'),
                'count' => count($logs),
                'logs' => $logs,
            ],
            200
        );
    }

    public function errorLogs(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        $rate = $this->guard->assertRateLimit('logs_errors');
        if ($rate instanceof WP_Error) {
            return $rate;
        }

        $limit = max(1, min(200, (int) $request->get_param('limit')));
        if ($limit === 1 && (string) $request->get_param('limit') === '') {
            $limit = 50;
        }

        $logs = $this->logger->errors($limit);
        $this->logger->log('error_logs_requested', 'info', ['limit' => $limit]);

        return new WP_REST_Response(
            [
                'generated_at' => gmdate('c'),
                'count' => count($logs),
                'logs' => $logs,
            ],
            200
        );
    }

    public function systemSnapshot(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        $rate = $this->guard->assertRateLimit('system_snapshot');
        if ($rate instanceof WP_Error) {
            return $rate;
        }

        $snapshot = $this->snapshot->createSnapshot();
        $limit = max(1, min(20, (int) $request->get_param('restore_points_limit')));
        if ($limit === 1 && (string) $request->get_param('restore_points_limit') === '') {
            $limit = 5;
        }
        $restorePoints = $this->snapshot->latestRestorePoints($limit);

        $this->logger->log('system_snapshot_requested', 'info', ['restore_points_limit' => $limit]);

        return new WP_REST_Response(
            [
                'snapshot' => $snapshot,
                'restore_points' => $restorePoints,
                'restore_points_count' => count($restorePoints),
            ],
            200
        );
    }

    public function createRestorePoint(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        $rate = $this->guard->assertRateLimit('restore_point_create');
        if ($rate instanceof WP_Error) {
            return $rate;
        }

        if (! $this->hasValidOptionalNonce($request)) {
            return new WP_Error('agent_invalid_nonce', 'Invalid nonce.', ['status' => 403]);
        }

        if (! $this->guard->checkWriteAccess()) {
            return new WP_Error('agent_write_disabled', 'Full write access is disabled.', ['status' => 403]);
        }

        $label = sanitize_text_field((string) $request->get_param('label'));
        $snapshot = $this->snapshot->createSnapshot();
        $point = $this->snapshot->storeRestorePoint($snapshot, $label);

        $this->logger->log('restore_point_created', 'warning', ['restore_point_id' => $point['restore_point_id'] ?? '', 'label' => $label]);

        return new WP_REST_Response(
            [
                'generated_at' => gmdate('c'),
                'restore_point' => $point,
            ],
            200
        );
    }

    public function setFullWriteAccess(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        $rate = $this->guard->assertRateLimit('full_write_access_set');
        if ($rate instanceof WP_Error) {
            return $rate;
        }

        if (! $this->hasValidOptionalNonce($request)) {
            return new WP_Error('agent_invalid_nonce', 'Invalid nonce.', ['status' => 403]);
        }

        $enabled = (bool) $request->get_param('enabled');
        $this->options->set('full_write_access', $enabled);
        $this->logger->log('full_write_access_updated', 'warning', ['enabled' => $enabled]);

        return new WP_REST_Response(
            [
                'generated_at' => gmdate('c'),
                'full_write_access' => $enabled,
            ],
            200
        );
    }

    public function updatesPreflight(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        $rate = $this->guard->assertRateLimit('updates_preflight');
        if ($rate instanceof WP_Error) {
            return $rate;
        }

        $targets = $this->normalizeUpdateTargets($request->get_param('targets'));
        $report = $this->updates->preflight($targets);
        $this->logger->log('updates_preflight_requested', 'info', ['targets' => $targets]);

        return new WP_REST_Response($report, 200);
    }

    public function updatesApply(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        $rate = $this->guard->assertRateLimit('updates_apply');
        if ($rate instanceof WP_Error) {
            return $rate;
        }

        if (! $this->hasValidOptionalNonce($request)) {
            return new WP_Error('agent_invalid_nonce', 'Invalid nonce.', ['status' => 403]);
        }

        if (! $this->guard->checkWriteAccess()) {
            return new WP_Error('agent_write_disabled', 'Full write access is disabled.', ['status' => 403]);
        }

        $targets = $this->normalizeUpdateTargets($request->get_param('targets'));
        $preflight = $this->updates->preflight($targets);

        $label = sanitize_text_field((string) $request->get_param('restore_label'));
        if ($label === '') {
            $label = 'pre_update_' . gmdate('Ymd_His');
        }

        $restorePoint = $this->snapshot->storeRestorePoint($this->snapshot->createSnapshot(), $label);
        $apply = $this->updates->apply($preflight);

        $this->logger->log(
            'updates_apply_executed',
            'warning',
            ['restore_point_id' => $restorePoint['restore_point_id'] ?? '', 'targets' => $targets]
        );

        return new WP_REST_Response(
            [
                'generated_at' => gmdate('c'),
                'restore_point' => $restorePoint,
                'preflight' => $preflight,
                'apply' => $apply,
            ],
            200
        );
    }

    private function evaluateActions(array $actions, bool $strictChangesetMode = false): array
    {
        $readonlyMode = (bool) $this->options->get('readonly_mode', true);
        $fullWriteAccess = (bool) $this->options->get('full_write_access', false);
        $emergencyLock = (bool) $this->options->get('emergency_lock', false);
        $projectedFullWriteAccess = $fullWriteAccess;
        $projectedEmergencyLock = $emergencyLock;

        $results = [];
        $validCount = 0;

        foreach ($actions as $index => $action) {
            $normalized = is_array($action) ? $action : [];
            $module = sanitize_key((string) ($normalized['module'] ?? ''));
            $operation = sanitize_key((string) ($normalized['operation'] ?? 'read'));
            $actionId = sanitize_text_field((string) ($normalized['id'] ?? ('action_' . (string) $index)));

            $requiresWrite = array_key_exists('requires_write', $normalized)
                ? (bool) $normalized['requires_write']
                : $this->isWriteOperation($operation);

            $requiredCaps = $this->extractRequiredCapabilities($normalized, $requiresWrite);
            $issues = [];

            if ($projectedEmergencyLock) {
                $issues[] = ['code' => 'emergency_lock_active', 'severity' => 'blocking', 'message' => 'Emergency lock blocks all agent operations.'];
            }

            if ($module !== '' && ! $this->isModuleEnabled($module)) {
                $issues[] = ['code' => 'module_disabled', 'severity' => 'blocking', 'message' => 'Target module is disabled.'];
            }

            if ($requiresWrite && ! $projectedFullWriteAccess) {
                $issues[] = ['code' => 'full_write_access_disabled', 'severity' => 'blocking', 'message' => 'Write operation blocked because full write access is disabled.'];
            }

            if ($strictChangesetMode && ! in_array($operation, self::ALLOWED_CHANGESET_OPERATIONS, true)) {
                $issues[] = ['code' => 'operation_not_supported', 'severity' => 'blocking', 'message' => 'Operation not supported by changeset endpoint.'];
            }

            if ($strictChangesetMode) {
                $operationIssues = $this->validateChangesetPayload($operation, $normalized);
                foreach ($operationIssues as $operationIssue) {
                    $issues[] = $operationIssue;
                }
            }

            foreach ($requiredCaps as $capability) {
                if (! current_user_can($capability)) {
                    $issues[] = [
                        'code' => 'missing_capability',
                        'severity' => 'blocking',
                        'message' => 'Missing capability: ' . $capability,
                        'capability' => $capability,
                    ];
                }
            }

            $isValid = $issues === [];
            if ($isValid) {
                $validCount++;
                if ($strictChangesetMode) {
                    $payload = is_array($normalized['payload'] ?? null) ? $normalized['payload'] : [];
                    if ($operation === 'set_readonly_mode' && array_key_exists('enabled', $payload)) {
                        $projectedFullWriteAccess = ! (bool) $payload['enabled'];
                    }
                    if ($operation === 'set_emergency_lock' && array_key_exists('enabled', $payload)) {
                        $projectedEmergencyLock = (bool) $payload['enabled'];
                    }
                }
            }

            $results[] = [
                'index' => $index,
                'id' => $actionId,
                'module' => $module,
                'operation' => $operation,
                'requires_write' => $requiresWrite,
                'required_capabilities' => $requiredCaps,
                'valid' => $isValid,
                'issues' => $issues,
            ];
        }

        return [
            'results' => $results,
            'valid_count' => $validCount,
            'overall_valid' => $validCount === count($actions),
            'readonly_mode' => $readonlyMode,
            'full_write_access' => $fullWriteAccess,
            'emergency_lock' => $emergencyLock,
        ];
    }

    private function hasValidOptionalNonce(WP_REST_Request $request): bool
    {
        $nonce = (string) $request->get_header('x_wp_nonce');
        if ($nonce === '') {
            return true;
        }

        return $this->guard->validateStateChangeNonce($nonce);
    }

    private function isModuleEnabled(string $module): bool
    {
        if ($module === 'system') {
            return true;
        }

        return (bool) $this->options->get('modules.' . $module, true);
    }

    private function validateChangesetPayload(string $operation, array $action): array
    {
        $payload = is_array($action['payload'] ?? null) ? $action['payload'] : [];
        $issues = [];

        if ($operation === 'set_readonly_mode' || $operation === 'set_emergency_lock') {
            if (! array_key_exists('enabled', $payload)) {
                $issues[] = ['code' => 'invalid_payload', 'severity' => 'blocking', 'message' => 'Payload must contain "enabled".'];
            }
            return $issues;
        }

        if ($operation === 'set_module') {
            $module = sanitize_key((string) ($payload['module'] ?? ''));
            if ($module === '') {
                $issues[] = ['code' => 'invalid_payload', 'severity' => 'blocking', 'message' => 'Payload must contain non-empty "module".'];
            }
            if (! array_key_exists('enabled', $payload)) {
                $issues[] = ['code' => 'invalid_payload', 'severity' => 'blocking', 'message' => 'Payload must contain "enabled".'];
            }
            return $issues;
        }

        if ($operation === 'set_rate_limit') {
            if (! array_key_exists('value', $payload)) {
                $issues[] = ['code' => 'invalid_payload', 'severity' => 'blocking', 'message' => 'Payload must contain "value".'];
            }
            return $issues;
        }

        return $issues;
    }

    private function isWriteOperation(string $operation): bool
    {
        return in_array($operation, ['create', 'update', 'delete', 'set', 'activate', 'deactivate', 'set_readonly_mode', 'set_emergency_lock', 'set_module', 'set_rate_limit'], true);
    }

    private function extractRequiredCapabilities(array $action, bool $requiresWrite): array
    {
        $fromPayload = $action['required_capabilities'] ?? [];

        if (! is_array($fromPayload)) {
            $fromPayload = [];
        }

        $caps = [];
        foreach ($fromPayload as $capability) {
            if (! is_string($capability) || $capability === '') {
                continue;
            }
            $caps[] = sanitize_key($capability);
        }

        if ($caps !== []) {
            return array_values(array_unique($caps));
        }

        return $requiresWrite ? ['manage_options'] : [];
    }

    private function generateAuditId(): string
    {
        return 'audit_' . gmdate('Ymd_His') . '_' . wp_generate_password(8, false, false);
    }

    private function normalizeUpdateTargets(mixed $targets): array
    {
        if (! is_array($targets)) {
            return ['core' => false, 'plugins' => [], 'themes' => []];
        }

        $plugins = is_array($targets['plugins'] ?? null) ? array_values(array_filter(array_map('strval', $targets['plugins']), static fn (string $v): bool => $v !== '')) : [];
        $themes = is_array($targets['themes'] ?? null) ? array_values(array_filter(array_map('strval', $targets['themes']), static fn (string $v): bool => $v !== '')) : [];

        return [
            'core' => (bool) ($targets['core'] ?? false),
            'plugins' => $plugins,
            'themes' => $themes,
        ];
    }
}
