<?php

declare(strict_types=1);

namespace Agent\Modules\System;

use Agent\Plugin;

final class SnapshotService
{
    public function createSnapshot(): array
    {
        $theme = wp_get_theme();

        return [
            'generated_at' => gmdate('c'),
            'wordpress' => [
                'version' => (string) get_bloginfo('version'),
                'site_url' => (string) get_site_url(),
            ],
            'runtime' => [
                'active_plugins' => array_values((array) get_option('active_plugins', [])),
                'theme' => [
                    'stylesheet' => $theme->get_stylesheet(),
                    'template' => $theme->get_template(),
                    'version' => (string) $theme->get('Version'),
                ],
                'permalink_structure' => (string) get_option('permalink_structure', ''),
            ],
            'agent' => [
                'settings' => get_option(Plugin::OPTION_SETTINGS, []),
            ],
        ];
    }

    public function storeRestorePoint(array $snapshot, string $label = ''): array
    {
        $points = $this->getRestorePoints();

        $point = [
            'restore_point_id' => $this->generateId(),
            'created_at' => gmdate('c'),
            'actor_user_id' => get_current_user_id(),
            'label' => $label,
            'snapshot' => $snapshot,
        ];

        $points[] = $point;

        if (count($points) > 50) {
            $points = array_slice($points, -50);
        }

        update_option(Plugin::OPTION_RESTORE_POINTS, $points, false);

        return $point;
    }

    public function latestRestorePoints(int $limit = 10): array
    {
        $points = $this->getRestorePoints();
        $points = array_reverse($points);

        return array_slice($points, 0, max(1, $limit));
    }

    private function getRestorePoints(): array
    {
        $raw = get_option(Plugin::OPTION_RESTORE_POINTS, []);

        return is_array($raw) ? $raw : [];
    }

    private function generateId(): string
    {
        try {
            return 'rp_' . bin2hex(random_bytes(8));
        } catch (\Throwable) {
            return 'rp_' . md5((string) microtime(true) . ':' . (string) wp_rand());
        }
    }
}
