<?php

declare(strict_types=1);

namespace Agent\Modules\Seo;

final class YoastAdapter implements SeoAdapterInterface
{
    public function key(): string
    {
        return 'yoast';
    }

    public function label(): string
    {
        return 'Yoast SEO';
    }

    public function isActive(array $activePlugins): bool
    {
        return in_array('wordpress-seo/wp-seo.php', $activePlugins, true);
    }

    public function writePostMeta(int $postId, array $seo): array
    {
        $written = [];

        $map = [
            'seo_title' => '_yoast_wpseo_title',
            'meta_description' => '_yoast_wpseo_metadesc',
            'focus_keyword' => '_yoast_wpseo_focuskw',
            'canonical' => '_yoast_wpseo_canonical',
            'og_description' => '_yoast_wpseo_opengraph-description',
        ];

        foreach ($map as $field => $metaKey) {
            if (! array_key_exists($field, $seo)) {
                continue;
            }

            $value = trim((string) $seo[$field]);
            update_post_meta($postId, $metaKey, $value);
            $written[] = $metaKey;
        }

        if (array_key_exists('robots', $seo)) {
            $robots = $this->parseRobots((string) $seo['robots']);
            update_post_meta($postId, '_yoast_wpseo_meta-robots-noindex', $robots['noindex'] ? '1' : '0');
            update_post_meta($postId, '_yoast_wpseo_meta-robots-nofollow', $robots['nofollow'] ? '1' : '0');
            $written[] = '_yoast_wpseo_meta-robots-noindex';
            $written[] = '_yoast_wpseo_meta-robots-nofollow';
        }

        return [
            'status' => 'written',
            'written_meta_keys' => $written,
        ];
    }

    private function parseRobots(string $value): array
    {
        $normalized = strtolower(str_replace(' ', '', $value));
        $parts = array_filter(explode(',', $normalized), static fn (string $entry): bool => $entry !== '');

        return [
            'noindex' => in_array('noindex', $parts, true),
            'nofollow' => in_array('nofollow', $parts, true),
        ];
    }
}

