<?php

declare(strict_types=1);

namespace Agent\Modules\Seo;

final class SeoMetaOrchestrator
{
    /** @var SeoAdapterInterface[] */
    private array $adapters;

    public function __construct()
    {
        $this->adapters = [
            new WpZeoAdapter(),
            new YoastAdapter(),
            new RankMathAdapter(),
        ];
    }

    public function writeForPost(int $postId, array $seo): array
    {
        $activePlugins = (array) get_option('active_plugins', []);
        $activeAdapters = $this->activeAdapters($activePlugins);

        $report = [
            'active_plugins' => array_values(array_map(static fn (SeoAdapterInterface $adapter): string => $adapter->key(), $activeAdapters)),
            'target_plugin' => null,
            'results' => [],
        ];

        if ($activeAdapters === []) {
            $report['results'][] = [
                'plugin' => 'none',
                'status' => 'skipped',
                'reason' => 'no_supported_seo_plugin_active',
            ];
            return $report;
        }

        $target = $activeAdapters[0];
        $report['target_plugin'] = $target->key();
        $result = $target->writePostMeta($postId, $seo);
        $report['results'][] = array_merge(
            [
                'plugin' => $target->key(),
                'label' => $target->label(),
            ],
            $result
        );

        foreach (array_slice($activeAdapters, 1) as $adapter) {
            $report['results'][] = [
                'plugin' => $adapter->key(),
                'label' => $adapter->label(),
                'status' => 'skipped',
                'reason' => 'inactive_priority_fallback',
            ];
        }

        return $report;
    }

    /**
     * @param array<int, string> $activePlugins
     * @return SeoAdapterInterface[]
     */
    private function activeAdapters(array $activePlugins): array
    {
        $active = [];
        foreach ($this->adapters as $adapter) {
            if ($adapter->isActive($activePlugins)) {
                $active[] = $adapter;
            }
        }

        return $active;
    }
}

