<?php

declare(strict_types=1);

namespace Agent\Modules\Planning;

use WP_Error;

final class WpZeoInstaller
{
    private const MANIFEST_URL = 'https://rademaker.online/downloads/wpzeo/wpzeo-zero-seo-latest.json';
    private const ALLOWED_HOSTS = ['rademaker.online'];

    public function installLatest(): array|WP_Error
    {
        if (! current_user_can('install_plugins')) {
            return new WP_Error('agent_wpzeo_install_forbidden', 'Du darfst keine Plugins installieren.', ['status' => 403]);
        }
        if (! function_exists('activate_plugin') || ! function_exists('is_plugin_active')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        if (is_plugin_active('wpzeo-zero-seo/wpzeo-zero-seo.php')) {
            return [
                'installed' => true,
                'activated' => true,
                'plugin_file' => 'wpzeo-zero-seo/wpzeo-zero-seo.php',
                'already_active' => true,
                'manifest_url' => self::MANIFEST_URL,
            ];
        }
        if (! function_exists('download_url')) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }

        $manifest = $this->fetchManifest();
        if ($manifest instanceof WP_Error) {
            return $manifest;
        }

        $package = $this->extractPackageData($manifest);
        if ($package instanceof WP_Error) {
            return $package;
        }

        $tmpFile = download_url($package['download_url'], 120);
        if ($tmpFile instanceof WP_Error) {
            return new WP_Error('agent_wpzeo_download_failed', 'Download der wpZeo-ZIP fehlgeschlagen: ' . $tmpFile->get_error_message(), ['status' => 502]);
        }

        try {
            $integrity = $this->verifyIntegrity($tmpFile, $package);
            if ($integrity instanceof WP_Error) {
                return $integrity;
            }

            $install = $this->installPackage($tmpFile);
            if ($install instanceof WP_Error) {
                return $install;
            }

            $pluginFile = (string) ($install['plugin_file'] ?? '');
            if ($pluginFile === '') {
                return new WP_Error('agent_wpzeo_install_unknown_plugin', 'Installation abgeschlossen, Plugin-Datei konnte nicht ermittelt werden.', ['status' => 500]);
            }

            $activation = activate_plugin($pluginFile);
            if ($activation instanceof WP_Error) {
                return new WP_Error('agent_wpzeo_activation_failed', 'wpZeo wurde installiert, aber nicht aktiviert: ' . $activation->get_error_message(), ['status' => 500, 'plugin_file' => $pluginFile]);
            }

            return [
                'installed' => true,
                'activated' => true,
                'plugin_file' => $pluginFile,
                'version' => (string) ($manifest['latest_version'] ?? ''),
                'manifest_url' => self::MANIFEST_URL,
                'package_url' => $package['download_url'],
                'sha256' => $package['sha256'],
                'sha512' => $package['sha512'],
            ];
        } finally {
            if (is_string($tmpFile) && $tmpFile !== '' && file_exists($tmpFile)) {
                @unlink($tmpFile);
            }
        }
    }

    private function fetchManifest(): array|WP_Error
    {
        $response = wp_remote_get(self::MANIFEST_URL, ['timeout' => 20]);
        if (is_wp_error($response)) {
            return new WP_Error('agent_wpzeo_manifest_failed', 'Manifest konnte nicht geladen werden: ' . $response->get_error_message(), ['status' => 502]);
        }

        $status = (int) wp_remote_retrieve_response_code($response);
        if ($status < 200 || $status >= 300) {
            return new WP_Error('agent_wpzeo_manifest_http', 'Manifest-Request fehlgeschlagen (HTTP ' . (string) $status . ').', ['status' => 502]);
        }

        $body = (string) wp_remote_retrieve_body($response);
        $manifest = json_decode($body, true);
        if (! is_array($manifest)) {
            return new WP_Error('agent_wpzeo_manifest_invalid', 'Manifest ist kein gueltiges JSON.', ['status' => 502]);
        }

        return $manifest;
    }

    private function extractPackageData(array $manifest): array|WP_Error
    {
        $file = is_array($manifest['file'] ?? null) ? $manifest['file'] : [];
        $downloadUrl = (string) ($file['download_url'] ?? '');
        $sha256 = strtolower(trim((string) ($file['sha256'] ?? '')));
        $sha512 = strtolower(trim((string) ($file['sha512'] ?? '')));
        $sizeBytes = (int) ($file['size_bytes'] ?? 0);

        if ($downloadUrl === '') {
            return new WP_Error('agent_wpzeo_manifest_missing_url', 'Manifest enthaelt keine Download-URL.', ['status' => 502]);
        }

        $urlHost = (string) wp_parse_url($downloadUrl, PHP_URL_HOST);
        if ($urlHost === '' || ! in_array(strtolower($urlHost), self::ALLOWED_HOSTS, true)) {
            return new WP_Error('agent_wpzeo_manifest_untrusted_host', 'Download-Host ist nicht erlaubt.', ['status' => 502, 'host' => $urlHost]);
        }

        if ($sha256 === '' || ! ctype_xdigit($sha256)) {
            return new WP_Error('agent_wpzeo_manifest_missing_hash', 'Manifest enthaelt keinen gueltigen SHA-256 Hash.', ['status' => 502]);
        }

        if ($sha512 !== '' && ! ctype_xdigit($sha512)) {
            return new WP_Error('agent_wpzeo_manifest_invalid_sha512', 'Manifest enthaelt einen ungueltigen SHA-512 Hash.', ['status' => 502]);
        }

        return [
            'download_url' => $downloadUrl,
            'sha256' => $sha256,
            'sha512' => $sha512,
            'size_bytes' => $sizeBytes,
        ];
    }

    private function verifyIntegrity(string $filePath, array $package): bool|WP_Error
    {
        if (! is_readable($filePath)) {
            return new WP_Error('agent_wpzeo_package_unreadable', 'Geladene ZIP-Datei ist nicht lesbar.', ['status' => 502]);
        }

        $expectedSize = (int) ($package['size_bytes'] ?? 0);
        if ($expectedSize > 0) {
            $actualSize = (int) filesize($filePath);
            if ($actualSize !== $expectedSize) {
                return new WP_Error('agent_wpzeo_size_mismatch', 'ZIP-Dateigroesse stimmt nicht mit dem Manifest ueberein.', ['status' => 502, 'expected' => $expectedSize, 'actual' => $actualSize]);
            }
        }

        $actualSha256 = strtolower((string) hash_file('sha256', $filePath));
        if (! hash_equals((string) $package['sha256'], $actualSha256)) {
            return new WP_Error('agent_wpzeo_sha256_mismatch', 'SHA-256 Integritaetspruefung fehlgeschlagen.', ['status' => 502]);
        }

        $expectedSha512 = (string) ($package['sha512'] ?? '');
        if ($expectedSha512 !== '') {
            $actualSha512 = strtolower((string) hash_file('sha512', $filePath));
            if (! hash_equals($expectedSha512, $actualSha512)) {
                return new WP_Error('agent_wpzeo_sha512_mismatch', 'SHA-512 Integritaetspruefung fehlgeschlagen.', ['status' => 502]);
            }
        }

        return true;
    }

    private function installPackage(string $packagePath): array|WP_Error
    {
        if (! function_exists('plugins_api')) {
            require_once ABSPATH . 'wp-admin/includes/plugin-install.php';
        }
        if (! function_exists('activate_plugin')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        if (! class_exists('Plugin_Upgrader')) {
            require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
        }
        if (! class_exists('Automatic_Upgrader_Skin')) {
            require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader-skins.php';
        }

        $skin = new \Automatic_Upgrader_Skin();
        $upgrader = new \Plugin_Upgrader($skin);
        $installed = $upgrader->install($packagePath, ['overwrite_package' => true]);

        if ($installed instanceof WP_Error) {
            return new WP_Error('agent_wpzeo_install_failed', 'Installation fehlgeschlagen: ' . $installed->get_error_message(), ['status' => 500]);
        }
        if ($installed !== true) {
            return new WP_Error('agent_wpzeo_install_failed', 'Installation konnte nicht abgeschlossen werden.', ['status' => 500]);
        }

        $pluginFile = '';
        if (method_exists($upgrader, 'plugin_info')) {
            $pluginFile = (string) $upgrader->plugin_info();
        }

        if ($pluginFile === '') {
            $plugins = function_exists('get_plugins') ? get_plugins() : [];
            foreach (array_keys((array) $plugins) as $candidate) {
                if (str_starts_with((string) $candidate, 'wpzeo-zero-seo/')) {
                    $pluginFile = (string) $candidate;
                    break;
                }
            }
        }

        return [
            'plugin_file' => $pluginFile,
        ];
    }
}
