<?php

declare(strict_types=1);

namespace Agent\Modules\Planning;

use Agent\Modules\OpenRouter\OpenRouterService;
use Agent\Support\Logger;
use Agent\Support\Options;
use WP_Error;

final class PlanningService
{
    public const OPTION_RUN = 'agent_planning_run';
    public const OPTION_SELECTED_SEO_PLUGIN = 'agent_planning_selected_seo_plugin';
    public const SCHEMA_VERSION = '1.0.0';

    private Options $options;
    private OpenRouterService $openRouter;
    private Logger $logger;

    public function __construct(Options $options, OpenRouterService $openRouter, Logger $logger)
    {
        $this->options = $options;
        $this->openRouter = $openRouter;
        $this->logger = $logger;
    }

    public function currentRun(): array
    {
        $run = get_option(self::OPTION_RUN, []);
        if (! is_array($run) || $run === []) {
            return $this->emptyRun();
        }

        if ($this->runNeedsRefresh($run)) {
            return $this->runGates();
        }

        return $run;
    }

    public function runGates(): array
    {
        $run = $this->emptyRun();
        $run['run_id'] = $this->newRunId();
        $run['status'] = 'running';
        $run['current_state'] = 'GATE_1_DEPENDENCIES';
        $run['started_at'] = gmdate('c');
        $run['updated_at'] = $run['started_at'];

        $run['gates']['dependencies'] = $this->validateDependencies();

        if ($run['gates']['dependencies']['status'] === 'blocked') {
            return $this->finalizeBlockedRun($run, 'GATE_1_DEPENDENCIES');
        }

        $run['current_state'] = 'GATE_2_CORE_BASELINE';
        $run['gates']['core_baseline'] = $this->validateCoreBaseline();
        if ($run['gates']['core_baseline']['status'] === 'blocked') {
            return $this->finalizeBlockedRun($run, 'GATE_2_CORE_BASELINE');
        }

        $run['current_state'] = 'GATE_3_OPENROUTER';
        $run['gates']['openrouter'] = $this->validateOpenRouter();
        if ($run['gates']['openrouter']['status'] === 'blocked') {
            return $this->finalizeBlockedRun($run, 'GATE_3_OPENROUTER');
        }

        $run['current_state'] = 'GATE_4_USER_INPUT';
        $run['status'] = 'ready_for_input';
        $run['updated_at'] = gmdate('c');
        $run['summary'] = $this->buildSummary($run);
        $this->persistRun($run);

        $this->logger->log('planning_gates_passed', 'info', [
            'run_id' => $run['run_id'],
            'warning_count' => (int) ($run['summary']['warning_count'] ?? 0),
        ]);

        return $run;
    }

    public function saveUserInput(array $input, int $userId): array|WP_Error
    {
        $run = $this->currentRun();
        if (! $this->canCollectInput($run)) {
            return new WP_Error('agent_planning_input_blocked', 'Planning-Eingaben sind erst nach erfolgreichen Gates erlaubt.', ['status' => 400]);
        }

        $data = $this->sanitizeInput($input);
        $run['gates']['user_input_v1'] = [
            'status' => 'completed',
            'collected_at' => gmdate('c'),
            'collected_by' => $userId,
            'data' => $data,
        ];
        $run['current_state'] = 'READY_FOR_PLANNING';
        $run['status'] = 'ready_for_planning';
        $run['updated_at'] = gmdate('c');
        $run['summary'] = $this->buildSummary($run);

        $this->persistRun($run);
        $this->logger->log('planning_input_saved', 'warning', [
            'run_id' => (string) ($run['run_id'] ?? ''),
            'by_user_id' => $userId,
        ]);

        return $run;
    }

    public function canCollectInput(array $run): bool
    {
        $depsOk = in_array((string) ($run['gates']['dependencies']['status'] ?? ''), ['passed', 'warning'], true);
        $coreOk = in_array((string) ($run['gates']['core_baseline']['status'] ?? ''), ['passed', 'warning'], true);
        $openRouterOk = in_array((string) ($run['gates']['openrouter']['status'] ?? ''), ['passed', 'warning'], true);

        return $depsOk && $coreOk && $openRouterOk;
    }

    private function validateDependencies(): array
    {
        $findings = [];
        $blockers = [];
        $warnings = [];

        $plugins = $this->allPlugins();
        $required = $this->requiredDependencyPlugins();

        foreach ($required as $dep) {
            $pluginFile = (string) ($dep['plugin'] ?? '');
            $label = (string) ($dep['label'] ?? $pluginFile);
            $minVersion = (string) ($dep['min_version'] ?? '');

            $installed = isset($plugins[$pluginFile]);
            $active = $this->isPluginActive($pluginFile);
            $version = $installed ? (string) ($plugins[$pluginFile]['Version'] ?? '') : '';

            if (! $installed) {
                $entry = $this->finding('dependency_missing', 'blocker', $label, 'Dieses Pflicht-Plugin fehlt. Ohne dieses Plugin kann wpAI nicht stabil planen.', $pluginFile, $this->pluginInstallAction($label));
                $findings[] = $entry;
                $blockers[] = $entry;
                continue;
            }

            if (! $active) {
                $entry = $this->finding('dependency_inactive', 'blocker', $label, 'Dieses Pflicht-Plugin ist installiert, aber deaktiviert. Bitte aktivieren.', $pluginFile, $this->pluginsPageAction());
                $findings[] = $entry;
                $blockers[] = $entry;
                continue;
            }

            if ($minVersion !== '' && $version !== '' && version_compare($version, $minVersion, '<')) {
                $entry = $this->finding('dependency_version_low', 'blocker', $label, 'Die installierte Version ist zu alt. Bitte auf die Mindestversion aktualisieren.', $version, $this->pluginsPageAction(), $minVersion);
                $findings[] = $entry;
                $blockers[] = $entry;
                continue;
            }

            $findings[] = $this->finding('dependency_ok', 'info', $label, 'Pflicht-Plugin ist aktiv und kompatibel.', $version, []);
        }

        if (! $this->hasAnySeoPlugin()) {
            $entry = $this->finding(
                'seo_plugin_missing',
                'blocker',
                'SEO Plugin',
                'Es ist kein unterstütztes SEO-Plugin aktiv. Bitte installiere und aktiviere wpZeo, Yoast oder Rank Math.',
                '',
                $this->pluginInstallAction('Yoast oder Rank Math installieren')
            );
            $findings[] = $entry;
            $blockers[] = $entry;
        } else {
            $installedSeo = $this->installedSeoPlugins();
            if (count($installedSeo) > 1) {
                $selectedSeo = (string) get_option(self::OPTION_SELECTED_SEO_PLUGIN, '');
                $installedLabels = [];
                $installedFiles = [];
                foreach ($installedSeo as $plugin) {
                    $installedLabels[] = (string) ($plugin['label'] ?? (string) ($plugin['plugin'] ?? ''));
                    $installedFiles[] = (string) ($plugin['plugin'] ?? '');
                }

                if ($selectedSeo === '' || ! in_array($selectedSeo, $installedFiles, true)) {
                    $entry = $this->finding(
                        'seo_multiple_installed',
                        'blocker',
                        'SEO Plugin Auswahl',
                        'Mehrere SEO-Plugins sind installiert. Lege jetzt ein primäres SEO-Plugin fest, damit Meta-Felder eindeutig geschrieben werden.',
                        implode(', ', $installedLabels),
                        $this->adminAction('SEO Plugins verwalten', 'plugins.php?s=seo')
                    );
                    $findings[] = $entry;
                    $blockers[] = $entry;
                } else {
                    $findings[] = $this->finding(
                        'seo_primary_selected',
                        'info',
                        'SEO Plugin Auswahl',
                        'Primäres SEO-Plugin ist festgelegt.',
                        $selectedSeo,
                        []
                    );
                }
            }

            $activeSeo = $this->activeSeoPlugins();
            if (count($activeSeo) > 1) {
                $labels = [];
                foreach ($activeSeo as $plugin) {
                    $labels[] = (string) ($plugin['label'] ?? (string) ($plugin['plugin'] ?? ''));
                }

                $entry = $this->finding(
                    'seo_multiple_active',
                    'blocker',
                    'SEO Plugin Konflikt',
                    'Mehrere SEO-Plugins sind gleichzeitig aktiv. Das führt zu widersprüchlichen Meta-Daten. Bitte genau eines aktiv lassen.',
                    implode(', ', $labels),
                    $this->adminAction('SEO Plugins verwalten', 'plugins.php?s=seo')
                );
                $findings[] = $entry;
                $blockers[] = $entry;
            }
        }

        return $this->gateResult('dependencies', $findings, $blockers, $warnings);
    }

    private function validateCoreBaseline(): array
    {
        $findings = [];
        $blockers = [];
        $warnings = [];

        $blogName = trim((string) get_option('blogname', ''));
        if ($blogName === '') {
            $entry = $this->finding('blogname_missing', 'blocker', 'Website-Name', 'Der Website-Name ist leer. Er wird für Planung, Markenführung und Seitentitel benötigt.', $blogName, $this->adminAction('Allgemein', 'options-general.php'));
            $findings[] = $entry;
            $blockers[] = $entry;
        } elseif ($this->isGenericSiteName($blogName)) {
            $entry = $this->finding(
                'blogname_generic',
                'warning',
                'Website-Name',
                'Der Website-Name wirkt generisch. Setze einen klaren Markennamen, damit Planung und SEO konsistent sind.',
                $blogName,
                $this->adminAction('Allgemein', 'options-general.php')
            );
            $findings[] = $entry;
            $warnings[] = $entry;
        }

        $tagline = trim((string) get_option('blogdescription', ''));
        if ($tagline === '') {
            $entry = $this->finding(
                'tagline_missing',
                'blocker',
                'Tagline',
                'Die Tagline ist leer. Für den Planungsmodus ist eine klare Kurzbeschreibung erforderlich.',
                $tagline,
                $this->adminAction('Allgemein', 'options-general.php')
            );
            $findings[] = $entry;
            $blockers[] = $entry;
        } elseif ($this->isGenericTagline($tagline)) {
            $entry = $this->finding(
                'tagline_generic',
                'warning',
                'Tagline',
                'Die Tagline ist sehr allgemein oder ein Standardtext. Formuliere den konkreten Nutzen der Website in einem Satz.',
                $tagline,
                $this->adminAction('Allgemein', 'options-general.php')
            );
            $findings[] = $entry;
            $warnings[] = $entry;
        }

        $permalink = (string) get_option('permalink_structure', '');
        if ($permalink === '' || $permalink === 'plain') {
            $entry = $this->finding('permalinks_plain', 'blocker', 'Permalinks', 'Permalinks sind auf "Einfach" gesetzt. Für saubere URL-Struktur und SEO bitte sprechende URLs aktivieren.', $permalink, $this->adminAction('Permalinks', 'options-permalink.php'));
            $findings[] = $entry;
            $blockers[] = $entry;
        }

        $siteUrl = (string) get_option('siteurl', '');
        $homeUrl = (string) get_option('home', '');
        if ($siteUrl === '' || $homeUrl === '' || rtrim($siteUrl, '/') !== rtrim($homeUrl, '/')) {
            $entry = $this->finding('url_mismatch', 'blocker', 'Site URL / Home URL', 'Site-URL und Home-URL unterscheiden sich. Das führt häufig zu fehlerhaften Links und API-Problemen.', $siteUrl . ' <> ' . $homeUrl, $this->adminAction('Allgemein', 'options-general.php'));
            $findings[] = $entry;
            $blockers[] = $entry;
        }

        $locale = function_exists('get_locale') ? (string) get_locale() : '';
        if ($locale === '') {
            $entry = $this->finding('locale_missing', 'warning', 'Sprache', 'Die WordPress-Sprache ist nicht sauber gesetzt. Inhalte können dadurch uneinheitlich werden.', '', $this->adminAction('Allgemein', 'options-general.php'));
            $findings[] = $entry;
            $warnings[] = $entry;
        }

        $timezone = (string) get_option('timezone_string', '');
        $offset = (string) get_option('gmt_offset', '');
        if ($timezone === '' && $offset === '') {
            $entry = $this->finding('timezone_missing', 'warning', 'Zeitzone', 'Die Zeitzone ist nicht eindeutig gesetzt. Zeitgesteuerte Abläufe können dadurch falsch laufen.', '', $this->adminAction('Allgemein', 'options-general.php'));
            $findings[] = $entry;
            $warnings[] = $entry;
        }

        $privacyPageId = (int) get_option('wp_page_for_privacy_policy', 0);
        if ($privacyPageId <= 0 || (function_exists('get_post_status') && get_post_status($privacyPageId) !== 'publish')) {
            $entry = $this->finding('privacy_policy_missing', 'warning', 'Datenschutzseite', 'Es gibt keine veröffentlichte Datenschutzseite. Für Live-Betrieb sollte diese Seite vorhanden sein.', (string) $privacyPageId, $this->adminAction('Datenschutz', 'options-privacy.php'));
            $findings[] = $entry;
            $warnings[] = $entry;
        }

        $restState = $this->restApiState();
        if ($restState['ok'] !== true) {
            $entry = $this->finding('rest_api_unavailable', 'blocker', 'REST API', 'Die REST API ist derzeit nicht erreichbar. Ohne API kann der Agent nicht arbeiten.', (string) ($restState['details'] ?? ''), []);
            $findings[] = $entry;
            $blockers[] = $entry;
        }

        return $this->gateResult('core_baseline', $findings, $blockers, $warnings);
    }

    private function validateOpenRouter(): array
    {
        $findings = [];
        $blockers = [];
        $warnings = [];
        $connectivity = [
            'attempted' => false,
            'ok' => false,
            'http_status' => 0,
            'error_code' => '',
            'error_message' => '',
            'checked_at' => gmdate('c'),
        ];

        $cfg = $this->openRouter->config();
        $enabled = (bool) ($cfg['enabled'] ?? false);
        $apiKey = (string) ($cfg['api_key'] ?? '');
        $plannerModel = trim((string) ($cfg['planner_model'] ?? ''));

        if ($enabled) {
            if ($apiKey === '') {
                $entry = $this->finding(
                    'openrouter_missing_key',
                    'blocker',
                    'OpenRouter API-Key',
                    'OpenRouter ist aktiviert, aber es fehlt ein API-Key. Ohne API-Key sind keine Modellabfragen möglich.',
                    '',
                    $this->adminAction('OpenRouter', 'admin.php?page=agent-openrouter')
                );
                $findings[] = $entry;
                $blockers[] = $entry;
            }
        } else {
            $entry = $this->finding(
                'openrouter_disabled',
                'blocker',
                'OpenRouter',
                'OpenRouter ist deaktiviert. Für planungsrelevante Modellabfragen muss OpenRouter aktiviert sein.',
                'disabled',
                $this->adminAction('OpenRouter', 'admin.php?page=agent-openrouter')
            );
            $findings[] = $entry;
            $blockers[] = $entry;

            if ($apiKey === '') {
                $findings[] = $this->finding(
                    'openrouter_missing_key_disabled_context',
                    'info',
                    'OpenRouter API-Key',
                    'Zusätzlich ist noch kein API-Key hinterlegt. Hinterlege den Key nach dem Aktivieren von OpenRouter.',
                    '',
                    $this->adminAction('OpenRouter', 'admin.php?page=agent-openrouter')
                );
            }
        }

        if ($plannerModel === '') {
            $entry = $this->finding('openrouter_missing_model', 'warning', 'Planer-Modell', 'Das Planer-Modell ist leer. Bitte ein konkretes Modell setzen, damit Ergebnisse reproduzierbar bleiben.', '', $this->adminAction('OpenRouter', 'admin.php?page=agent-openrouter'));
            $findings[] = $entry;
            $warnings[] = $entry;
        }

        if ($blockers === []) {
            $connectivity['attempted'] = true;
            $result = $this->openRouter->models();
            if ($result instanceof WP_Error) {
                $status = (int) ($result->data['status'] ?? 0);
                $code = (string) $result->code;
                $message = (string) $result->get_error_message();
                $connectivity['http_status'] = $status;
                $connectivity['error_code'] = $code;
                $connectivity['error_message'] = $message;

                if ($status === 429 || $code === 'agent_openrouter_rate_limited') {
                    $entry = $this->finding('openrouter_rate_limited', 'warning', 'OpenRouter Verbindung', 'OpenRouter hat das Request-Limit erreicht. Warte kurz und teste erneut.', $message, $this->adminAction('OpenRouter erneut testen', 'admin.php?page=agent-planning'));
                    $findings[] = $entry;
                    $warnings[] = $entry;
                } else {
                    $entry = $this->finding('openrouter_connectivity_failed', 'blocker', 'OpenRouter Verbindung', 'Der Verbindungstest zu OpenRouter ist fehlgeschlagen. Bitte Key, Modell und Endpoint prüfen.', $message, $this->adminAction('OpenRouter', 'admin.php?page=agent-openrouter'));
                    $findings[] = $entry;
                    $blockers[] = $entry;
                }
            } else {
                $connectivity['ok'] = true;
                $findings[] = $this->finding('openrouter_connected', 'info', 'OpenRouter Verbindung', 'OpenRouter-Verbindungstest erfolgreich.', 'ok', []);
            }
        }

        $result = $this->gateResult('openrouter', $findings, $blockers, $warnings);
        $result['connectivity_test'] = $connectivity;
        return $result;
    }

    private function buildSummary(array $run): array
    {
        $blockerCount = 0;
        $warningCount = 0;
        foreach (['dependencies', 'core_baseline', 'openrouter'] as $gate) {
            $blockerCount += count((array) ($run['gates'][$gate]['blockers'] ?? []));
            $warningCount += count((array) ($run['gates'][$gate]['warnings'] ?? []));
        }

        return [
            'blocker_count' => $blockerCount,
            'warning_count' => $warningCount,
            'can_proceed_to_planning' => $this->canCollectInput($run) && (($run['gates']['user_input_v1']['status'] ?? 'pending') === 'completed'),
            'can_collect_input' => $this->canCollectInput($run),
        ];
    }

    private function finalizeBlockedRun(array $run, string $state): array
    {
        $run['status'] = 'waiting_for_fix';
        $run['current_state'] = $state;
        $run['updated_at'] = gmdate('c');
        $run['summary'] = $this->buildSummary($run);
        $this->persistRun($run);

        $this->logger->log('planning_gate_blocked', 'warning', [
            'run_id' => $run['run_id'],
            'state' => $state,
            'blocker_count' => (int) ($run['summary']['blocker_count'] ?? 0),
        ]);

        return $run;
    }

    private function gateResult(string $gate, array $findings, array $blockers, array $warnings): array
    {
        $status = 'passed';
        if ($blockers !== []) {
            $status = 'blocked';
        } elseif ($warnings !== []) {
            $status = 'warning';
        }

        return [
            'gate' => $gate,
            'status' => $status,
            'checked_at' => gmdate('c'),
            'findings' => $findings,
            'blockers' => $blockers,
            'warnings' => $warnings,
            'actions' => $this->extractActions($findings),
        ];
    }

    private function extractActions(array $findings): array
    {
        $actions = [];
        foreach ($findings as $finding) {
            if (! is_array($finding)) {
                continue;
            }
            $action = is_array($finding['action'] ?? null) ? $finding['action'] : [];
            if (($action['label'] ?? '') === '') {
                continue;
            }
            $actions[] = $action;
        }

        return $actions;
    }

    private function finding(string $code, string $severity, string $label, string $message, string $value, array $action, string $expected = ''): array
    {
        return [
            'code' => $code,
            'severity' => $severity,
            'label' => $label,
            'message' => $message,
            'value' => $value,
            'expected' => $expected,
            'action' => $action,
        ];
    }

    private function restApiState(): array
    {
        if (! function_exists('rest_url')) {
            return ['ok' => false, 'details' => 'rest_url() unavailable'];
        }

        $url = rest_url();
        if (! function_exists('wp_remote_get')) {
            return ['ok' => $url !== '', 'details' => $url];
        }

        $response = wp_remote_get($url, ['timeout' => 10]);
        if (is_wp_error($response)) {
            return ['ok' => false, 'details' => $response->get_error_message()];
        }

        $status = (int) wp_remote_retrieve_response_code($response);
        return ['ok' => ($status >= 200 && $status < 400), 'details' => 'HTTP ' . (string) $status];
    }

    private function allPlugins(): array
    {
        if (! function_exists('get_plugins')) {
            if (defined('ABSPATH')) {
                $pluginFile = ABSPATH . 'wp-admin/includes/plugin.php';
                if (is_readable($pluginFile)) {
                    require_once $pluginFile;
                }
            }
        }

        if (! function_exists('get_plugins')) {
            return [];
        }

        $plugins = get_plugins();
        return is_array($plugins) ? $plugins : [];
    }

    private function isPluginActive(string $pluginFile): bool
    {
        if (! function_exists('is_plugin_active')) {
            if (defined('ABSPATH')) {
                $path = ABSPATH . 'wp-admin/includes/plugin.php';
                if (is_readable($path)) {
                    require_once $path;
                }
            }
        }

        if (! function_exists('is_plugin_active')) {
            return false;
        }

        return is_plugin_active($pluginFile);
    }

    private function requiredDependencyPlugins(): array
    {
        $required = [
            [
                'plugin' => 'wpai-agent/wpai-agent.php',
                'label' => 'Agent Bridge',
                'min_version' => '0.1.0',
            ],
        ];

        $filtered = apply_filters('agent_planning_required_plugins', $required);
        return is_array($filtered) ? $filtered : $required;
    }

    private function hasAnySeoPlugin(): bool
    {
        return $this->activeSeoPlugins() !== [];
    }

    private function installedSeoPlugins(): array
    {
        $plugins = $this->allPlugins();
        if ($plugins === []) {
            return [];
        }

        $matches = [];
        foreach ($plugins as $file => $meta) {
            $fileString = (string) $file;
            $candidate = strtolower($fileString);
            if (! (str_contains($candidate, 'wordpress-seo')
                || str_contains($candidate, 'wp-seo')
                || str_contains($candidate, 'rank-math')
                || str_contains($candidate, 'wpzeo'))
            ) {
                continue;
            }
            $matches[] = [
                'plugin' => $fileString,
                'label' => (string) ($meta['Name'] ?? $fileString),
            ];
        }

        return $matches;
    }

    private function activeSeoPlugins(): array
    {
        $plugins = $this->allPlugins();
        if ($plugins === []) {
            return [];
        }

        $matches = [];
        foreach ($plugins as $file => $meta) {
            $fileString = (string) $file;
            $candidate = strtolower($fileString);
            if (! (str_contains($candidate, 'wordpress-seo')
                || str_contains($candidate, 'wp-seo')
                || str_contains($candidate, 'rank-math')
                || str_contains($candidate, 'wpzeo'))
            ) {
                continue;
            }
            if (! $this->isPluginActive($fileString)) {
                continue;
            }
            $matches[] = [
                'plugin' => $fileString,
                'label' => (string) ($meta['Name'] ?? $fileString),
            ];
        }

        return $matches;
    }

    private function sanitizeInput(array $input): array
    {
        return [
            'project_goal' => sanitize_text_field((string) ($input['project_goal'] ?? '')),
            'primary_audience' => sanitize_text_field((string) ($input['primary_audience'] ?? '')),
            'geo_focus' => sanitize_text_field((string) ($input['geo_focus'] ?? '')),
            'main_services_or_topics' => sanitize_text_field((string) ($input['main_services_or_topics'] ?? '')),
            'brand_tone' => sanitize_text_field((string) ($input['brand_tone'] ?? '')),
            'conversion_goal' => sanitize_text_field((string) ($input['conversion_goal'] ?? '')),
            'content_depth' => sanitize_text_field((string) ($input['content_depth'] ?? '')),
            'constraints' => sanitize_textarea_field((string) ($input['constraints'] ?? '')),
        ];
    }

    private function persistRun(array $run): void
    {
        update_option(self::OPTION_RUN, $run, false);
    }

    private function emptyRun(): array
    {
        $settings = $this->options->all();

        return [
            'run_id' => '',
            'schema_version' => self::SCHEMA_VERSION,
            'plugin_version' => defined('AGENT_PLUGIN_VERSION') ? (string) AGENT_PLUGIN_VERSION : '0.1.0',
            'status' => 'idle',
            'current_state' => 'IDLE',
            'started_at' => null,
            'updated_at' => gmdate('c'),
            'finished_at' => null,
            'gates' => [
                'dependencies' => ['status' => 'pending', 'checked_at' => null, 'findings' => [], 'blockers' => [], 'warnings' => [], 'actions' => []],
                'core_baseline' => ['status' => 'pending', 'checked_at' => null, 'findings' => [], 'blockers' => [], 'warnings' => [], 'actions' => []],
                'openrouter' => ['status' => 'pending', 'checked_at' => null, 'findings' => [], 'blockers' => [], 'warnings' => [], 'actions' => [], 'connectivity_test' => ['attempted' => false, 'ok' => false, 'http_status' => 0, 'error_code' => '', 'error_message' => '', 'checked_at' => null]],
                'user_input_v1' => ['status' => 'pending', 'collected_at' => null, 'data' => []],
            ],
            'overrides' => [],
            'summary' => [
                'blocker_count' => 0,
                'warning_count' => 0,
                'can_proceed_to_planning' => false,
                'can_collect_input' => false,
            ],
            'planning_context_min' => [
                'site_url' => (string) get_option('siteurl', ''),
                'site_name' => (string) get_option('blogname', ''),
                'language' => function_exists('get_locale') ? (string) get_locale() : '',
                'timezone' => (string) ($settings['timezone'] ?? get_option('timezone_string', '')),
            ],
        ];
    }

    private function newRunId(): string
    {
        if (function_exists('wp_generate_uuid4')) {
            return (string) wp_generate_uuid4();
        }

        return 'run_' . md5((string) microtime(true) . '_' . (string) wp_rand());
    }

    private function adminAction(string $label, string $path): array
    {
        $url = function_exists('admin_url') ? admin_url($path) : $path;
        return ['label' => $label, 'url' => $url];
    }

    private function pluginsPageAction(): array
    {
        return $this->adminAction('Plugins öffnen', 'plugins.php');
    }

    private function pluginInstallAction(string $term): array
    {
        return $this->adminAction('Plugin suchen', 'plugin-install.php?s=' . rawurlencode($term));
    }

    private function isGenericSiteName(string $name): bool
    {
        $normalized = $this->normalizeGenericText($name);
        if ($normalized === '') {
            return true;
        }

        $siteUrl = (string) get_option('siteurl', '');
        $host = strtolower((string) wp_parse_url($siteUrl, PHP_URL_HOST));
        $host = preg_replace('/^www\./', '', $host);
        $hostNormalized = $this->normalizeGenericText((string) $host);
        if ($hostNormalized !== '' && $normalized === $hostNormalized) {
            return true;
        }

        $patterns = [
            'wordpress',
            'my blog',
            'mein blog',
            'meine website',
            'beispielseite',
            'example',
            'test',
            'testing',
            'staging',
            'dev',
            'deploy',
            'localhost',
        ];

        foreach ($patterns as $pattern) {
            if (str_contains($normalized, $this->normalizeGenericText($pattern))) {
                return true;
            }
        }

        return false;
    }

    private function isGenericTagline(string $tagline): bool
    {
        $normalized = $this->normalizeGenericText($tagline);
        if ($normalized === '') {
            return true;
        }

        $genericDefaults = [
            'just another wordpress site',
            'just another wp site',
            'eine weitere wordpress website',
            'eine weitere wordpress seite',
            'willkommen',
            'willkommen auf meiner website',
            'test',
            'staging',
            'deploy',
            'example',
        ];

        foreach ($genericDefaults as $candidate) {
            if ($normalized === $this->normalizeGenericText($candidate)) {
                return true;
            }
        }

        return false;
    }

    private function normalizeGenericText(string $input): string
    {
        $text = strtolower(trim($input));
        $text = preg_replace('/[^a-z0-9äöüß\s\-\.]/u', ' ', $text);
        $text = preg_replace('/[\s\.\-]+/u', ' ', (string) $text);
        return trim((string) $text);
    }

    private function runNeedsRefresh(array $run): bool
    {
        $openrouterGate = is_array($run['gates']['openrouter'] ?? null) ? (array) $run['gates']['openrouter'] : [];
        $blockers = is_array($openrouterGate['blockers'] ?? null) ? (array) $openrouterGate['blockers'] : [];
        if ($blockers === []) {
            return false;
        }

        $cfg = $this->openRouter->config();
        $enabled = (bool) ($cfg['enabled'] ?? false);
        $apiKeyPresent = ((string) ($cfg['api_key'] ?? '')) !== '';

        foreach ($blockers as $blocker) {
            if (! is_array($blocker)) {
                continue;
            }
            $code = (string) ($blocker['code'] ?? '');
            if ($code === 'openrouter_disabled' && $enabled) {
                return true;
            }
            if ($code === 'openrouter_missing_key' && ! $enabled) {
                return true;
            }
            if ($code === 'openrouter_missing_key' && $apiKeyPresent) {
                return true;
            }
        }

        return false;
    }
}
