<?php

declare(strict_types=1);

namespace Agent\Modules\OpenRouter;

use Agent\Modules\Seo\SeoMetaOrchestrator;
use Agent\Security\RequestGuard;
use Agent\Support\Logger;
use Agent\Support\Options;
use WP_Error;
use WP_REST_Request;
use WP_REST_Response;

final class OpenRouterController
{
    private OpenRouterService $service;
    private SeoMetaOrchestrator $seoOrchestrator;
    private RequestGuard $guard;
    private Logger $logger;
    private Options $options;

    public function __construct(OpenRouterService $service, SeoMetaOrchestrator $seoOrchestrator, RequestGuard $guard, Logger $logger, Options $options)
    {
        $this->service = $service;
        $this->seoOrchestrator = $seoOrchestrator;
        $this->guard = $guard;
        $this->logger = $logger;
        $this->options = $options;
    }

    public function registerRoutes(): void
    {
        register_rest_route(
            'wpai/v1',
            '/openrouter/models',
            [
                'methods' => 'GET',
                'callback' => [$this, 'models'],
                'permission_callback' => [$this, 'canRead'],
            ]
        );

        register_rest_route(
            'wpai/v1',
            '/openrouter/chat/completions',
            [
                'methods' => 'POST',
                'callback' => [$this, 'chatCompletions'],
                'permission_callback' => [$this, 'canRead'],
            ]
        );

        register_rest_route(
            'wpai/v1',
            '/openrouter/posts/generate',
            [
                'methods' => 'POST',
                'callback' => [$this, 'generatePost'],
                'permission_callback' => [$this, 'canRead'],
            ]
        );

        register_rest_route(
            'wpai/v1',
            '/openrouter/comments/workflow',
            [
                'methods' => 'POST',
                'callback' => [$this, 'commentWorkflow'],
                'permission_callback' => [$this, 'canRead'],
            ]
        );

        register_rest_route(
            'wpai/v1',
            '/openrouter/config',
            [
                [
                    'methods' => 'GET',
                    'callback' => [$this, 'getConfig'],
                    'permission_callback' => static fn (): bool => current_user_can('manage_options'),
                ],
                [
                    'methods' => 'POST',
                    'callback' => [$this, 'setConfig'],
                    'permission_callback' => static fn (): bool => current_user_can('manage_options'),
                ],
            ]
        );
    }

    public function canRead(): bool
    {
        return $this->guard->checkReadAccess();
    }

    public function models(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        if (($rate = $this->guard->assertRateLimit('openrouter_models')) instanceof WP_Error) {
            return $rate;
        }

        if (! $this->service->isEnabled()) {
            return new WP_Error('agent_openrouter_disabled', 'OpenRouter integration is disabled.', ['status' => 503]);
        }

        $result = $this->service->models();
        if ($result instanceof WP_Error) {
            return $result;
        }

        $this->logger->log('openrouter_models_requested', 'info');

        return new WP_REST_Response($result, 200);
    }

    public function chatCompletions(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        if (($rate = $this->guard->assertRateLimit('openrouter_chat')) instanceof WP_Error) {
            return $rate;
        }

        if (! $this->service->isEnabled()) {
            return new WP_Error('agent_openrouter_disabled', 'OpenRouter integration is disabled.', ['status' => 503]);
        }

        if (! $this->guard->checkWriteAccess()) {
            return new WP_Error('agent_write_disabled', 'Full write access is disabled.', ['status' => 403]);
        }

        $payload = $request->get_json_params();
        if (! is_array($payload)) {
            $payload = [];
        }

        $result = $this->service->chatCompletions($payload);
        if ($result instanceof WP_Error) {
            return $result;
        }

        $this->logger->log('openrouter_chat_completed', 'warning', [
            'model' => (string) ($payload['model'] ?? ''),
        ]);

        return new WP_REST_Response($result, 200);
    }

    public function generatePost(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        if (($rate = $this->guard->assertRateLimit('openrouter_generate_post')) instanceof WP_Error) {
            return $rate;
        }

        if (! $this->service->isEnabled()) {
            return new WP_Error('agent_openrouter_disabled', 'OpenRouter integration is disabled.', ['status' => 503]);
        }

        if (! $this->guard->checkWriteAccess()) {
            return new WP_Error('agent_write_disabled', 'Full write access is disabled.', ['status' => 403]);
        }

        $payload = $request->get_json_params();
        if (! is_array($payload)) {
            $payload = [];
        }

        $title = sanitize_text_field((string) ($payload['title'] ?? ''));
        $prompt = trim((string) ($payload['prompt'] ?? ''));
        if ($title === '' && $prompt === '') {
            return new WP_Error('agent_invalid_payload', 'Either "title" or "prompt" is required.', ['status' => 400]);
        }

        $postType = sanitize_key((string) ($payload['post_type'] ?? 'post'));
        if ($postType === '') {
            $postType = 'post';
        }

        $postStatus = sanitize_key((string) ($payload['post_status'] ?? 'draft'));
        if (! in_array($postStatus, ['draft', 'pending', 'private', 'publish'], true)) {
            $postStatus = 'draft';
        }

        $cfg = $this->service->config();
        $model = sanitize_text_field((string) ($payload['model'] ?? $cfg['writer_model'] ?? 'openrouter/free'));
        if ($model === '') {
            $model = 'openrouter/free';
        }

        $requestPayload = $this->buildPostGenerationPayload($model, $title, $prompt);
        $result = $this->service->chatCompletions($requestPayload);
        if ($result instanceof WP_Error) {
            return $result;
        }

        $rawText = trim($this->extractGeneratedText($result));
        if ($rawText === '') {
            return new WP_Error('agent_empty_generation', 'OpenRouter response did not contain generated content.', ['status' => 502]);
        }

        $parsed = $this->extractGeneratedPostPayload($rawText);
        $generatedContent = trim((string) ($parsed['content'] ?? $rawText));
        if ($generatedContent === '') {
            return new WP_Error('agent_empty_generation', 'Generated post content is empty.', ['status' => 502]);
        }

        $postTitle = sanitize_text_field((string) ($parsed['title'] ?? $title));
        if ($postTitle === '') {
            $postTitle = 'Generated Post ' . gmdate('Y-m-d H:i');
        }

        $postArr = [
            'post_type' => $postType,
            'post_status' => $postStatus,
            'post_title' => $postTitle,
            'post_content' => wp_kses_post($generatedContent),
            'post_author' => get_current_user_id(),
        ];

        $excerpt = sanitize_textarea_field((string) ($parsed['excerpt'] ?? ''));
        if ($excerpt !== '') {
            $postArr['post_excerpt'] = $excerpt;
        }

        $postId = wp_insert_post($postArr, true);
        if ($postId instanceof WP_Error) {
            return $postId;
        }

        $seoFromModel = is_array($parsed['seo'] ?? null) ? (array) $parsed['seo'] : [];
        $seoFromRequest = is_array($payload['seo'] ?? null) ? (array) $payload['seo'] : [];
        $seoData = $this->normalizeSeoData(array_merge($seoFromModel, $seoFromRequest));
        $seoReport = $this->seoOrchestrator->writeForPost((int) $postId, $seoData);

        $this->logger->log('openrouter_post_generated', 'warning', [
            'post_id' => (int) $postId,
            'post_type' => $postType,
            'post_status' => $postStatus,
            'model' => $model,
            'seo_target_plugin' => (string) ($seoReport['target_plugin'] ?? ''),
        ]);

        $response = [
            'generated_at' => gmdate('c'),
            'post_id' => (int) $postId,
            'post_type' => $postType,
            'post_status' => $postStatus,
            'title' => $postTitle,
            'model' => $model,
            'seo_input' => $seoData,
            'seo_result' => $seoReport,
        ];

        if (function_exists('get_permalink')) {
            $response['permalink'] = (string) get_permalink((int) $postId);
        }

        return new WP_REST_Response($response, 200);
    }

    public function commentWorkflow(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        if (($rate = $this->guard->assertRateLimit('openrouter_comment_workflow')) instanceof WP_Error) {
            return $rate;
        }

        if (! $this->service->isEnabled()) {
            return new WP_Error('agent_openrouter_disabled', 'OpenRouter integration is disabled.', ['status' => 503]);
        }

        if (! $this->guard->checkWriteAccess()) {
            return new WP_Error('agent_write_disabled', 'Full write access is disabled.', ['status' => 403]);
        }

        $payload = $request->get_json_params();
        if (! is_array($payload)) {
            $payload = [];
        }

        $postId = (int) ($payload['post_id'] ?? 0);
        if ($postId <= 0) {
            return new WP_Error('agent_invalid_payload', 'Field "post_id" is required.', ['status' => 400]);
        }

        $post = get_post($postId);
        if (! $post instanceof \WP_Post) {
            return new WP_Error('agent_post_not_found', 'Post not found.', ['status' => 404]);
        }

        $mode = sanitize_key((string) ($payload['mode'] ?? 'preview'));
        if (! in_array($mode, ['preview', 'publish'], true)) {
            $mode = 'preview';
        }

        $count = max(1, min(10, (int) ($payload['count'] ?? 1)));
        $tone = sanitize_text_field((string) ($payload['tone'] ?? 'helpful, concise, human'));
        $minWords = max(10, min(120, (int) ($payload['min_words'] ?? 25)));
        $maxWords = max($minWords, min(300, (int) ($payload['max_words'] ?? 70)));
        $model = sanitize_text_field((string) ($payload['model'] ?? ($this->service->config()['writer_model'] ?? 'openrouter/free')));
        if ($model === '') {
            $model = 'openrouter/free';
        }

        $generated = $this->generateCommentsForPost($post, $model, $count, $tone, $minWords, $maxWords);
        if ($generated instanceof WP_Error) {
            return $generated;
        }

        $authorName = sanitize_text_field((string) ($payload['comment_author'] ?? 'Agent'));
        $authorEmail = sanitize_email((string) ($payload['comment_author_email'] ?? 'agent@example.invalid'));
        $posted = [];

        if ($mode === 'publish') {
            foreach ($generated as $commentText) {
                $commentData = [
                    'comment_post_ID' => $postId,
                    'comment_content' => $commentText,
                    'comment_approved' => 1,
                    'comment_author' => $authorName !== '' ? $authorName : 'Agent',
                    'comment_author_email' => $authorEmail !== '' ? $authorEmail : 'agent@example.invalid',
                    'user_id' => get_current_user_id(),
                ];

                $commentId = wp_insert_comment($commentData);
                if ($commentId > 0) {
                    $posted[] = (int) $commentId;
                }
            }
        }

        $this->logger->log('openrouter_comment_workflow', 'warning', [
            'post_id' => $postId,
            'mode' => $mode,
            'count' => $count,
            'posted_count' => count($posted),
            'model' => $model,
        ]);

        return new WP_REST_Response(
            [
                'generated_at' => gmdate('c'),
                'post_id' => $postId,
                'mode' => $mode,
                'requested_count' => $count,
                'generated_comments' => $generated,
                'posted_comment_ids' => $posted,
                'posted_count' => count($posted),
                'model' => $model,
            ],
            200
        );
    }

    public function getConfig(WP_REST_Request $request): WP_REST_Response
    {
        $cfg = $this->service->config();
        $apiKey = (string) ($cfg['api_key'] ?? '');
        $cfg['api_key'] = $apiKey === '' ? '' : str_repeat('*', max(0, strlen($apiKey) - 4)) . substr($apiKey, -4);

        return new WP_REST_Response($cfg, 200);
    }

    public function setConfig(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        if (! $this->hasValidOptionalNonce($request)) {
            return new WP_Error('agent_invalid_nonce', 'Invalid nonce.', ['status' => 403]);
        }

        $payload = $request->get_json_params();
        if (! is_array($payload)) {
            $payload = [];
        }

        $current = (array) $this->options->get('openrouter', []);
        $current['enabled'] = array_key_exists('enabled', $payload) ? (bool) $payload['enabled'] : (bool) ($current['enabled'] ?? false);
        $current['base_url'] = isset($payload['base_url']) ? esc_url_raw((string) $payload['base_url']) : (string) ($current['base_url'] ?? 'https://openrouter.ai/api/v1');
        $current['site_url'] = isset($payload['site_url']) ? esc_url_raw((string) $payload['site_url']) : (string) ($current['site_url'] ?? '');
        $current['app_name'] = isset($payload['app_name']) ? sanitize_text_field((string) $payload['app_name']) : (string) ($current['app_name'] ?? 'Agent Bridge');
        $current['timeout'] = isset($payload['timeout']) ? max(5, min(180, (int) $payload['timeout'])) : max(5, (int) ($current['timeout'] ?? 60));
        $current['planner_model'] = isset($payload['planner_model']) ? sanitize_text_field((string) $payload['planner_model']) : (string) ($current['planner_model'] ?? 'openrouter/free');
        $current['writer_model'] = isset($payload['writer_model']) ? sanitize_text_field((string) $payload['writer_model']) : (string) ($current['writer_model'] ?? 'openrouter/free');
        $current['image_model'] = isset($payload['image_model']) ? sanitize_text_field((string) $payload['image_model']) : (string) ($current['image_model'] ?? 'openrouter/free');
        $current['model_filter_enabled'] = array_key_exists('model_filter_enabled', $payload) ? (bool) $payload['model_filter_enabled'] : (bool) ($current['model_filter_enabled'] ?? true);
        $current['model_filter'] = isset($payload['model_filter']) ? sanitize_text_field((string) $payload['model_filter']) : (string) ($current['model_filter'] ?? 'openai|x-ai|google|anthropic|meta-llama|free');

        if (isset($payload['api_key'])) {
            $current['api_key'] = sanitize_text_field((string) $payload['api_key']);
        }

        $this->options->set('openrouter', $current);
        $this->logger->log('openrouter_config_updated', 'warning', ['enabled' => (bool) $current['enabled']]);

        return new WP_REST_Response(['updated' => true, 'enabled' => (bool) $current['enabled']], 200);
    }

    private function hasValidOptionalNonce(WP_REST_Request $request): bool
    {
        $nonce = (string) $request->get_header('x_wp_nonce');
        if ($nonce === '') {
            return true;
        }

        return $this->guard->validateStateChangeNonce($nonce);
    }

    private function buildPostGenerationPayload(string $model, string $title, string $prompt): array
    {
        $siteName = function_exists('get_bloginfo') ? (string) get_bloginfo('name') : '';
        $siteTagline = function_exists('get_bloginfo') ? (string) get_bloginfo('description') : '';
        $siteLang = function_exists('get_bloginfo') ? (string) get_bloginfo('language') : '';

        $subject = $prompt !== '' ? $prompt : $title;

        $system = 'You generate WordPress post drafts and SEO metadata as strict JSON.'
            . ' Output JSON only, no markdown code fences.'
            . ' Required keys: title, content, excerpt, seo.'
            . ' SEO keys: seo_title, meta_description, focus_keyword, canonical, robots, og_title, og_description, twitter_title, twitter_description, og_image.';
        $user = 'Site name: ' . $siteName . "\n"
            . 'Tagline: ' . $siteTagline . "\n"
            . 'Language: ' . $siteLang . "\n"
            . 'Post subject: ' . $subject . "\n"
            . 'If title was provided, preserve it unless a better concise variant is clearly superior.'
            . ' Content should be production-ready and structured.';

        return [
            'model' => $model,
            'messages' => [
                ['role' => 'system', 'content' => $system],
                ['role' => 'user', 'content' => $user],
            ],
            'temperature' => 0.4,
            'max_tokens' => 2400,
        ];
    }

    private function extractGeneratedText(array $result): string
    {
        $choices = $result['choices'] ?? null;
        if (! is_array($choices) || $choices === []) {
            return '';
        }

        $first = $choices[0] ?? null;
        if (! is_array($first)) {
            return '';
        }

        $message = $first['message'] ?? null;
        if (is_array($message) && is_string($message['content'] ?? null)) {
            return (string) $message['content'];
        }

        if (is_string($first['text'] ?? null)) {
            return (string) $first['text'];
        }

        return '';
    }

    private function extractGeneratedPostPayload(string $raw): array
    {
        $decoded = json_decode($raw, true);
        if (is_array($decoded)) {
            return $decoded;
        }

        $start = strpos($raw, '{');
        $end = strrpos($raw, '}');
        if ($start === false || $end === false || $end <= $start) {
            return ['content' => $raw];
        }

        $json = substr($raw, $start, ($end - $start + 1));
        $decoded = json_decode($json, true);
        if (is_array($decoded)) {
            return $decoded;
        }

        return ['content' => $raw];
    }

    private function normalizeSeoData(array $input): array
    {
        $allowed = [
            'seo_title',
            'meta_description',
            'focus_keyword',
            'canonical',
            'robots',
            'og_title',
            'og_description',
            'twitter_title',
            'twitter_description',
            'og_image',
        ];

        $normalized = [];
        foreach ($allowed as $field) {
            if (! array_key_exists($field, $input)) {
                continue;
            }

            $value = trim((string) $input[$field]);
            if ($value === '') {
                continue;
            }

            $normalized[$field] = $value;
        }

        return $normalized;
    }

    private function generateCommentsForPost(\WP_Post $post, string $model, int $count, string $tone, int $minWords, int $maxWords): array|WP_Error
    {
        $title = sanitize_text_field((string) ($post->post_title ?? ''));
        $contentRaw = wp_strip_all_tags((string) ($post->post_content ?? ''));
        $contentPreview = function_exists('wp_trim_words')
            ? (string) wp_trim_words($contentRaw, 220, '...')
            : mb_substr($contentRaw, 0, 1400);

        $system = 'You write authentic website comments in the same language as the input.'
            . ' Output JSON only with key "comments" containing an array of strings.'
            . ' No markdown, no HTML, no numbering.';
        $user = 'Create ' . (string) $count . ' unique comments for this WordPress post.'
            . "\nTone: " . $tone
            . "\nWord range per comment: " . (string) $minWords . '-' . (string) $maxWords
            . "\nPost title: " . $title
            . "\nPost content excerpt:\n" . $contentPreview;

        $payload = [
            'model' => $model,
            'messages' => [
                ['role' => 'system', 'content' => $system],
                ['role' => 'user', 'content' => $user],
            ],
            'temperature' => 0.7,
            'max_tokens' => 1400,
        ];

        $result = $this->service->chatCompletions($payload);
        if ($result instanceof WP_Error) {
            return $result;
        }

        $raw = trim($this->extractGeneratedText($result));
        if ($raw === '') {
            return new WP_Error('agent_empty_generation', 'OpenRouter did not return comment content.', ['status' => 502]);
        }

        $decoded = json_decode($raw, true);
        if (! is_array($decoded)) {
            $decoded = $this->extractGeneratedPostPayload($raw);
        }

        $comments = $decoded['comments'] ?? null;
        if (! is_array($comments)) {
            $fallback = trim($raw);
            if ($fallback === '') {
                return new WP_Error('agent_empty_generation', 'No comment candidates found.', ['status' => 502]);
            }
            $comments = [$fallback];
        }

        $out = [];
        foreach ($comments as $item) {
            $text = trim(wp_strip_all_tags((string) $item));
            if ($text === '') {
                continue;
            }
            $out[] = $text;
            if (count($out) >= $count) {
                break;
            }
        }

        if ($out === []) {
            return new WP_Error('agent_empty_generation', 'Generated comments are empty after normalization.', ['status' => 502]);
        }

        return $out;
    }
}
