<?php

declare(strict_types=1);

namespace Agent\Modules\Jobs;

use Agent\Security\RequestGuard;
use Agent\Support\Logger;
use WP_Error;
use WP_REST_Request;
use WP_REST_Response;

final class JobsController
{
    private JobQueueService $jobs;
    private JobExecutionService $executor;
    private RequestGuard $guard;
    private Logger $logger;

    public function __construct(JobQueueService $jobs, JobExecutionService $executor, RequestGuard $guard, Logger $logger)
    {
        $this->jobs = $jobs;
        $this->executor = $executor;
        $this->guard = $guard;
        $this->logger = $logger;
    }

    public function registerRoutes(): void
    {
        register_rest_route(
            'wpai/v1',
            '/jobs/enqueue',
            [
                'methods' => 'POST',
                'callback' => [$this, 'enqueue'],
                'permission_callback' => [$this, 'canRead'],
            ]
        );

        register_rest_route(
            'wpai/v1',
            '/jobs/enqueue-batch',
            [
                'methods' => 'POST',
                'callback' => [$this, 'enqueueBatch'],
                'permission_callback' => [$this, 'canRead'],
            ]
        );

        register_rest_route(
            'wpai/v1',
            '/jobs/(?P<id>[\d]+)',
            [
                'methods' => 'GET',
                'callback' => [$this, 'getJob'],
                'permission_callback' => [$this, 'canRead'],
            ]
        );

        register_rest_route(
            'wpai/v1',
            '/jobs/queue-lines',
            [
                'methods' => 'GET',
                'callback' => [$this, 'queueLines'],
                'permission_callback' => [$this, 'canRead'],
            ]
        );

        register_rest_route(
            'wpai/v1',
            '/jobs/(?P<id>[\d]+)/retry',
            [
                'methods' => 'POST',
                'callback' => [$this, 'retryJob'],
                'permission_callback' => [$this, 'canRead'],
            ]
        );

        register_rest_route(
            'wpai/v1',
            '/jobs/(?P<id>[\d]+)/cancel',
            [
                'methods' => 'POST',
                'callback' => [$this, 'cancelJob'],
                'permission_callback' => [$this, 'canRead'],
            ]
        );

        register_rest_route(
            'wpai/v1',
            '/jobs/run-next',
            [
                'methods' => 'POST',
                'callback' => [$this, 'runNextJob'],
                'permission_callback' => [$this, 'canRead'],
            ]
        );
    }

    public function canRead(): bool
    {
        return $this->guard->checkReadAccess();
    }

    public function enqueue(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        if (($rate = $this->guard->assertRateLimit('jobs_enqueue')) instanceof WP_Error) {
            return $rate;
        }
        if (! $this->guard->checkWriteAccess()) {
            return new WP_Error('agent_write_disabled', 'Full write access is disabled.', ['status' => 403]);
        }

        $payload = $request->get_json_params();
        if (! is_array($payload)) {
            $payload = [];
        }

        $jobType = sanitize_key((string) ($payload['job_type'] ?? ''));
        $jobPayload = is_array($payload['payload'] ?? null) ? (array) $payload['payload'] : [];
        $scheduledAt = isset($payload['scheduled_at']) ? sanitize_text_field((string) $payload['scheduled_at']) : null;
        $queueKey = isset($payload['queue_key']) ? sanitize_key((string) $payload['queue_key']) : null;

        $job = $this->jobs->enqueue($jobType, $jobPayload, (int) get_current_user_id(), $scheduledAt, $queueKey);
        if ($job instanceof WP_Error) {
            return $job;
        }

        $this->logger->log('job_enqueued', 'warning', ['job_id' => (int) ($job['id'] ?? 0), 'job_type' => $jobType]);

        return new WP_REST_Response(['job' => $job], 200);
    }

    public function enqueueBatch(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        if (($rate = $this->guard->assertRateLimit('jobs_enqueue_batch')) instanceof WP_Error) {
            return $rate;
        }
        if (! $this->guard->checkWriteAccess()) {
            return new WP_Error('agent_write_disabled', 'Full write access is disabled.', ['status' => 403]);
        }

        $payload = $request->get_json_params();
        if (! is_array($payload)) {
            $payload = [];
        }

        $jobs = is_array($payload['jobs'] ?? null) ? array_values((array) $payload['jobs']) : [];
        $queueKey = isset($payload['queue_key']) ? sanitize_key((string) $payload['queue_key']) : null;
        $batch = $this->jobs->enqueueBatch($jobs, (int) get_current_user_id(), $queueKey);
        if ($batch instanceof WP_Error) {
            return $batch;
        }

        $this->logger->log(
            'job_batch_enqueued',
            'warning',
            [
                'queue_key' => (string) ($batch['queue_key'] ?? ''),
                'count' => (int) ($batch['count'] ?? 0),
            ]
        );

        return new WP_REST_Response(['batch' => $batch], 200);
    }

    public function getJob(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        if (($rate = $this->guard->assertRateLimit('jobs_get')) instanceof WP_Error) {
            return $rate;
        }

        $id = (int) $request->get_param('id');
        if ($id <= 0) {
            return new WP_Error('agent_invalid_job_id', 'Invalid job id.', ['status' => 400]);
        }

        $job = $this->jobs->get($id);
        if ($job === null) {
            return new WP_Error('agent_job_not_found', 'Job not found.', ['status' => 404]);
        }

        return new WP_REST_Response(['job' => $job], 200);
    }

    public function queueLines(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        if (($rate = $this->guard->assertRateLimit('jobs_queue_lines')) instanceof WP_Error) {
            return $rate;
        }

        $limit = (int) $request->get_param('limit');
        if ($limit <= 0) {
            $limit = 100;
        }

        $lines = $this->jobs->queueLines($limit);
        return new WP_REST_Response(['lines' => $lines], 200);
    }

    public function retryJob(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        if (($rate = $this->guard->assertRateLimit('jobs_retry')) instanceof WP_Error) {
            return $rate;
        }
        if (! $this->guard->checkWriteAccess()) {
            return new WP_Error('agent_write_disabled', 'Full write access is disabled.', ['status' => 403]);
        }

        $id = (int) $request->get_param('id');
        if ($id <= 0) {
            return new WP_Error('agent_invalid_job_id', 'Invalid job id.', ['status' => 400]);
        }

        $payload = $request->get_json_params();
        $override = null;
        if (is_array($payload) && is_array($payload['payload_override'] ?? null)) {
            $override = (array) $payload['payload_override'];
        }

        $job = $this->jobs->retry($id, $override);
        if ($job instanceof WP_Error) {
            return $job;
        }

        $this->logger->log('job_retried', 'warning', ['job_id' => $id]);
        return new WP_REST_Response(['job' => $job], 200);
    }

    public function cancelJob(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        if (($rate = $this->guard->assertRateLimit('jobs_cancel')) instanceof WP_Error) {
            return $rate;
        }
        if (! $this->guard->checkWriteAccess()) {
            return new WP_Error('agent_write_disabled', 'Full write access is disabled.', ['status' => 403]);
        }

        $id = (int) $request->get_param('id');
        if ($id <= 0) {
            return new WP_Error('agent_invalid_job_id', 'Invalid job id.', ['status' => 400]);
        }

        $job = $this->jobs->cancel($id);
        if ($job instanceof WP_Error) {
            return $job;
        }

        $this->logger->log('job_canceled', 'warning', ['job_id' => $id]);
        return new WP_REST_Response(['job' => $job], 200);
    }

    public function runNextJob(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        if (($rate = $this->guard->assertRateLimit('jobs_run_next')) instanceof WP_Error) {
            return $rate;
        }
        if (! $this->guard->checkWriteAccess()) {
            return new WP_Error('agent_write_disabled', 'Full write access is disabled.', ['status' => 403]);
        }

        $result = $this->jobs->runNext(fn (array $job): array|WP_Error => $this->executor->execute($job));
        if ($result instanceof WP_Error) {
            return $result;
        }
        if ($result === null) {
            return new WP_REST_Response(['processed' => false, 'message' => 'No queued jobs found.'], 200);
        }

        $status = (string) ($result['status'] ?? 'unknown');
        $severity = $status === 'completed' ? 'info' : 'warning';
        $this->logger->log('job_run_next_processed', $severity, ['job_id' => (int) ($result['id'] ?? 0), 'status' => $status, 'job_type' => (string) ($result['job_type'] ?? '')]);

        return new WP_REST_Response(['processed' => true, 'job' => $result], 200);
    }
}
