<?php

declare(strict_types=1);

namespace Agent\Modules\Jobs;

use WP_Error;

final class JobQueueService
{
    private const FALLBACK_OPTION = 'agent_jobs_fallback';
    private const WORKER_LOCK_KEY = 'agent_jobs_worker_lock';
    private const WORKER_LOCK_TTL = 60;

    public function enqueue(string $jobType, array $payload, int $userId, ?string $scheduledAt = null, ?string $queueKey = null): array|WP_Error
    {
        $jobType = sanitize_key($jobType);
        if ($jobType === '') {
            return new WP_Error('agent_invalid_job_type', 'Field "job_type" is required.', ['status' => 400]);
        }

        $now = gmdate('c');
        $scheduled = $scheduledAt !== null && $scheduledAt !== '' ? $scheduledAt : $now;
        $normalizedQueueKey = $this->normalizeQueueKey($queueKey);
        $job = [
            'id' => $this->nextFallbackId(),
            'job_type' => $jobType,
            'queue_key' => $normalizedQueueKey,
            'payload' => $payload,
            'status' => 'queued',
            'error_code' => '',
            'error_message' => '',
            'failed_step' => '',
            'attempts' => 0,
            'created_by' => $userId,
            'created_at' => $now,
            'scheduled_at' => $scheduled,
            'started_at' => null,
            'finished_at' => null,
        ];

        if ($this->isDbReady()) {
            global $wpdb;
            $table = $this->jobsTable();
            $inserted = $wpdb->insert(
                $table,
                [
                    'job_type' => $jobType,
                    'queue_key' => $normalizedQueueKey,
                    'payload_json' => (string) wp_json_encode($payload),
                    'status' => 'queued',
                    'error_code' => '',
                    'error_message' => '',
                    'failed_step' => '',
                    'attempts' => 0,
                    'created_by' => $userId,
                    'created_at' => $this->toDbDatetime($now),
                    'scheduled_at' => $this->toDbDatetime($scheduled),
                    'started_at' => null,
                    'finished_at' => null,
                ],
                ['%s', '%s', '%s', '%s', '%s', '%s', '%s', '%d', '%d', '%s', '%s', '%s', '%s']
            );

            if ($inserted === false) {
                return new WP_Error('agent_job_insert_failed', 'Failed to enqueue job.', ['status' => 500]);
            }

            $job['id'] = (int) $wpdb->insert_id;
            return $job;
        }

        $jobs = $this->fallbackJobs();
        $jobs[] = $job;
        update_option(self::FALLBACK_OPTION, $jobs, false);

        return $job;
    }

    public function enqueueBatch(array $items, int $userId, ?string $queueKey = null): array|WP_Error
    {
        if ($items === []) {
            return new WP_Error('agent_invalid_batch', 'Field "jobs" must contain at least one job.', ['status' => 400]);
        }
        if (count($items) > 200) {
            return new WP_Error('agent_batch_too_large', 'Batch size exceeds 200 jobs.', ['status' => 400]);
        }

        $resolvedQueueKey = $this->normalizeQueueKey($queueKey);
        if ($resolvedQueueKey === '') {
            $resolvedQueueKey = 'batch_' . gmdate('YmdHis') . '_' . substr((string) wp_rand(), 0, 8);
            $resolvedQueueKey = $this->normalizeQueueKey($resolvedQueueKey);
        }

        $normalizedItems = [];
        foreach ($items as $index => $item) {
            if (! is_array($item)) {
                return new WP_Error('agent_invalid_batch_item', 'Each batch job must be an object.', ['status' => 400, 'item_index' => $index]);
            }
            $jobType = sanitize_key((string) ($item['job_type'] ?? ''));
            if ($jobType === '') {
                return new WP_Error('agent_invalid_job_type', 'Field "job_type" is required for each batch item.', ['status' => 400, 'item_index' => $index]);
            }
            $payload = is_array($item['payload'] ?? null) ? (array) $item['payload'] : [];
            $scheduledAt = isset($item['scheduled_at']) ? sanitize_text_field((string) $item['scheduled_at']) : null;
            $itemQueue = isset($item['queue_key']) ? $this->normalizeQueueKey((string) $item['queue_key']) : '';

            $normalizedItems[] = [
                'job_type' => $jobType,
                'payload' => $payload,
                'scheduled_at' => $scheduledAt,
                'queue_key' => $itemQueue !== '' ? $itemQueue : $resolvedQueueKey,
            ];
        }

        $created = [];
        foreach ($normalizedItems as $item) {
            $job = $this->enqueue(
                (string) $item['job_type'],
                (array) $item['payload'],
                $userId,
                is_string($item['scheduled_at']) ? $item['scheduled_at'] : null,
                (string) $item['queue_key']
            );
            if ($job instanceof WP_Error) {
                return $job;
            }
            $created[] = $job;
        }

        return [
            'queue_key' => $resolvedQueueKey,
            'count' => count($created),
            'jobs' => $created,
        ];
    }

    public function get(int $id): ?array
    {
        if ($this->isDbReady()) {
            global $wpdb;
            $row = $wpdb->get_row(
                $wpdb->prepare(
                    'SELECT * FROM ' . $this->jobsTable() . ' WHERE id = %d LIMIT 1',
                    $id
                ),
                ARRAY_A
            );

            if (! is_array($row)) {
                return null;
            }

            return $this->normalizeDbRow($row);
        }

        $jobs = $this->fallbackJobs();
        foreach ($jobs as $job) {
            if ((int) ($job['id'] ?? 0) === $id) {
                return $job;
            }
        }

        return null;
    }

    public function cancel(int $id): array|WP_Error
    {
        if ($this->isDbReady()) {
            $job = $this->get($id);
            if ($job === null) {
                return new WP_Error('agent_job_not_found', 'Job not found.', ['status' => 404]);
            }

            $status = (string) ($job['status'] ?? '');
            if ($status === 'completed') {
                return new WP_Error('agent_job_completed', 'Completed jobs cannot be canceled.', ['status' => 409]);
            }
            if ($status === 'canceled') {
                return new WP_Error('agent_job_canceled', 'Job is already canceled.', ['status' => 409]);
            }

            global $wpdb;
            $updated = $wpdb->update(
                $this->jobsTable(),
                [
                    'status' => 'canceled',
                    'finished_at' => gmdate('Y-m-d H:i:s'),
                ],
                ['id' => $id],
                ['%s', '%s'],
                ['%d']
            );
            if ($updated === false) {
                return new WP_Error('agent_job_update_failed', 'Failed to cancel job.', ['status' => 500]);
            }

            $next = $this->get($id);
            return is_array($next) ? $next : $job;
        }

        $jobs = $this->fallbackJobs();
        foreach ($jobs as $index => $job) {
            if ((int) ($job['id'] ?? 0) !== $id) {
                continue;
            }

            $status = (string) ($job['status'] ?? '');
            if ($status === 'completed') {
                return new WP_Error('agent_job_completed', 'Completed jobs cannot be canceled.', ['status' => 409]);
            }
            if ($status === 'canceled') {
                return new WP_Error('agent_job_canceled', 'Job is already canceled.', ['status' => 409]);
            }

            $job['status'] = 'canceled';
            $job['finished_at'] = gmdate('c');
            $jobs[$index] = $job;
            update_option(self::FALLBACK_OPTION, $jobs, false);
            return $job;
        }

        return new WP_Error('agent_job_not_found', 'Job not found.', ['status' => 404]);
    }

    public function retry(int $id, ?array $payloadOverride = null): array|WP_Error
    {
        if ($this->isDbReady()) {
            $job = $this->get($id);
            if ($job === null) {
                return new WP_Error('agent_job_not_found', 'Job not found.', ['status' => 404]);
            }

            $status = (string) ($job['status'] ?? '');
            if (! in_array($status, ['failed', 'canceled'], true)) {
                return new WP_Error('agent_job_retry_invalid_state', 'Only failed or canceled jobs can be retried.', ['status' => 409]);
            }

            global $wpdb;
            $payloadJson = $payloadOverride !== null ? (string) wp_json_encode($payloadOverride) : (string) wp_json_encode((array) ($job['payload'] ?? []));
            $updated = $wpdb->update(
                $this->jobsTable(),
                [
                    'status' => 'queued',
                    'error_code' => '',
                    'error_message' => '',
                    'failed_step' => '',
                    'payload_json' => $payloadJson,
                    'started_at' => null,
                    'finished_at' => null,
                ],
                ['id' => $id],
                ['%s', '%s', '%s', '%s', '%s', '%s', '%s'],
                ['%d']
            );

            if ($updated === false) {
                return new WP_Error('agent_job_update_failed', 'Failed to retry job.', ['status' => 500]);
            }

            $next = $this->get($id);
            return is_array($next) ? $next : $job;
        }

        $jobs = $this->fallbackJobs();
        foreach ($jobs as $index => $job) {
            if ((int) ($job['id'] ?? 0) !== $id) {
                continue;
            }

            $status = (string) ($job['status'] ?? '');
            if (! in_array($status, ['failed', 'canceled'], true)) {
                return new WP_Error('agent_job_retry_invalid_state', 'Only failed or canceled jobs can be retried.', ['status' => 409]);
            }

            $job['status'] = 'queued';
            $job['error_code'] = '';
            $job['error_message'] = '';
            $job['failed_step'] = '';
            $job['started_at'] = null;
            $job['finished_at'] = null;
            if ($payloadOverride !== null) {
                $job['payload'] = $payloadOverride;
            }
            $jobs[$index] = $job;
            update_option(self::FALLBACK_OPTION, $jobs, false);
            return $job;
        }

        return new WP_Error('agent_job_not_found', 'Job not found.', ['status' => 404]);
    }

    public function runNext(?callable $executor = null): array|WP_Error|null
    {
        if (! $this->acquireWorkerLock()) {
            return new WP_Error('agent_job_worker_locked', 'Worker already running.', ['status' => 409]);
        }

        try {
            if ($this->isDbReady()) {
                return $this->runNextFromDb($executor);
            }

            return $this->runNextFallback($executor);
        } finally {
            $this->releaseWorkerLock();
        }
    }

    public function queueLines(int $limit = 100): array
    {
        $limit = max(1, min(500, $limit));

        if ($this->isDbReady()) {
            return $this->queueLinesFromDb($limit);
        }

        return $this->queueLinesFromFallback($limit);
    }

    public function hasPendingJobForPost(string $jobType, int $postId): bool
    {
        $jobType = sanitize_key($jobType);
        if ($jobType === '' || $postId <= 0) {
            return false;
        }

        if ($this->isDbReady()) {
            global $wpdb;
            $needle = '"post_id":' . (string) $postId;
            $count = $wpdb->get_var(
                $wpdb->prepare(
                    'SELECT COUNT(1) FROM ' . $this->jobsTable() . ' WHERE job_type = %s AND status IN (%s, %s) AND payload_json LIKE %s',
                    $jobType,
                    'queued',
                    'running',
                    '%' . $needle . '%'
                )
            );
            return is_numeric($count) && (int) $count > 0;
        }

        $jobs = $this->fallbackJobs();
        foreach ($jobs as $job) {
            if (! is_array($job)) {
                continue;
            }
            if ((string) ($job['job_type'] ?? '') !== $jobType) {
                continue;
            }
            $status = (string) ($job['status'] ?? '');
            if (! in_array($status, ['queued', 'running'], true)) {
                continue;
            }
            $payload = is_array($job['payload'] ?? null) ? (array) $job['payload'] : [];
            if ((int) ($payload['post_id'] ?? 0) === $postId) {
                return true;
            }
        }

        return false;
    }

    public function fail(int $id, string $errorCode, string $errorMessage, string $failedStep = ''): void
    {
        if ($this->isDbReady()) {
            global $wpdb;
            $wpdb->update(
                $this->jobsTable(),
                [
                    'status' => 'failed',
                    'error_code' => sanitize_key($errorCode),
                    'error_message' => sanitize_text_field($errorMessage),
                    'failed_step' => sanitize_key($failedStep),
                    'finished_at' => gmdate('Y-m-d H:i:s'),
                ],
                ['id' => $id],
                ['%s', '%s', '%s', '%s', '%s'],
                ['%d']
            );
            return;
        }

        $jobs = $this->fallbackJobs();
        foreach ($jobs as $index => $job) {
            if ((int) ($job['id'] ?? 0) !== $id) {
                continue;
            }

            $job['status'] = 'failed';
            $job['error_code'] = sanitize_key($errorCode);
            $job['error_message'] = sanitize_text_field($errorMessage);
            $job['failed_step'] = sanitize_key($failedStep);
            $job['finished_at'] = gmdate('c');
            $jobs[$index] = $job;
            update_option(self::FALLBACK_OPTION, $jobs, false);
            return;
        }
    }

    private function runNextFromDb(?callable $executor = null): array|WP_Error|null
    {
        global $wpdb;
        $now = gmdate('Y-m-d H:i:s');
        $blocked = $this->findBlockingQueueLineDb($now);
        if (is_array($blocked)) {
            return $this->queueBlockedError($blocked);
        }

        $row = $wpdb->get_row(
            $wpdb->prepare(
                'SELECT * FROM ' . $this->jobsTable() . ' WHERE status = %s AND scheduled_at <= %s ORDER BY id ASC LIMIT 1',
                'queued',
                $now
            ),
            ARRAY_A
        );

        if (! is_array($row)) {
            return null;
        }

        $id = (int) ($row['id'] ?? 0);
        if ($id <= 0) {
            return null;
        }

        $updated = $wpdb->query(
            $wpdb->prepare(
                'UPDATE ' . $this->jobsTable() . ' SET status = %s, started_at = %s, attempts = attempts + 1 WHERE id = %d AND status = %s',
                'running',
                $now,
                $id,
                'queued'
            )
        );
        if ($updated === false || (int) $updated < 1) {
            return null;
        }

        if ($executor !== null) {
            try {
                $job = $this->get($id);
                $result = is_array($job) ? $executor($job) : null;
                if ($result instanceof WP_Error) {
                    $failedStep = (string) ($result->data['failed_step'] ?? 'execute');
                    $this->fail($id, (string) $result->code, $result->get_error_message(), $failedStep);
                    $failedJob = $this->get($id);
                    return is_array($failedJob) ? $failedJob : $result;
                }
            } catch (\Throwable $e) {
                $this->fail($id, 'agent_job_execution_exception', $e->getMessage(), 'execute');
                $failedJob = $this->get($id);
                return is_array($failedJob) ? $failedJob : new WP_Error('agent_job_execution_exception', $e->getMessage(), ['status' => 500]);
            }
        }

        $wpdb->update(
            $this->jobsTable(),
            [
                'status' => 'completed',
                'finished_at' => gmdate('Y-m-d H:i:s'),
            ],
            ['id' => $id],
            ['%s', '%s'],
            ['%d']
        );

        $job = $this->get($id);
        return is_array($job) ? $job : null;
    }

    private function runNextFallback(?callable $executor = null): array|WP_Error|null
    {
        $jobs = $this->fallbackJobs();
        $blocked = $this->findBlockingQueueLineFallback($jobs);
        if (is_array($blocked)) {
            return $this->queueBlockedError($blocked);
        }

        foreach ($jobs as $index => $job) {
            if ((string) ($job['status'] ?? '') !== 'queued') {
                continue;
            }

            $job['status'] = 'running';
            $job['started_at'] = gmdate('c');
            $job['attempts'] = (int) ($job['attempts'] ?? 0) + 1;
            $jobs[$index] = $job;
            update_option(self::FALLBACK_OPTION, $jobs, false);

            if ($executor !== null) {
                try {
                    $result = $executor($job);
                    if ($result instanceof WP_Error) {
                        $job['status'] = 'failed';
                        $job['error_code'] = sanitize_key((string) $result->code);
                        $job['error_message'] = sanitize_text_field($result->get_error_message());
                        $job['failed_step'] = sanitize_key((string) ($result->data['failed_step'] ?? 'execute'));
                        $job['finished_at'] = gmdate('c');
                        $jobs[$index] = $job;
                        update_option(self::FALLBACK_OPTION, $jobs, false);
                        return $job;
                    }
                } catch (\Throwable $e) {
                    $job['status'] = 'failed';
                    $job['error_code'] = 'agent_job_execution_exception';
                    $job['error_message'] = sanitize_text_field($e->getMessage());
                    $job['failed_step'] = 'execute';
                    $job['finished_at'] = gmdate('c');
                    $jobs[$index] = $job;
                    update_option(self::FALLBACK_OPTION, $jobs, false);
                    return $job;
                }
            }

            $job['status'] = 'completed';
            $job['finished_at'] = gmdate('c');
            $jobs[$index] = $job;
            update_option(self::FALLBACK_OPTION, $jobs, false);

            return $job;
        }

        return null;
    }

    private function jobsTable(): string
    {
        global $wpdb;
        return $wpdb->prefix . 'ai_jobs';
    }

    private function isDbReady(): bool
    {
        global $wpdb;
        if (! isset($wpdb) || ! is_object($wpdb)) {
            return false;
        }
        if (! isset($wpdb->prefix) || ! is_string($wpdb->prefix) || $wpdb->prefix === '') {
            return false;
        }
        if (! method_exists($wpdb, 'insert') || ! method_exists($wpdb, 'get_row') || ! method_exists($wpdb, 'update') || ! method_exists($wpdb, 'prepare')) {
            return false;
        }

        return true;
    }

    private function normalizeDbRow(array $row): array
    {
        $payload = [];
        $decoded = json_decode((string) ($row['payload_json'] ?? ''), true);
        if (is_array($decoded)) {
            $payload = $decoded;
        }

        return [
            'id' => (int) ($row['id'] ?? 0),
            'job_type' => (string) ($row['job_type'] ?? ''),
            'queue_key' => (string) ($row['queue_key'] ?? ''),
            'payload' => $payload,
            'status' => (string) ($row['status'] ?? ''),
            'error_code' => (string) ($row['error_code'] ?? ''),
            'error_message' => (string) ($row['error_message'] ?? ''),
            'failed_step' => (string) ($row['failed_step'] ?? ''),
            'attempts' => (int) ($row['attempts'] ?? 0),
            'created_by' => (int) ($row['created_by'] ?? 0),
            'created_at' => $this->dbDatetimeToIso((string) ($row['created_at'] ?? '')),
            'scheduled_at' => $this->dbDatetimeToIso((string) ($row['scheduled_at'] ?? '')),
            'started_at' => $this->dbDatetimeToIsoNullable($row['started_at'] ?? null),
            'finished_at' => $this->dbDatetimeToIsoNullable($row['finished_at'] ?? null),
        ];
    }

    private function toDbDatetime(string $iso): string
    {
        $ts = strtotime($iso);
        if ($ts === false) {
            $ts = time();
        }
        return gmdate('Y-m-d H:i:s', $ts);
    }

    private function dbDatetimeToIso(string $value): string
    {
        $value = trim($value);
        if ($value === '') {
            return gmdate('c');
        }

        $ts = strtotime($value . ' UTC');
        if ($ts === false) {
            return gmdate('c');
        }

        return gmdate('c', $ts);
    }

    private function dbDatetimeToIsoNullable(mixed $value): ?string
    {
        if (! is_string($value) || trim($value) === '' || $value === '0000-00-00 00:00:00') {
            return null;
        }

        return $this->dbDatetimeToIso($value);
    }

    private function acquireWorkerLock(): bool
    {
        if (! function_exists('get_transient') || ! function_exists('set_transient')) {
            return true;
        }

        $current = get_transient(self::WORKER_LOCK_KEY);
        if ($current !== false && $current !== null && $current !== '') {
            return false;
        }

        return set_transient(self::WORKER_LOCK_KEY, gmdate('c'), self::WORKER_LOCK_TTL) === true;
    }

    private function releaseWorkerLock(): void
    {
        if (function_exists('delete_transient')) {
            delete_transient(self::WORKER_LOCK_KEY);
            return;
        }
        if (function_exists('set_transient')) {
            set_transient(self::WORKER_LOCK_KEY, '', 1);
        }
    }

    private function nextFallbackId(): int
    {
        $jobs = $this->fallbackJobs();
        $max = 0;
        foreach ($jobs as $job) {
            $id = (int) ($job['id'] ?? 0);
            if ($id > $max) {
                $max = $id;
            }
        }

        return $max + 1;
    }

    private function fallbackJobs(): array
    {
        $jobs = get_option(self::FALLBACK_OPTION, []);
        return is_array($jobs) ? array_values($jobs) : [];
    }

    private function queueLinesFromDb(int $limit): array
    {
        global $wpdb;

        $rows = $wpdb->get_results(
            $wpdb->prepare(
                'SELECT queue_key,
                        COUNT(*) AS total_jobs,
                        SUM(CASE WHEN status = %s THEN 1 ELSE 0 END) AS queued_count,
                        SUM(CASE WHEN status = %s THEN 1 ELSE 0 END) AS running_count,
                        SUM(CASE WHEN status = %s THEN 1 ELSE 0 END) AS completed_count,
                        SUM(CASE WHEN status = %s THEN 1 ELSE 0 END) AS failed_count,
                        SUM(CASE WHEN status = %s THEN 1 ELSE 0 END) AS canceled_count,
                        MIN(CASE WHEN status = %s THEN id ELSE NULL END) AS first_queued_id,
                        MIN(CASE WHEN status = %s THEN id ELSE NULL END) AS first_failed_id,
                        MAX(created_at) AS last_created_at
                 FROM ' . $this->jobsTable() . '
                 WHERE queue_key <> %s
                 GROUP BY queue_key
                 ORDER BY MAX(id) DESC
                 LIMIT %d',
                'queued',
                'running',
                'completed',
                'failed',
                'canceled',
                'queued',
                'failed',
                '',
                $limit
            ),
            ARRAY_A
        );

        if (! is_array($rows)) {
            return [];
        }

        $lines = [];
        foreach ($rows as $row) {
            if (! is_array($row)) {
                continue;
            }
            $firstQueuedId = (int) ($row['first_queued_id'] ?? 0);
            $firstFailedId = (int) ($row['first_failed_id'] ?? 0);
            $blocked = $firstQueuedId > 0 && $firstFailedId > 0 && $firstFailedId < $firstQueuedId;

            $lines[] = [
                'queue_key' => (string) ($row['queue_key'] ?? ''),
                'total_jobs' => (int) ($row['total_jobs'] ?? 0),
                'queued_count' => (int) ($row['queued_count'] ?? 0),
                'running_count' => (int) ($row['running_count'] ?? 0),
                'completed_count' => (int) ($row['completed_count'] ?? 0),
                'failed_count' => (int) ($row['failed_count'] ?? 0),
                'canceled_count' => (int) ($row['canceled_count'] ?? 0),
                'first_queued_id' => $firstQueuedId > 0 ? $firstQueuedId : null,
                'first_failed_id' => $firstFailedId > 0 ? $firstFailedId : null,
                'blocked' => $blocked,
                'last_created_at' => $this->dbDatetimeToIsoNullable($row['last_created_at'] ?? null),
            ];
        }

        return $lines;
    }

    private function queueLinesFromFallback(int $limit): array
    {
        $jobs = $this->fallbackJobs();
        $groups = [];

        foreach ($jobs as $job) {
            $queueKey = $this->normalizeQueueKey((string) ($job['queue_key'] ?? ''));
            if ($queueKey === '') {
                continue;
            }

            if (! isset($groups[$queueKey])) {
                $groups[$queueKey] = [
                    'queue_key' => $queueKey,
                    'total_jobs' => 0,
                    'queued_count' => 0,
                    'running_count' => 0,
                    'completed_count' => 0,
                    'failed_count' => 0,
                    'canceled_count' => 0,
                    'first_queued_id' => null,
                    'first_failed_id' => null,
                    'blocked' => false,
                    'last_created_at' => null,
                    '_max_id' => 0,
                ];
            }

            $entry = $groups[$queueKey];
            $entry['total_jobs']++;
            $status = (string) ($job['status'] ?? '');
            if ($status === 'queued') {
                $entry['queued_count']++;
            } elseif ($status === 'running') {
                $entry['running_count']++;
            } elseif ($status === 'completed') {
                $entry['completed_count']++;
            } elseif ($status === 'failed') {
                $entry['failed_count']++;
            } elseif ($status === 'canceled') {
                $entry['canceled_count']++;
            }

            $jobId = (int) ($job['id'] ?? 0);
            if ($status === 'queued' && $jobId > 0 && ($entry['first_queued_id'] === null || $jobId < (int) $entry['first_queued_id'])) {
                $entry['first_queued_id'] = $jobId;
            }
            if ($status === 'failed' && $jobId > 0 && ($entry['first_failed_id'] === null || $jobId < (int) $entry['first_failed_id'])) {
                $entry['first_failed_id'] = $jobId;
            }
            if ($jobId > (int) $entry['_max_id']) {
                $entry['_max_id'] = $jobId;
            }

            $createdAt = is_string($job['created_at'] ?? null) ? (string) $job['created_at'] : null;
            if ($createdAt !== null && ($entry['last_created_at'] === null || strtotime($createdAt) > strtotime((string) $entry['last_created_at']))) {
                $entry['last_created_at'] = $createdAt;
            }

            $groups[$queueKey] = $entry;
        }

        usort($groups, static function (array $a, array $b): int {
            return ((int) ($b['_max_id'] ?? 0)) <=> ((int) ($a['_max_id'] ?? 0));
        });
        $groups = array_slice($groups, 0, $limit);

        $lines = [];
        foreach ($groups as $entry) {
            $firstQueuedId = (int) ($entry['first_queued_id'] ?? 0);
            $firstFailedId = (int) ($entry['first_failed_id'] ?? 0);
            $entry['blocked'] = $firstQueuedId > 0 && $firstFailedId > 0 && $firstFailedId < $firstQueuedId;
            unset($entry['_max_id']);
            $lines[] = $entry;
        }

        return $lines;
    }

    private function normalizeQueueKey(?string $value): string
    {
        if (! is_string($value)) {
            return '';
        }

        return sanitize_key(trim($value));
    }

    private function findBlockingQueueLineDb(string $nowDb): ?array
    {
        global $wpdb;
        $row = $wpdb->get_row(
            $wpdb->prepare(
                'SELECT q.id AS blocked_job_id, q.queue_key AS queue_key, f.id AS failed_job_id
                 FROM ' . $this->jobsTable() . ' q
                 INNER JOIN ' . $this->jobsTable() . ' f
                   ON f.queue_key = q.queue_key
                  AND f.status = %s
                  AND f.id < q.id
                 WHERE q.status = %s
                   AND q.queue_key <> %s
                   AND q.scheduled_at <= %s
                 ORDER BY q.id ASC
                 LIMIT 1',
                'failed',
                'queued',
                '',
                $nowDb
            ),
            ARRAY_A
        );

        if (! is_array($row)) {
            return null;
        }

        return [
            'queue_key' => (string) ($row['queue_key'] ?? ''),
            'failed_job_id' => (int) ($row['failed_job_id'] ?? 0),
            'blocked_job_id' => (int) ($row['blocked_job_id'] ?? 0),
        ];
    }

    private function findBlockingQueueLineFallback(array $jobs): ?array
    {
        $nowTs = time();
        foreach ($jobs as $job) {
            if ((string) ($job['status'] ?? '') !== 'queued') {
                continue;
            }

            $scheduled = (string) ($job['scheduled_at'] ?? '');
            $scheduledTs = strtotime($scheduled);
            if ($scheduledTs !== false && $scheduledTs > $nowTs) {
                continue;
            }

            $queueKey = $this->normalizeQueueKey((string) ($job['queue_key'] ?? ''));
            if ($queueKey === '') {
                continue;
            }

            $jobId = (int) ($job['id'] ?? 0);
            if ($jobId <= 0) {
                continue;
            }

            foreach ($jobs as $candidate) {
                if ((string) ($candidate['status'] ?? '') !== 'failed') {
                    continue;
                }
                if ($this->normalizeQueueKey((string) ($candidate['queue_key'] ?? '')) !== $queueKey) {
                    continue;
                }
                $failedId = (int) ($candidate['id'] ?? 0);
                if ($failedId > 0 && $failedId < $jobId) {
                    return [
                        'queue_key' => $queueKey,
                        'failed_job_id' => $failedId,
                        'blocked_job_id' => $jobId,
                    ];
                }
            }
        }

        return null;
    }

    private function queueBlockedError(array $blocked): WP_Error
    {
        $queueKey = (string) ($blocked['queue_key'] ?? '');
        $failedJobId = (int) ($blocked['failed_job_id'] ?? 0);
        $blockedJobId = (int) ($blocked['blocked_job_id'] ?? 0);

        $message = 'Queue line';
        if ($queueKey !== '') {
            $message .= ' "' . $queueKey . '"';
        }
        $message .= ' is blocked by failed job #' . (string) $failedJobId . '. Retry or cancel the failed job before continuing (blocked job #' . (string) $blockedJobId . ').';

        return new WP_Error(
            'agent_queue_line_blocked',
            $message,
            [
                'status' => 409,
                'failed_step' => 'queue_guard',
                'queue_key' => $queueKey,
                'failed_job_id' => $failedJobId,
                'blocked_job_id' => $blockedJobId,
            ]
        );
    }
}
