<?php

declare(strict_types=1);

namespace Agent\Modules\Jobs;

use Agent\Modules\Indexing\ChunkPipelineService;
use Agent\Modules\OpenRouter\OpenRouterService;
use Agent\Modules\Seo\SeoMetaOrchestrator;
use Agent\Support\Logger;
use Agent\Support\Options;
use WP_Error;

final class JobExecutionService
{
    private OpenRouterService $openRouter;
    private SeoMetaOrchestrator $seo;
    private ChunkPipelineService $chunkPipeline;
    private Options $options;
    private Logger $logger;

    public function __construct(OpenRouterService $openRouter, SeoMetaOrchestrator $seo, ChunkPipelineService $chunkPipeline, Options $options, Logger $logger)
    {
        $this->openRouter = $openRouter;
        $this->seo = $seo;
        $this->chunkPipeline = $chunkPipeline;
        $this->options = $options;
        $this->logger = $logger;
    }

    public function execute(array $job): array|WP_Error
    {
        if ((bool) $this->options->get('emergency_lock', false) === true) {
            return new WP_Error('agent_emergency_lock', 'Emergency lock is active.', ['status' => 403, 'failed_step' => 'runtime']);
        }
        if ((bool) $this->options->get('full_write_access', false) !== true) {
            return new WP_Error('agent_write_disabled', 'Full write access is disabled.', ['status' => 403, 'failed_step' => 'runtime']);
        }

        $jobType = (string) ($job['job_type'] ?? '');
        $payload = is_array($job['payload'] ?? null) ? (array) $job['payload'] : [];

        // Only text/media generation jobs require OpenRouter availability.
        if (in_array($jobType, ['generate_post_page', 'generate_taxonomy_text', 'comment_workflow_publish'], true) && ! $this->openRouter->isEnabled()) {
            return new WP_Error('agent_openrouter_disabled', 'OpenRouter integration is disabled.', ['status' => 503, 'failed_step' => 'runtime']);
        }

        return match ($jobType) {
            'generate_post_page' => $this->runGeneratePostPage($payload),
            'generate_taxonomy_text' => $this->runGenerateTaxonomyText($payload),
            'comment_workflow_publish' => $this->runCommentWorkflowPublish($payload),
            'reindex_post' => $this->runReindexPost($payload),
            default => new WP_Error('agent_job_type_unsupported', 'Unsupported job type: ' . $jobType, ['status' => 400, 'failed_step' => 'dispatch']),
        };
    }

    private function runReindexPost(array $payload): array|WP_Error
    {
        $postId = (int) ($payload['post_id'] ?? 0);
        if ($postId <= 0) {
            return new WP_Error('agent_invalid_post_id', 'Field post_id is required.', ['status' => 400, 'failed_step' => 'validate']);
        }

        $result = $this->chunkPipeline->run(
            $postId,
            [
                'chunk_size' => (int) ($payload['chunk_size'] ?? 1200),
                'chunk_overlap' => (int) ($payload['chunk_overlap'] ?? 120),
                'embedding_model' => isset($payload['embedding_model']) ? sanitize_text_field((string) $payload['embedding_model']) : null,
                'allow_local_fallback' => array_key_exists('allow_local_fallback', $payload) ? (bool) $payload['allow_local_fallback'] : true,
            ]
        );
        if ($result instanceof WP_Error) {
            return $result;
        }

        $this->logger->log('job_reindex_post_completed', 'warning', ['post_id' => $postId, 'run_id' => (string) ($result['run_id'] ?? ''), 'trigger' => (string) ($payload['trigger'] ?? 'manual')]);

        return [
            'post_id' => $postId,
            'trigger' => (string) ($payload['trigger'] ?? 'manual'),
            'pipeline' => $result,
        ];
    }

    private function runGeneratePostPage(array $payload): array|WP_Error
    {
        $title = sanitize_text_field((string) ($payload['title'] ?? ''));
        $prompt = trim((string) ($payload['prompt'] ?? ''));
        if ($title === '' && $prompt === '') {
            return new WP_Error('agent_invalid_payload', 'Either title or prompt is required.', ['status' => 400, 'failed_step' => 'validate']);
        }

        $postType = sanitize_key((string) ($payload['post_type'] ?? 'post'));
        if (! in_array($postType, ['post', 'page'], true)) {
            return new WP_Error('agent_invalid_post_type', 'Only post and page are supported.', ['status' => 400, 'failed_step' => 'validate']);
        }

        $postStatus = sanitize_key((string) ($payload['post_status'] ?? 'draft'));
        if (! in_array($postStatus, ['draft', 'pending', 'private', 'publish'], true)) {
            $postStatus = 'draft';
        }

        $cfg = $this->openRouter->config();
        $model = sanitize_text_field((string) ($payload['model'] ?? $cfg['writer_model'] ?? 'openrouter/free'));
        if ($model === '') {
            $model = 'openrouter/free';
        }

        $requestPayload = $this->buildPostGenerationPayload($model, $title, $prompt);
        $result = $this->openRouter->chatCompletions($requestPayload);
        if ($result instanceof WP_Error) {
            return new WP_Error($result->code, $result->message, ['status' => (int) ($result->data['status'] ?? 502), 'failed_step' => 'generation']);
        }

        $rawText = trim($this->extractGeneratedText($result));
        if ($rawText === '') {
            return new WP_Error('agent_empty_generation', 'Generated content is empty.', ['status' => 502, 'failed_step' => 'generation']);
        }

        $parsed = $this->extractJsonPayload($rawText);
        $content = trim((string) ($parsed['content'] ?? $rawText));
        if ($content === '') {
            return new WP_Error('agent_empty_generation', 'Generated post body is empty.', ['status' => 502, 'failed_step' => 'generation']);
        }

        $finalTitle = sanitize_text_field((string) ($parsed['title'] ?? $title));
        if ($finalTitle === '') {
            $finalTitle = 'Generated ' . ucfirst($postType) . ' ' . gmdate('Y-m-d H:i');
        }

        $postArr = [
            'post_type' => $postType,
            'post_status' => $postStatus,
            'post_title' => $finalTitle,
            'post_content' => wp_kses_post($content),
            'post_author' => (int) ($payload['post_author'] ?? get_current_user_id()),
        ];

        $postId = wp_insert_post($postArr, true);
        if ($postId instanceof WP_Error) {
            return new WP_Error('agent_post_insert_failed', $postId->get_error_message(), ['status' => 500, 'failed_step' => 'persist']);
        }

        $seoFromModel = is_array($parsed['seo'] ?? null) ? (array) $parsed['seo'] : [];
        $seoFromPayload = is_array($payload['seo'] ?? null) ? (array) $payload['seo'] : [];
        $seoInput = $this->normalizeSeoData(array_merge($seoFromModel, $seoFromPayload));
        $seoResult = $this->seo->writeForPost((int) $postId, $seoInput);

        $this->logger->log('job_generate_post_page_completed', 'warning', ['post_id' => (int) $postId, 'post_type' => $postType, 'model' => $model]);

        return [
            'post_id' => (int) $postId,
            'post_type' => $postType,
            'post_status' => $postStatus,
            'title' => $finalTitle,
            'seo_result' => $seoResult,
        ];
    }

    private function runGenerateTaxonomyText(array $payload): array|WP_Error
    {
        $taxonomy = sanitize_key((string) ($payload['taxonomy'] ?? ''));
        if (! in_array($taxonomy, ['category', 'post_tag'], true)) {
            return new WP_Error('agent_invalid_taxonomy', 'Only category and post_tag are supported.', ['status' => 400, 'failed_step' => 'validate']);
        }

        $termId = (int) ($payload['term_id'] ?? 0);
        if ($termId <= 0) {
            return new WP_Error('agent_invalid_term_id', 'Field term_id is required.', ['status' => 400, 'failed_step' => 'validate']);
        }

        $term = get_term($termId, $taxonomy);
        if (! $term || is_wp_error($term)) {
            return new WP_Error('agent_term_not_found', 'Term not found.', ['status' => 404, 'failed_step' => 'validate']);
        }

        $brief = trim((string) ($payload['brief'] ?? ''));
        $cfg = $this->openRouter->config();
        $model = sanitize_text_field((string) ($payload['model'] ?? $cfg['writer_model'] ?? 'openrouter/free'));
        if ($model === '') {
            $model = 'openrouter/free';
        }

        $prompt = 'Write a concise taxonomy description for a WordPress ' . $taxonomy . " page.\n"
            . 'Term: ' . (string) ($term->name ?? '') . "\n"
            . 'Optional brief: ' . $brief . "\n"
            . 'Output plain text only.';
        $result = $this->openRouter->chatCompletions([
            'model' => $model,
            'messages' => [
                ['role' => 'system', 'content' => 'You write concise, factual taxonomy descriptions.'],
                ['role' => 'user', 'content' => $prompt],
            ],
            'temperature' => 0.3,
            'max_tokens' => 500,
        ]);
        if ($result instanceof WP_Error) {
            return new WP_Error($result->code, $result->message, ['status' => (int) ($result->data['status'] ?? 502), 'failed_step' => 'generation']);
        }

        $description = trim(wp_strip_all_tags($this->extractGeneratedText($result)));
        if ($description === '') {
            return new WP_Error('agent_empty_generation', 'Generated taxonomy description is empty.', ['status' => 502, 'failed_step' => 'generation']);
        }

        $updated = wp_update_term($termId, $taxonomy, ['description' => $description]);
        if (is_wp_error($updated)) {
            return new WP_Error('agent_term_update_failed', $updated->get_error_message(), ['status' => 500, 'failed_step' => 'persist']);
        }

        $this->logger->log('job_generate_taxonomy_text_completed', 'warning', ['taxonomy' => $taxonomy, 'term_id' => $termId, 'model' => $model]);
        return [
            'taxonomy' => $taxonomy,
            'term_id' => $termId,
            'description' => $description,
        ];
    }

    private function runCommentWorkflowPublish(array $payload): array|WP_Error
    {
        $postId = (int) ($payload['post_id'] ?? 0);
        if ($postId <= 0) {
            return new WP_Error('agent_invalid_payload', 'Field post_id is required.', ['status' => 400, 'failed_step' => 'validate']);
        }

        $post = get_post($postId);
        if (! $post instanceof \WP_Post) {
            return new WP_Error('agent_post_not_found', 'Post not found.', ['status' => 404, 'failed_step' => 'validate']);
        }

        $count = max(1, min(10, (int) ($payload['count'] ?? 1)));
        $tone = sanitize_text_field((string) ($payload['tone'] ?? 'helpful, concise, human'));
        $minWords = max(10, min(120, (int) ($payload['min_words'] ?? 25)));
        $maxWords = max($minWords, min(300, (int) ($payload['max_words'] ?? 70)));
        $cfg = $this->openRouter->config();
        $model = sanitize_text_field((string) ($payload['model'] ?? $cfg['writer_model'] ?? 'openrouter/free'));

        $comments = $this->generateCommentsForPost($post, $model, $count, $tone, $minWords, $maxWords);
        if ($comments instanceof WP_Error) {
            return $comments;
        }

        $authorName = sanitize_text_field((string) ($payload['comment_author'] ?? 'Agent'));
        $authorEmail = sanitize_email((string) ($payload['comment_author_email'] ?? 'agent@example.invalid'));
        $posted = [];
        foreach ($comments as $commentText) {
            $commentId = wp_insert_comment([
                'comment_post_ID' => $postId,
                'comment_content' => $commentText,
                'comment_approved' => 1,
                'comment_author' => $authorName !== '' ? $authorName : 'Agent',
                'comment_author_email' => $authorEmail !== '' ? $authorEmail : 'agent@example.invalid',
                'user_id' => get_current_user_id(),
            ]);
            if ($commentId > 0) {
                $posted[] = (int) $commentId;
            }
        }

        $this->logger->log('job_comment_workflow_publish_completed', 'warning', ['post_id' => $postId, 'posted_count' => count($posted), 'model' => $model]);
        return [
            'post_id' => $postId,
            'generated_comments' => $comments,
            'posted_comment_ids' => $posted,
            'posted_count' => count($posted),
        ];
    }

    private function buildPostGenerationPayload(string $model, string $title, string $prompt): array
    {
        $siteName = function_exists('get_bloginfo') ? (string) get_bloginfo('name') : '';
        $siteTagline = function_exists('get_bloginfo') ? (string) get_bloginfo('description') : '';
        $siteLang = function_exists('get_bloginfo') ? (string) get_bloginfo('language') : '';
        $subject = $prompt !== '' ? $prompt : $title;

        return [
            'model' => $model,
            'messages' => [
                ['role' => 'system', 'content' => 'You generate WordPress post drafts as strict JSON with keys title, content, excerpt, seo. Output JSON only.'],
                ['role' => 'user', 'content' => 'Site name: ' . $siteName . "\nTagline: " . $siteTagline . "\nLanguage: " . $siteLang . "\nPost subject: " . $subject],
            ],
            'temperature' => 0.4,
            'max_tokens' => 2400,
        ];
    }

    private function extractGeneratedText(array $result): string
    {
        $choices = $result['choices'] ?? null;
        if (! is_array($choices) || $choices === []) {
            return '';
        }
        $first = $choices[0] ?? null;
        if (! is_array($first)) {
            return '';
        }
        $message = $first['message'] ?? null;
        if (is_array($message) && is_string($message['content'] ?? null)) {
            return (string) $message['content'];
        }
        if (is_string($first['text'] ?? null)) {
            return (string) $first['text'];
        }
        return '';
    }

    private function extractJsonPayload(string $raw): array
    {
        $decoded = json_decode($raw, true);
        if (is_array($decoded)) {
            return $decoded;
        }
        $start = strpos($raw, '{');
        $end = strrpos($raw, '}');
        if ($start === false || $end === false || $end <= $start) {
            return ['content' => $raw];
        }
        $json = substr($raw, $start, ($end - $start + 1));
        $decoded = json_decode($json, true);
        return is_array($decoded) ? $decoded : ['content' => $raw];
    }

    private function normalizeSeoData(array $input): array
    {
        $allowed = ['seo_title', 'meta_description', 'focus_keyword', 'canonical', 'robots', 'og_title', 'og_description', 'twitter_title', 'twitter_description', 'og_image'];
        $normalized = [];
        foreach ($allowed as $field) {
            if (! array_key_exists($field, $input)) {
                continue;
            }
            $value = trim((string) $input[$field]);
            if ($value !== '') {
                $normalized[$field] = $value;
            }
        }
        return $normalized;
    }

    private function generateCommentsForPost(\WP_Post $post, string $model, int $count, string $tone, int $minWords, int $maxWords): array|WP_Error
    {
        $title = sanitize_text_field((string) ($post->post_title ?? ''));
        $contentRaw = wp_strip_all_tags((string) ($post->post_content ?? ''));
        $preview = function_exists('wp_trim_words') ? (string) wp_trim_words($contentRaw, 220, '...') : mb_substr($contentRaw, 0, 1400);

        $result = $this->openRouter->chatCompletions([
            'model' => $model !== '' ? $model : 'openrouter/free',
            'messages' => [
                ['role' => 'system', 'content' => 'Output JSON only: {\"comments\":[\"...\"]}. No markdown or HTML.'],
                ['role' => 'user', 'content' => 'Create ' . (string) $count . ' unique comments. Tone: ' . $tone . '. Word range per comment: ' . (string) $minWords . '-' . (string) $maxWords . '. Post title: ' . $title . '. Excerpt: ' . $preview],
            ],
            'temperature' => 0.7,
            'max_tokens' => 1400,
        ]);
        if ($result instanceof WP_Error) {
            return new WP_Error($result->code, $result->message, ['status' => (int) ($result->data['status'] ?? 502), 'failed_step' => 'generation']);
        }

        $raw = trim($this->extractGeneratedText($result));
        if ($raw === '') {
            return new WP_Error('agent_empty_generation', 'No generated comment content.', ['status' => 502, 'failed_step' => 'generation']);
        }
        $decoded = json_decode($raw, true);
        if (! is_array($decoded)) {
            $decoded = $this->extractJsonPayload($raw);
        }

        $comments = $decoded['comments'] ?? null;
        if (! is_array($comments)) {
            $comments = [$raw];
        }

        $out = [];
        foreach ($comments as $entry) {
            $text = trim(wp_strip_all_tags((string) $entry));
            if ($text === '') {
                continue;
            }
            $out[] = $text;
            if (count($out) >= $count) {
                break;
            }
        }
        if ($out === []) {
            return new WP_Error('agent_empty_generation', 'Generated comments are empty.', ['status' => 502, 'failed_step' => 'generation']);
        }

        return $out;
    }
}
