<?php

declare(strict_types=1);

namespace Agent\Modules\CoreAnalyzer;

final class CoreAnalyzerService
{
    public function generateReport(): array
    {
        global $wp_version;

        $activePlugins = (array) get_option('active_plugins', []);
        $seoPlugins = $this->detectSeoPlugins($activePlugins);
        $theme = wp_get_theme();
        $permalink = (string) get_option('permalink_structure', '');
        $restEnabled = $this->isRestApiEnabled();
        $debugEnabled = (bool) (defined('WP_DEBUG') && WP_DEBUG);
        $objectCacheEnabled = wp_using_ext_object_cache();
        $uploads = wp_get_upload_dir();

        $systemState = [
            'wordpress_version' => (string) $wp_version,
            'php_version' => PHP_VERSION,
            'active_plugins' => array_values($activePlugins),
            'seo_plugins' => $seoPlugins,
            'active_theme' => [
                'name' => $theme->get('Name'),
                'version' => $theme->get('Version'),
                'stylesheet' => $theme->get_stylesheet(),
                'template' => $theme->get_template(),
            ],
            'rest_api' => [
                'enabled' => $restEnabled,
                'base_url' => esc_url_raw(get_rest_url()),
            ],
            'permalinks' => [
                'structure' => $permalink,
                'pretty_permalinks' => $permalink !== '',
            ],
            'debug' => [
                'wp_debug' => $debugEnabled,
                'wp_debug_log' => (bool) (defined('WP_DEBUG_LOG') && WP_DEBUG_LOG),
                'wp_debug_display' => (bool) (defined('WP_DEBUG_DISPLAY') && WP_DEBUG_DISPLAY),
            ],
            'object_cache' => [
                'enabled' => $objectCacheEnabled,
            ],
            'server' => [
                'software' => sanitize_text_field((string) ($_SERVER['SERVER_SOFTWARE'] ?? 'unknown')),
                'sapi' => PHP_SAPI,
                'memory_limit' => ini_get('memory_limit'),
                'max_execution_time' => ini_get('max_execution_time'),
            ],
            'write_permissions' => [
                'wp_content_writable' => wp_is_writable(WP_CONTENT_DIR),
                'uploads_writable' => isset($uploads['basedir']) ? wp_is_writable((string) $uploads['basedir']) : false,
                'plugins_writable' => wp_is_writable(WP_PLUGIN_DIR),
            ],
        ];

        return [
            'generated_at' => gmdate('c'),
            'system_state' => $systemState,
            'security_risks' => $this->buildSecurityRisks($systemState),
            'performance_risks' => $this->buildPerformanceRisks($systemState),
            'api_availability' => $this->buildApiAvailability($systemState),
            'conflict_potential' => $this->buildConflictPotential($systemState),
        ];
    }

    private function isRestApiEnabled(): bool
    {
        $url = rest_url();
        return is_string($url) && $url !== '';
    }

    private function buildSecurityRisks(array $state): array
    {
        $risks = [];

        if (($state['debug']['wp_debug_display'] ?? false) === true) {
            $risks[] = [
                'id' => 'debug_display_enabled',
                'severity' => 'high',
                'message' => 'WP_DEBUG_DISPLAY is enabled and may expose internal details.',
            ];
        }

        if (($state['write_permissions']['plugins_writable'] ?? false) === true) {
            $risks[] = [
                'id' => 'plugins_dir_writable',
                'severity' => 'medium',
                'message' => 'Plugins directory is writable by PHP process.',
            ];
        }

        return $risks;
    }

    private function buildPerformanceRisks(array $state): array
    {
        $risks = [];

        if (($state['object_cache']['enabled'] ?? false) === false) {
            $risks[] = [
                'id' => 'no_persistent_object_cache',
                'severity' => 'medium',
                'message' => 'Persistent object cache is not active.',
            ];
        }

        if (count((array) ($state['active_plugins'] ?? [])) > 35) {
            $risks[] = [
                'id' => 'high_plugin_count',
                'severity' => 'medium',
                'message' => 'High number of active plugins may increase request overhead.',
            ];
        }

        return $risks;
    }

    private function buildApiAvailability(array $state): array
    {
        $available = (($state['rest_api']['enabled'] ?? false) === true) && (($state['permalinks']['pretty_permalinks'] ?? false) === true);

        return [
            'agent_api_ready' => $available,
            'rest_api_enabled' => (bool) ($state['rest_api']['enabled'] ?? false),
            'pretty_permalinks' => (bool) ($state['permalinks']['pretty_permalinks'] ?? false),
            'readonly_mode' => false,
        ];
    }

    private function buildConflictPotential(array $state): array
    {
        $plugins = (array) ($state['active_plugins'] ?? []);
        $conflicts = [];

        foreach ($plugins as $pluginFile) {
            $pluginFile = (string) $pluginFile;
            if (str_contains($pluginFile, 'security') || str_contains($pluginFile, 'firewall')) {
                $conflicts[] = [
                    'type' => 'plugin',
                    'slug' => $pluginFile,
                    'reason' => 'Security plugin may restrict REST requests or authentication flows.',
                ];
            }

            if (str_contains($pluginFile, 'cache') || str_contains($pluginFile, 'optimize')) {
                $conflicts[] = [
                    'type' => 'plugin',
                    'slug' => $pluginFile,
                    'reason' => 'Cache or optimization plugin may alter API responses or timings.',
                ];
            }
        }

        return $conflicts;
    }

    private function detectSeoPlugins(array $activePlugins): array
    {
        $activeSet = [];
        foreach ($activePlugins as $pluginFile) {
            $activeSet[(string) $pluginFile] = true;
        }

        $installed = $this->loadInstalledPlugins();
        foreach ($activePlugins as $pluginFile) {
            $key = (string) $pluginFile;
            if ($key !== '' && ! array_key_exists($key, $installed)) {
                $installed[$key] = ['Name' => '', 'TextDomain' => ''];
            }
        }
        $matches = [];

        foreach ($installed as $pluginFile => $pluginData) {
            $normalizedFile = strtolower((string) $pluginFile);
            $name = strtolower((string) ($pluginData['Name'] ?? ''));
            $textDomain = strtolower((string) ($pluginData['TextDomain'] ?? ''));
            $haystack = $normalizedFile . ' ' . $name . ' ' . $textDomain;

            $provider = '';
            $label = '';

            if (str_contains($haystack, 'wpzeo')) {
                $provider = 'wpzeo';
                $label = 'wpZeo';
            } elseif (str_contains($haystack, 'yoast')) {
                $provider = 'yoast';
                $label = 'Yoast SEO';
            } elseif (
                str_contains($haystack, 'rank-math')
                || str_contains($haystack, 'rank math')
                || (str_contains($haystack, 'math') && str_contains($haystack, 'seo'))
            ) {
                $provider = 'rank_math';
                $label = 'Rank Math';
            }

            if ($provider === '') {
                continue;
            }

            $matches[] = [
                'provider' => $provider,
                'label' => $label,
                'plugin_file' => (string) $pluginFile,
                'active' => isset($activeSet[(string) $pluginFile]),
            ];
        }

        $activeMatches = array_values(array_filter($matches, static fn (array $entry): bool => ($entry['active'] ?? false) === true));

        return [
            'supported_detected' => $matches !== [],
            'supported_active' => $activeMatches !== [],
            'installed' => array_values($matches),
            'active' => $activeMatches,
        ];
    }

    private function loadInstalledPlugins(): array
    {
        if (! function_exists('get_plugins')) {
            $pluginApiFile = defined('ABSPATH') ? ABSPATH . 'wp-admin/includes/plugin.php' : '';
            if ($pluginApiFile !== '' && file_exists($pluginApiFile)) {
                require_once $pluginApiFile;
            }
        }

        if (! function_exists('get_plugins')) {
            return [];
        }

        $plugins = get_plugins();
        return is_array($plugins) ? $plugins : [];
    }
}
