<?php

declare(strict_types=1);

namespace Agent\Modules\CoreAnalyzer;

use Agent\Security\RequestGuard;
use Agent\Support\Logger;
use Agent\Support\Options;
use WP_Error;
use WP_REST_Request;
use WP_REST_Response;

final class CoreAnalyzerController
{
    private CoreAnalyzerService $analyzer;
    private RequestGuard $guard;
    private Logger $logger;
    private Options $options;

    public function __construct(CoreAnalyzerService $analyzer, RequestGuard $guard, Logger $logger, Options $options)
    {
        $this->analyzer = $analyzer;
        $this->guard = $guard;
        $this->logger = $logger;
        $this->options = $options;
    }

    public function registerRoutes(): void
    {
        register_rest_route(
            'wpai/v1',
            '/core/report',
            [
                'methods' => 'GET',
                'callback' => [$this, 'coreReport'],
                'permission_callback' => [$this, 'canRead'],
            ]
        );

        register_rest_route(
            'wpai/v1',
            '/system/status',
            [
                'methods' => 'GET',
                'callback' => [$this, 'systemStatus'],
                'permission_callback' => [$this, 'canRead'],
            ]
        );

        register_rest_route(
            'wpai/v1',
            '/system/emergency-lock',
            [
                'methods' => 'POST',
                'callback' => [$this, 'setEmergencyLock'],
                'permission_callback' => static fn (): bool => current_user_can('manage_options'),
            ]
        );
    }

    public function canRead(): bool
    {
        return $this->guard->checkReadAccess();
    }

    public function coreReport(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        $rate = $this->guard->assertRateLimit('core_report');
        if ($rate instanceof WP_Error) {
            return $rate;
        }

        if (! (bool) $this->options->get('modules.core_analyzer', true)) {
            return new WP_Error('agent_module_disabled', 'Core analyzer module is disabled.', ['status' => 503]);
        }

        $report = $this->analyzer->generateReport();
        $report['api_availability']['readonly_mode'] = (bool) $this->options->get('readonly_mode', true);
        $report['api_availability']['full_write_access'] = (bool) $this->options->get('full_write_access', false);

        $this->logger->log('core_report_requested', 'info');

        return new WP_REST_Response($report, 200);
    }

    public function systemStatus(WP_REST_Request $request): WP_REST_Response
    {
        $this->logger->log('system_status_requested', 'info');

        return new WP_REST_Response(
            [
                'generated_at' => gmdate('c'),
                'emergency_lock' => (bool) $this->options->get('emergency_lock', false),
                'readonly_mode' => (bool) $this->options->get('readonly_mode', true),
                'full_write_access' => (bool) $this->options->get('full_write_access', false),
                'modules' => (array) $this->options->get('modules', []),
                'openrouter' => [
                    'enabled' => (bool) $this->options->get('openrouter.enabled', false),
                    'has_api_key' => ((string) $this->options->get('openrouter.api_key', '')) !== '',
                ],
                'rate_limit_per_minute' => (int) $this->options->get('rate_limit_per_minute', 60),
            ],
            200
        );
    }

    public function setEmergencyLock(WP_REST_Request $request): WP_REST_Response|WP_Error
    {
        if (! current_user_can('manage_options') || (bool) $this->options->get('full_write_access', false) !== true) {
            return new WP_Error('agent_write_disabled', 'Full write access is disabled.', ['status' => 403]);
        }

        $nonce = $request->get_header('x_wp_nonce');
        if ($nonce !== null && $nonce !== '' && ! $this->guard->validateStateChangeNonce($nonce)) {
            return new WP_Error('agent_invalid_nonce', 'Invalid nonce.', ['status' => 403]);
        }

        $value = (bool) $request->get_param('enabled');
        $this->options->set('emergency_lock', $value);
        $this->logger->log('emergency_lock_updated', 'warning', ['enabled' => $value]);

        return new WP_REST_Response(['emergency_lock' => $value], 200);
    }
}
