<?php

declare(strict_types=1);

namespace WpOrchestrator\Modules\WpAi;

use WpOrchestrator\Modules\Audit\AuditRepository;
use WpOrchestrator\Modules\Monitoring\HealthCheckRepository;
use WpOrchestrator\Modules\Transport\TransportClient;
use WpOrchestrator\Support\Logger;

final class WpAiSyncService
{
    public function __construct(
        private readonly TransportClient $transport,
        private readonly HealthCheckRepository $checks,
        private readonly AuditRepository $audit,
        private readonly Logger $logger
    ) {
    }

    /**
     * @param array<string, mixed> $target
     */
    public function sync(int $targetId, array $target): void
    {
        $baseUrl = untrailingslashit((string) ($target['base_url'] ?? ''));
        $authUser = (string) ($target['auth_user'] ?? '');
        $authSecret = (string) ($target['auth_secret'] ?? '');

        $this->syncSystemStatus($targetId, $baseUrl, $authUser, $authSecret);
        $this->syncAuditLogs($targetId, $baseUrl, $authUser, $authSecret);
    }

    private function syncSystemStatus(int $targetId, string $baseUrl, string $authUser, string $authSecret): void
    {
        $response = $this->transport->getJson($baseUrl . '/wp-json/wpai/v1/system/status', $authUser, $authSecret, 12, 2);

        $result = [
            'status' => $response['ok'] ? 'ok' : (string) $response['error_class'],
            'message' => $response['ok'] ? 'wpAI system status retrieved.' : (string) $response['error_message'],
            'http_code' => (int) $response['status_code'],
            'checked_at' => current_time('mysql'),
            'payload' => $response['json'],
        ];

        $score = $response['ok'] ? 100 : 0;

        $this->checks->add($targetId, 'wpai_system', $result, $score);
    }

    private function syncAuditLogs(int $targetId, string $baseUrl, string $authUser, string $authSecret): void
    {
        $response = $this->transport->getJson($baseUrl . '/wp-json/wpai/v1/logs/audit', $authUser, $authSecret, 12, 2);

        if (!$response['ok']) {
            $this->logger->error('wpAI audit sync failed.', [
                'target_id' => $targetId,
                'status_code' => $response['status_code'],
                'error_class' => $response['error_class'],
                'error_message' => $response['error_message'],
            ]);
            return;
        }

        $entries = $this->normalizeAuditEntries($response['json']);
        $imported = 0;

        foreach ($entries as $entry) {
            $actionRef = $this->resolveActionRef($targetId, $entry);
            $outcome = (string) ($entry['outcome'] ?? $entry['status'] ?? 'info');

            $this->audit->addRemoteAudit(
                $targetId,
                $actionRef,
                sanitize_key($outcome) !== '' ? sanitize_key($outcome) : 'info',
                [
                    'source' => 'wpai_remote',
                    'entry' => $entry,
                ]
            );
            $imported++;
        }

        $this->logger->info('wpAI audit sync completed.', [
            'target_id' => $targetId,
            'imported' => $imported,
            'fetched' => count($entries),
        ]);
    }

    /**
     * @param array<string, mixed>|null $payload
     * @return array<int, array<string, mixed>>
     */
    private function normalizeAuditEntries(?array $payload): array
    {
        if (!is_array($payload)) {
            return [];
        }

        if (isset($payload['items']) && is_array($payload['items'])) {
            return array_values(array_filter($payload['items'], 'is_array'));
        }

        if (isset($payload['logs']) && is_array($payload['logs'])) {
            return array_values(array_filter($payload['logs'], 'is_array'));
        }

        if (array_is_list($payload)) {
            return array_values(array_filter($payload, 'is_array'));
        }

        return [];
    }

    /**
     * @param array<string, mixed> $entry
     */
    private function resolveActionRef(int $targetId, array $entry): string
    {
        foreach (['id', 'event_id', 'action_ref', 'uuid'] as $key) {
            if (isset($entry[$key]) && $entry[$key] !== '') {
                return 'wpai:' . $targetId . ':' . sanitize_key((string) $entry[$key]);
            }
        }

        return 'wpai:' . $targetId . ':' . md5(wp_json_encode($entry));
    }
}
