<?php

declare(strict_types=1);

namespace WpOrchestrator\Modules\WpAi;

use WpOrchestrator\Modules\Actions\ActionRepository;
use WpOrchestrator\Modules\Audit\AuditRepository;
use WpOrchestrator\Modules\Monitoring\HealthCheckRepository;
use WpOrchestrator\Modules\Transport\TransportClient;
use WpOrchestrator\Support\Logger;

final class WpAiChangesetService
{
    public function __construct(
        private readonly TransportClient $transport,
        private readonly HealthCheckRepository $checks,
        private readonly ActionRepository $actions,
        private readonly AuditRepository $audit,
        private readonly Logger $logger
    ) {
    }

    /**
     * @param array<string, mixed> $target
     * @param array<string, mixed> $changeset
     * @return array{ok: bool, status: string, message: string, action_uuid: string}
     */
    public function apply(int $targetId, array $target, int $actorId, array $changeset): array
    {
        $baseUrl = untrailingslashit((string) ($target['base_url'] ?? ''));
        $authUser = (string) ($target['auth_user'] ?? '');
        $authSecret = (string) ($target['auth_secret'] ?? '');

        $actionUuid = $this->actions->start($targetId, 'wpai_changeset_apply', [
            'endpoint' => '/wp-json/wpai/v1/changeset/apply',
            'changeset' => $changeset,
        ]);

        $response = $this->transport->requestJson(
            method: 'POST',
            url: $baseUrl . '/wp-json/wpai/v1/changeset/apply',
            authUser: $authUser,
            authSecret: $authSecret,
            timeout: 30,
            maxAttempts: 2,
            body: [
                'source' => 'wp-orchestrator',
                'action_uuid' => $actionUuid,
                'changeset' => $changeset,
            ]
        );

        $ok = (bool) $response['ok'];
        $status = $ok ? 'success' : (string) $response['error_class'];
        $message = $ok
            ? __('wpAI changeset applied.', 'wp-orchestrator')
            : ((string) $response['error_message'] !== '' ? (string) $response['error_message'] : __('Unable to apply wpAI changeset.', 'wp-orchestrator'));

        $payload = [
            'status' => $status,
            'message' => $message,
            'http_code' => (int) $response['status_code'],
            'response' => $response['json'],
            'action_uuid' => $actionUuid,
            'changeset' => $changeset,
        ];

        $this->actions->finish($actionUuid, $ok ? 'success' : 'failed', $payload);
        $this->checks->add($targetId, 'wpai_changeset_apply', $payload, $ok ? 100 : 0);

        $this->audit->addLocalAudit(
            $targetId,
            $actorId,
            'local:' . $actionUuid,
            $ok ? 'success' : 'failed',
            [
                'action_type' => 'wpai_changeset_apply',
                'status' => $status,
                'http_code' => (int) $response['status_code'],
            ]
        );

        $this->logger->info('wpAI changeset apply executed.', [
            'target_id' => $targetId,
            'action_uuid' => $actionUuid,
            'status' => $status,
            'http_code' => $response['status_code'],
        ]);

        return [
            'ok' => $ok,
            'status' => $status,
            'message' => $message,
            'action_uuid' => $actionUuid,
        ];
    }
}
