<?php

declare(strict_types=1);

namespace WpOrchestrator\Modules\Transport;

final class TransportClient
{
    /**
     * @return array{ok: bool, status_code: int, error_class: string, error_message: string, body: string, json: array<string, mixed>|null, attempts: int}
     */
    public function getJson(
        string $url,
        string $authUser,
        string $authSecret,
        int $timeout = 12,
        int $maxAttempts = 3
    ): array {
        return $this->requestJson('GET', $url, $authUser, $authSecret, $timeout, $maxAttempts);
    }

    /**
     * @return array{ok: bool, status_code: int, error_class: string, error_message: string, body: string, json: array<string, mixed>|null, attempts: int}
     */
    public function requestJson(
        string $method,
        string $url,
        string $authUser,
        string $authSecret,
        int $timeout = 12,
        int $maxAttempts = 3,
        ?array $body = null
    ): array {
        $attempt = 0;
        $maxAttempts = max(1, $maxAttempts);

        while ($attempt < $maxAttempts) {
            $attempt++;

            $args = [
                'method' => strtoupper($method),
                'timeout' => $timeout,
                'redirection' => 2,
                'headers' => [
                    'Authorization' => 'Basic ' . base64_encode($authUser . ':' . $authSecret),
                    'Accept' => 'application/json',
                ],
            ];

            if ($body !== null) {
                $args['headers']['Content-Type'] = 'application/json';
                $args['body'] = wp_json_encode($body);
            }

            $response = wp_remote_request($url, $args);

            if (is_wp_error($response)) {
                $errorClass = $this->mapWpErrorClass((string) $response->get_error_code());
                $message = (string) $response->get_error_message();
                $retryable = in_array($errorClass, ['timeout', 'dns_error', 'tls_error', 'unreachable'], true);

                if ($retryable && $attempt < $maxAttempts) {
                    usleep($this->backoffMicroseconds($attempt));
                    continue;
                }

                return [
                    'ok' => false,
                    'status_code' => 0,
                    'error_class' => $errorClass,
                    'error_message' => $message,
                    'body' => '',
                    'json' => null,
                    'attempts' => $attempt,
                ];
            }

            $statusCode = (int) wp_remote_retrieve_response_code($response);
            $bodyText = (string) wp_remote_retrieve_body($response);
            $decoded = json_decode($bodyText, true);
            $json = is_array($decoded) ? $decoded : null;

            if ($statusCode >= 200 && $statusCode < 300) {
                return [
                    'ok' => true,
                    'status_code' => $statusCode,
                    'error_class' => 'none',
                    'error_message' => '',
                    'body' => $bodyText,
                    'json' => $json,
                    'attempts' => $attempt,
                ];
            }

            $errorClass = $this->mapHttpErrorClass($statusCode);
            $retryable = $statusCode === 429 || $statusCode >= 500;

            if ($retryable && $attempt < $maxAttempts) {
                usleep($this->backoffMicroseconds($attempt));
                continue;
            }

            $message = sprintf(
                /* translators: %d is an HTTP status code */
                __('Unexpected HTTP response: %d', 'wp-orchestrator'),
                $statusCode
            );

            return [
                'ok' => false,
                'status_code' => $statusCode,
                'error_class' => $errorClass,
                'error_message' => $message,
                'body' => $bodyText,
                'json' => $json,
                'attempts' => $attempt,
            ];
        }

        return [
            'ok' => false,
            'status_code' => 0,
            'error_class' => 'unreachable',
            'error_message' => __('Request attempts exhausted.', 'wp-orchestrator'),
            'body' => '',
            'json' => null,
            'attempts' => $attempt,
        ];
    }

    private function backoffMicroseconds(int $attempt): int
    {
        $milliseconds = 200 * (2 ** max(0, $attempt - 1));
        return min(1500, $milliseconds) * 1000;
    }

    private function mapWpErrorClass(string $wpErrorCode): string
    {
        $code = strtolower($wpErrorCode);

        if (str_contains($code, 'ssl') || str_contains($code, 'tls')) {
            return 'tls_error';
        }

        if (str_contains($code, 'dns') || str_contains($code, 'resolve')) {
            return 'dns_error';
        }

        if (str_contains($code, 'timeout')) {
            return 'timeout';
        }

        return 'unreachable';
    }

    private function mapHttpErrorClass(int $statusCode): string
    {
        if ($statusCode === 401 || $statusCode === 403) {
            return 'auth_failed';
        }

        if ($statusCode === 429) {
            return 'rate_limited';
        }

        if ($statusCode >= 500) {
            return 'upstream_5xx';
        }

        if ($statusCode >= 400) {
            return 'http_4xx';
        }

        return 'http_error';
    }
}
