<?php

declare(strict_types=1);

namespace WpOrchestrator\Modules\Targets;

use WpOrchestrator\Support\SecretBox;

final class TargetRepository
{
    private SecretBox $secretBox;

    public function __construct()
    {
        $this->secretBox = new SecretBox();
    }

    /**
     * @return array<int, array<string, mixed>>
     */
    public function all(): array
    {
        global $wpdb;

        $table = $wpdb->prefix . 'wpo_targets';

        $rows = $wpdb->get_results(
            "SELECT id, name, base_url, auth_user, tags, status, wpai_available, capabilities_json, last_seen_at, created_at, updated_at
             FROM {$table}
             ORDER BY id DESC",
            ARRAY_A
        );

        return is_array($rows) ? $rows : [];
    }

    /**
     * @return array<int, array<string, mixed>>
     */
    public function allMonitorable(): array
    {
        global $wpdb;

        $table = $wpdb->prefix . 'wpo_targets';

        $rows = $wpdb->get_results(
            "SELECT id, name, base_url, auth_user, tags, status, wpai_available, capabilities_json, last_seen_at, created_at, updated_at
             FROM {$table}
             WHERE status <> 'disabled'
             ORDER BY id DESC",
            ARRAY_A
        );

        return is_array($rows) ? $rows : [];
    }

    public function create(string $name, string $baseUrl, string $authUser, string $authSecret, string $tags = ''): int
    {
        global $wpdb;

        $table = $wpdb->prefix . 'wpo_targets';
        $now = current_time('mysql');

        $inserted = $wpdb->insert(
            $table,
            [
                'name' => $name,
                'base_url' => untrailingslashit($baseUrl),
                'auth_user' => $authUser,
                'auth_secret_enc' => $this->secretBox->encrypt($authSecret),
                'tags' => $tags,
                'status' => 'active',
                'wpai_available' => 0,
                'capabilities_json' => null,
                'last_seen_at' => null,
                'created_at' => $now,
                'updated_at' => $now,
            ],
            [
                '%s', '%s', '%s', '%s', '%s', '%s', '%d', '%s', '%s', '%s', '%s',
            ]
        );

        if ($inserted !== 1) {
            throw new \RuntimeException('Unable to persist target instance.');
        }

        return (int) $wpdb->insert_id;
    }

    /**
     * @return array<string, mixed>|null
     */
    public function findById(int $id): ?array
    {
        global $wpdb;

        $table = $wpdb->prefix . 'wpo_targets';
        $row = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT id, name, base_url, auth_user, auth_secret_enc, tags, status, last_seen_at, created_at, updated_at
                        , wpai_available, capabilities_json
                 FROM {$table}
                 WHERE id = %d",
                $id
            ),
            ARRAY_A
        );

        if (!is_array($row)) {
            return null;
        }

        $row['auth_secret'] = $this->secretBox->decrypt((string) $row['auth_secret_enc']);
        unset($row['auth_secret_enc']);

        return $row;
    }

    public function updateStatus(int $id, string $status, ?string $lastSeenAt = null): void
    {
        global $wpdb;

        $table = $wpdb->prefix . 'wpo_targets';

        $wpdb->update(
            $table,
            [
                'status' => $status,
                'last_seen_at' => $lastSeenAt,
                'updated_at' => current_time('mysql'),
            ],
            ['id' => $id],
            ['%s', '%s', '%s'],
            ['%d']
        );
    }

    /**
     * @param array<string, mixed> $capabilities
     */
    public function updateCapabilities(int $id, bool $wpaiAvailable, array $capabilities): void
    {
        global $wpdb;

        $table = $wpdb->prefix . 'wpo_targets';

        $wpdb->update(
            $table,
            [
                'wpai_available' => $wpaiAvailable ? 1 : 0,
                'capabilities_json' => wp_json_encode($capabilities),
                'updated_at' => current_time('mysql'),
            ],
            ['id' => $id],
            ['%d', '%s', '%s'],
            ['%d']
        );
    }
}
