<?php

declare(strict_types=1);

namespace WpOrchestrator\Modules\Targets;

use WpOrchestrator\Modules\Transport\TransportClient;

final class ConnectionTester
{
    public function __construct(
        private readonly TransportClient $transport = new TransportClient()
    ) {
    }

    /**
     * @param array<string, mixed> $target
     * @return array{ok: bool, status: string, message: string, http_code: int, payload: array<string, mixed>|null}
     */
    public function test(array $target): array
    {
        $baseUrl = untrailingslashit((string) ($target['base_url'] ?? ''));
        $authUser = (string) ($target['auth_user'] ?? '');
        $authSecret = (string) ($target['auth_secret'] ?? '');

        $response = $this->transport->getJson($baseUrl . '/wp-json/', $authUser, $authSecret, 12, 3);

        if (!$response['ok']) {
            return [
                'ok' => false,
                'status' => $this->mapStatus((string) $response['error_class']),
                'message' => (string) $response['error_message'],
                'http_code' => (int) $response['status_code'],
                'payload' => is_array($response['json']) ? $response['json'] : null,
            ];
        }

        return [
            'ok' => true,
            'status' => 'active',
            'message' => __('Connection successful.', 'wp-orchestrator'),
            'http_code' => (int) $response['status_code'],
            'payload' => is_array($response['json']) ? $response['json'] : null,
        ];
    }

    private function mapStatus(string $errorClass): string
    {
        $normalized = strtolower($errorClass);

        if ($normalized === 'tls_error') {
            return 'tls_error';
        }

        if ($normalized === 'dns_error') {
            return 'dns_error';
        }

        if ($normalized === 'timeout') {
            return 'timeout';
        }

        if ($normalized === 'auth_failed') {
            return 'auth_failed';
        }

        if ($normalized === 'rate_limited') {
            return 'rate_limited';
        }

        return 'unreachable';
    }
}
