<?php

declare(strict_types=1);

namespace WpOrchestrator\Modules\Actions;

final class ActionRepository
{
    /**
     * @param array<string, mixed>|null $payload
     */
    public function start(int $targetId, string $actionType, ?array $payload = null): string
    {
        global $wpdb;

        $table = $wpdb->prefix . 'wpo_actions';
        $now = current_time('mysql');
        $actionUuid = wp_generate_uuid4();

        $inserted = $wpdb->insert(
            $table,
            [
                'target_id' => $targetId,
                'action_uuid' => $actionUuid,
                'action_type' => $actionType,
                'payload_json' => wp_json_encode($payload ?? []),
                'status' => 'running',
                'started_at' => $now,
                'finished_at' => null,
                'created_at' => $now,
            ],
            ['%d', '%s', '%s', '%s', '%s', '%s', '%s', '%s']
        );

        if ($inserted !== 1) {
            throw new \RuntimeException('Unable to create action record.');
        }

        return $actionUuid;
    }

    /**
     * @param array<string, mixed>|null $payload
     */
    public function finish(string $actionUuid, string $status, ?array $payload = null): void
    {
        global $wpdb;

        $table = $wpdb->prefix . 'wpo_actions';

        $wpdb->update(
            $table,
            [
                'status' => $status,
                'payload_json' => wp_json_encode($payload ?? []),
                'finished_at' => current_time('mysql'),
            ],
            ['action_uuid' => $actionUuid],
            ['%s', '%s', '%s'],
            ['%s']
        );
    }

    /**
     * @return array<int, array<string, mixed>>
     */
    public function recent(int $limit = 20): array
    {
        global $wpdb;

        $table = $wpdb->prefix . 'wpo_actions';
        $limit = max(1, min(100, $limit));

        $sql = $wpdb->prepare(
            "SELECT id, target_id, action_uuid, action_type, payload_json, status, started_at, finished_at, created_at
             FROM {$table}
             ORDER BY id DESC
             LIMIT %d",
            $limit
        );

        if ($sql === null) {
            return [];
        }

        $rows = $wpdb->get_results($sql, ARRAY_A);
        return is_array($rows) ? $rows : [];
    }

    /**
     * @param array<int, int> $targetIds
     * @return array<int, array<string, mixed>>
     */
    public function latestFailuresByTargetIds(array $targetIds): array
    {
        if ($targetIds === []) {
            return [];
        }

        global $wpdb;

        $table = $wpdb->prefix . 'wpo_actions';
        $ids = array_values(array_unique(array_map('intval', $targetIds)));
        $placeholders = implode(',', array_fill(0, count($ids), '%d'));

        $sql = "
            SELECT a.target_id, a.action_type, a.status, a.finished_at, a.payload_json
            FROM {$table} a
            INNER JOIN (
                SELECT target_id, MAX(id) AS latest_id
                FROM {$table}
                WHERE target_id IN ({$placeholders}) AND status = %s
                GROUP BY target_id
            ) latest ON latest.latest_id = a.id
        ";

        $params = array_merge($ids, ['failed']);
        $prepared = $wpdb->prepare($sql, ...$params);

        if ($prepared === null) {
            return [];
        }

        $rows = $wpdb->get_results($prepared, ARRAY_A);
        $mapped = [];

        if (!is_array($rows)) {
            return $mapped;
        }

        foreach ($rows as $row) {
            $mapped[(int) $row['target_id']] = $row;
        }

        return $mapped;
    }

    /**
     * @return array<int, array<string, mixed>>
     */
    public function recentByTarget(int $targetId, int $limit = 20): array
    {
        global $wpdb;

        $table = $wpdb->prefix . 'wpo_actions';
        $limit = max(1, min(100, $limit));

        $sql = $wpdb->prepare(
            "SELECT id, target_id, action_uuid, action_type, payload_json, status, started_at, finished_at, created_at
             FROM {$table}
             WHERE target_id = %d
             ORDER BY id DESC
             LIMIT %d",
            $targetId,
            $limit
        );

        if ($sql === null) {
            return [];
        }

        $rows = $wpdb->get_results($sql, ARRAY_A);
        return is_array($rows) ? $rows : [];
    }
}
