<?php

declare(strict_types=1);

namespace WpOrchestrator\Admin;

use WpOrchestrator\Modules\Actions\ActionRepository;
use WpOrchestrator\Modules\Audit\AuditRepository;
use WpOrchestrator\Modules\Monitoring\HealthCheckRepository;
use WpOrchestrator\Modules\Targets\ConnectionTester;
use WpOrchestrator\Modules\Targets\TargetRepository;
use WpOrchestrator\Modules\Transport\TransportClient;
use WpOrchestrator\Modules\WpAi\WpAiChangesetService;
use WpOrchestrator\Modules\WpAi\WpAiJobService;
use WpOrchestrator\Modules\WpAi\WpAiUpdateService;
use WpOrchestrator\Security\AccessGuard;
use WpOrchestrator\Support\Logger;

final class AdminPages
{
    private const MENU_SLUG = 'wp-orchestrator';

    public function __construct(
        private readonly AccessGuard $guard,
        private readonly TargetRepository $targets,
        private readonly Logger $logger,
        private readonly ConnectionTester $connectionTester = new ConnectionTester(),
        private readonly HealthCheckRepository $healthChecks = new HealthCheckRepository()
    ) {
    }

    public function register(): void
    {
        add_action('admin_menu', [$this, 'registerMenu']);
        add_action('admin_post_wpo_save_target', [$this, 'handleSaveTarget']);
        add_action('admin_post_wpo_test_target', [$this, 'handleTestTarget']);
        add_action('admin_post_wpo_save_alert_settings', [$this, 'handleSaveAlertSettings']);
        add_action('admin_post_wpo_run_checks_now', [$this, 'handleRunChecksNow']);
        add_action('admin_post_wpo_wpai_preflight', [$this, 'handleWpAiPreflight']);
        add_action('admin_post_wpo_wpai_apply', [$this, 'handleWpAiApply']);
        add_action('admin_post_wpo_wpai_enqueue_job', [$this, 'handleWpAiEnqueueJob']);
        add_action('admin_post_wpo_wpai_poll_job', [$this, 'handleWpAiPollJob']);
        add_action('admin_post_wpo_wpai_changeset_apply', [$this, 'handleWpAiChangesetApply']);
        add_action('admin_post_wpo_run_target_debug_probe', [$this, 'handleRunTargetDebugProbe']);
    }

    public function registerMenu(): void
    {
        add_management_page(
            __('wpOrchestrator', 'wp-orchestrator'),
            __('wpOrchestrator', 'wp-orchestrator'),
            'manage_options',
            self::MENU_SLUG,
            [$this, 'renderPage']
        );
    }

    public function renderPage(): void
    {
        $this->guard->assertAccess();

        $targetId = isset($_GET['target_id']) ? (int) $_GET['target_id'] : 0;

        if ($targetId > 0) {
            $target = $this->targets->findById($targetId);
            $this->renderTargetDetailPage($targetId, $target);
            return;
        }

        $this->renderOverviewPage();
    }

    private function renderOverviewPage(): void
    {
        $rows = $this->targets->all();
        $targetIds = array_map(static fn(array $row): int => (int) $row['id'], $rows);
        $scores = $this->healthChecks->latestScoresByTargetIds($targetIds);
        $wpaiScores = $this->healthChecks->latestScoresByTargetIdsByType($targetIds, 'wpai_system');
        $coreByTarget = [];
        foreach ($targetIds as $id) {
            $core = $this->healthChecks->latestResultByTargetAndType($id, 'core');
            $coreByTarget[$id] = is_array($core['result_json'] ?? null) ? $core['result_json'] : [];
        }
        $actionRepository = new ActionRepository();
        $lastFailuresByTarget = $actionRepository->latestFailuresByTargetIds($targetIds);

        ?>
        <div class="wrap">
            <h1><?php echo esc_html__('wpOrchestrator', 'wp-orchestrator'); ?></h1>
            <p><?php echo esc_html__('Verwaltet externe WordPress-Zielinstanzen.', 'wp-orchestrator'); ?></p>

            <?php $this->renderNotices(); ?>

            <h2><?php echo esc_html__('Neue Zielinstanz', 'wp-orchestrator'); ?></h2>
            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                <input type="hidden" name="action" value="wpo_save_target">
                <?php wp_nonce_field('wpo_save_target'); ?>

                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><label for="wpo_name"><?php echo esc_html__('Name', 'wp-orchestrator'); ?></label></th>
                        <td><input type="text" id="wpo_name" name="name" class="regular-text" required></td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="wpo_base_url"><?php echo esc_html__('Base URL', 'wp-orchestrator'); ?></label></th>
                        <td><input type="url" id="wpo_base_url" name="base_url" class="regular-text" required placeholder="https://example.com"></td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="wpo_auth_user"><?php echo esc_html__('API User', 'wp-orchestrator'); ?></label></th>
                        <td><input type="text" id="wpo_auth_user" name="auth_user" class="regular-text" required></td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="wpo_auth_secret"><?php echo esc_html__('App Password', 'wp-orchestrator'); ?></label></th>
                        <td><input type="password" id="wpo_auth_secret" name="auth_secret" class="regular-text" required autocomplete="new-password"></td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="wpo_tags"><?php echo esc_html__('Tags', 'wp-orchestrator'); ?></label></th>
                        <td><input type="text" id="wpo_tags" name="tags" class="regular-text" placeholder="production, customer-a"></td>
                    </tr>
                </table>

                <?php submit_button(__('Zielinstanz speichern', 'wp-orchestrator')); ?>
            </form>

            <h2><?php echo esc_html__('Alerting Einstellungen', 'wp-orchestrator'); ?></h2>
            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                <input type="hidden" name="action" value="wpo_save_alert_settings">
                <?php wp_nonce_field('wpo_save_alert_settings'); ?>

                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><label for="wpo_alert_email_recipients"><?php echo esc_html__('E-Mail Empfaenger', 'wp-orchestrator'); ?></label></th>
                        <td>
                            <input type="text" id="wpo_alert_email_recipients" name="wpo_alert_email_recipients" class="regular-text" value="<?php echo esc_attr((string) get_option('wpo_alert_email_recipients', '')); ?>" placeholder="ops@example.com, admin@example.com">
                            <p class="description"><?php echo esc_html__('Leer = WordPress Admin-E-Mail.', 'wp-orchestrator'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="wpo_alert_score_threshold"><?php echo esc_html__('Score Schwelle (0-100)', 'wp-orchestrator'); ?></label></th>
                        <td><input type="number" id="wpo_alert_score_threshold" name="wpo_alert_score_threshold" min="0" max="100" value="<?php echo esc_attr((string) get_option('wpo_alert_score_threshold', 60)); ?>"></td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="wpo_telegram_enabled"><?php echo esc_html__('Telegram aktiv', 'wp-orchestrator'); ?></label></th>
                        <td><input type="checkbox" id="wpo_telegram_enabled" name="wpo_telegram_enabled" value="1" <?php checked((string) get_option('wpo_telegram_enabled', '0'), '1'); ?>></td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="wpo_telegram_bot_token"><?php echo esc_html__('Telegram Bot Token', 'wp-orchestrator'); ?></label></th>
                        <td><input type="password" id="wpo_telegram_bot_token" name="wpo_telegram_bot_token" class="regular-text" value="<?php echo esc_attr((string) get_option('wpo_telegram_bot_token', '')); ?>" autocomplete="new-password"></td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="wpo_telegram_chat_id"><?php echo esc_html__('Telegram Chat ID', 'wp-orchestrator'); ?></label></th>
                        <td><input type="text" id="wpo_telegram_chat_id" name="wpo_telegram_chat_id" class="regular-text" value="<?php echo esc_attr((string) get_option('wpo_telegram_chat_id', '')); ?>"></td>
                    </tr>
                </table>

                <?php submit_button(__('Alerting speichern', 'wp-orchestrator')); ?>
            </form>

            <h2><?php echo esc_html__('Zielinstanzen', 'wp-orchestrator'); ?></h2>
            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="margin: 0 0 12px;">
                <input type="hidden" name="action" value="wpo_run_checks_now">
                <?php wp_nonce_field('wpo_run_checks_now'); ?>
                <?php submit_button(__('Run Health Checks now', 'wp-orchestrator'), 'secondary', 'submit', false); ?>
            </form>
            <style>
                .wpo-col-id { width: 56px; }
                .wpo-col-icon { width: 54px; text-align: center; }
                .wpo-col-score { width: 82px; text-align: right; }
                .wpo-col-updates { width: 86px; text-align: right; }
                .wpo-col-seen { width: 140px; }
                .wpo-col-details { width: 90px; }
            </style>
            <table class="widefat fixed striped">
                <thead>
                    <tr>
                        <th class="wpo-col-id"><?php echo esc_html__('ID', 'wp-orchestrator'); ?></th>
                        <th><?php echo esc_html__('Name', 'wp-orchestrator'); ?></th>
                        <th class="wpo-col-icon"><?php echo esc_html__('Status', 'wp-orchestrator'); ?></th>
                        <th class="wpo-col-icon"><?php echo esc_html__('wpAI', 'wp-orchestrator'); ?></th>
                        <th class="wpo-col-score"><?php echo esc_html__('Last Score', 'wp-orchestrator'); ?></th>
                        <th class="wpo-col-score"><?php echo esc_html__('wpAI Score', 'wp-orchestrator'); ?></th>
                        <th class="wpo-col-updates"><?php echo esc_html__('Updates', 'wp-orchestrator'); ?></th>
                        <th><?php echo esc_html__('Last Error', 'wp-orchestrator'); ?></th>
                        <th class="wpo-col-seen"><?php echo esc_html__('Last Seen', 'wp-orchestrator'); ?></th>
                        <th class="wpo-col-details"><?php echo esc_html__('Details', 'wp-orchestrator'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php if ($rows === []) : ?>
                        <tr><td colspan="10"><?php echo esc_html__('Noch keine Zielinstanzen angelegt.', 'wp-orchestrator'); ?></td></tr>
                    <?php else : ?>
                        <?php foreach ($rows as $row) : ?>
                            <?php
                            $targetId = (int) $row['id'];
                            $corePayload = is_array($coreByTarget[$targetId] ?? null) ? $coreByTarget[$targetId] : [];
                            $updatesTotal = isset($corePayload['updates_available_total']) && is_int($corePayload['updates_available_total'])
                                ? (int) $corePayload['updates_available_total']
                                : null;
                            $updatesState = $updatesTotal === null ? 'unknown' : ($updatesTotal > 0 ? 'warn' : 'ok');
                            $statusRaw = (string) ($row['status'] ?? '');
                            $statusState = $statusRaw === 'active' ? 'ok' : ($statusRaw === 'degraded' ? 'warn' : 'fail');
                            $wpaiState = (int) ($row['wpai_available'] ?? 0) === 1 ? 'ok' : 'unknown';
                            ?>
                            <tr>
                                <td><?php echo esc_html((string) $targetId); ?></td>
                                <td><?php echo esc_html((string) $row['name']); ?></td>
                                <td><?php echo $this->renderDashicon($statusState); ?></td>
                                <td><?php echo $this->renderDashicon($wpaiState); ?></td>
                                <td><?php echo esc_html(isset($scores[$targetId]) ? (string) $scores[$targetId] : '-'); ?></td>
                                <td><?php echo esc_html(isset($wpaiScores[$targetId]) ? (string) $wpaiScores[$targetId] : '-'); ?></td>
                                <td><?php echo $this->renderDashicon($updatesState); ?> <?php echo esc_html($updatesTotal !== null ? (string) $updatesTotal : '-'); ?></td>
                                <td>
                                    <?php
                                    $failure = $lastFailuresByTarget[$targetId] ?? null;
                                    echo esc_html($failure !== null ? (string) ($failure['action_type'] . ' @ ' . ($failure['finished_at'] ?? '-')) : '-');
                                    ?>
                                </td>
                                <td><?php echo esc_html($this->formatRelativeTime(isset($row['last_seen_at']) ? (string) $row['last_seen_at'] : null)); ?></td>
                                <td><a class="button button-small" href="<?php echo esc_url($this->detailUrl($targetId)); ?>"><?php echo esc_html__('Open', 'wp-orchestrator'); ?></a></td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
        <?php
    }

    /**
     * @param array<string, mixed>|null $target
     */
    private function renderTargetDetailPage(int $targetId, ?array $target): void
    {
        ?>
        <div class="wrap">
            <h1><?php echo esc_html__('wpOrchestrator Target Details', 'wp-orchestrator'); ?></h1>
            <p><a href="<?php echo esc_url($this->baseUrl()); ?>">&larr; <?php echo esc_html__('Back to Overview', 'wp-orchestrator'); ?></a></p>
            <?php $this->renderNotices(); ?>
            <?php
            if ($target === null) {
                echo '<div class="notice notice-error"><p>' . esc_html__('Zielinstanz nicht gefunden.', 'wp-orchestrator') . '</p></div>';
                echo '</div>';
                return;
            }

            $targetRows = [$target];
            $scores = $this->healthChecks->latestScoresByTargetIds([$targetId]);
            $wpaiScores = $this->healthChecks->latestScoresByTargetIdsByType([$targetId], 'wpai_system');
            $latestCore = $this->healthChecks->latestResultByTargetAndType($targetId, 'core');
            $corePayload = is_array($latestCore['result_json'] ?? null) ? $latestCore['result_json'] : [];
            $wpAiJobService = $this->buildWpAiJobService();
            $lastJobId = $wpAiJobService->getLastJobId($targetId);
            $actionRepository = new ActionRepository();
            $recentActions = $actionRepository->recentByTarget($targetId, 20);
            $lastFailure = $actionRepository->latestFailuresByTargetIds([$targetId])[$targetId] ?? null;
            $debugProbe = get_transient($this->debugProbeKey($targetId));
            $wpAiAvailable = (int) ($target['wpai_available'] ?? 0) === 1;
            $updateSource = (string) ($corePayload['updates_source'] ?? 'unknown');
            $pluginStatus = (string) ($corePayload['plugin_status'] ?? '');
            $themeStatus = (string) ($corePayload['theme_status'] ?? '');
            $needsWpAiHint = $updateSource === 'unknown'
                || $pluginStatus === 'ok_no_update_signal'
                || $themeStatus === 'ok_no_update_signal';
            ?>

            <h2><?php echo esc_html((string) ($target['name'] ?? 'Target')); ?></h2>
            <table class="widefat fixed striped" style="max-width:900px;">
                <tbody>
                    <tr><th><?php echo esc_html__('ID', 'wp-orchestrator'); ?></th><td><?php echo esc_html((string) $targetId); ?></td></tr>
                    <tr><th><?php echo esc_html__('URL', 'wp-orchestrator'); ?></th><td><?php echo esc_html((string) ($target['base_url'] ?? '')); ?></td></tr>
                    <tr><th><?php echo esc_html__('API User', 'wp-orchestrator'); ?></th><td><?php echo esc_html((string) ($target['auth_user'] ?? '')); ?></td></tr>
                    <tr><th><?php echo esc_html__('Status', 'wp-orchestrator'); ?></th><td><?php echo $this->renderDashicon(((string) ($target['status'] ?? '') === 'active') ? 'ok' : (((string) ($target['status'] ?? '') === 'degraded') ? 'warn' : 'fail')); ?></td></tr>
                    <tr><th><?php echo esc_html__('wpAI', 'wp-orchestrator'); ?></th><td><?php echo $this->renderDashicon((int) ($target['wpai_available'] ?? 0) === 1 ? 'ok' : 'unknown'); ?></td></tr>
                    <tr><th><?php echo esc_html__('Last Score', 'wp-orchestrator'); ?></th><td><?php echo esc_html(isset($scores[$targetId]) ? (string) $scores[$targetId] : '-'); ?></td></tr>
                    <tr><th><?php echo esc_html__('wpAI Score', 'wp-orchestrator'); ?></th><td><?php echo esc_html(isset($wpaiScores[$targetId]) ? (string) $wpaiScores[$targetId] : '-'); ?></td></tr>
                    <tr><th><?php echo esc_html__('Last Job ID', 'wp-orchestrator'); ?></th><td><code><?php echo esc_html($lastJobId !== '' ? $lastJobId : '-'); ?></code></td></tr>
                    <tr><th><?php echo esc_html__('Last Error', 'wp-orchestrator'); ?></th><td><?php echo esc_html($lastFailure !== null ? (string) ($lastFailure['action_type'] . ' @ ' . ($lastFailure['finished_at'] ?? '-')) : '-'); ?></td></tr>
                    <tr><th><?php echo esc_html__('Last Seen', 'wp-orchestrator'); ?></th><td><?php echo esc_html($this->formatRelativeTime(isset($target['last_seen_at']) ? (string) $target['last_seen_at'] : null)); ?></td></tr>
                </tbody>
            </table>

            <h2 style="margin-top:24px;"><?php echo esc_html__('Readiness', 'wp-orchestrator'); ?></h2>
            <?php
            $pluginProbeCode = $this->probeStatusCode($debugProbe, '/wp-json/wp/v2/plugins?context=edit&per_page=100', 'GET');
            $wpAiPreflightCode = $this->probeStatusCode($debugProbe, '/wp-json/wpai/v1/updates/preflight', 'POST');
            $restUpdatesReady = $updateSource === 'wp_v2_plugins_themes';
            $credentialScopeReady = $pluginProbeCode === null ? null : ($pluginProbeCode >= 200 && $pluginProbeCode < 300);
            $wpAiRouteReady = $wpAiPreflightCode === null ? null : ($wpAiPreflightCode >= 200 && $wpAiPreflightCode < 300);
            ?>
            <table class="widefat fixed striped" style="max-width:900px;">
                <tbody>
                    <tr>
                        <th><?php echo esc_html__('REST Updates Metadata', 'wp-orchestrator'); ?></th>
                        <td><?php echo $this->renderDashicon($restUpdatesReady ? 'ok' : 'warn'); ?> <?php echo esc_html($restUpdatesReady ? 'available' : 'not available'); ?></td>
                    </tr>
                    <tr>
                        <th><?php echo esc_html__('Credential Scope (Plugins Endpoint)', 'wp-orchestrator'); ?></th>
                        <td><?php echo $this->renderDashicon($credentialScopeReady === null ? 'unknown' : ($credentialScopeReady ? 'ok' : 'fail')); ?> <?php echo esc_html($credentialScopeReady === null ? 'not probed' : ($credentialScopeReady ? 'ok' : 'insufficient')); ?></td>
                    </tr>
                    <tr>
                        <th><?php echo esc_html__('wpAI Updates Route', 'wp-orchestrator'); ?></th>
                        <td><?php echo $this->renderDashicon($wpAiRouteReady === null ? 'unknown' : ($wpAiRouteReady ? 'ok' : 'fail')); ?> <?php echo esc_html($wpAiRouteReady === null ? 'not probed' : ($wpAiRouteReady ? 'available' : 'missing/unreachable')); ?></td>
                    </tr>
                </tbody>
            </table>

            <h2 style="margin-top:24px;"><?php echo esc_html__('System Status', 'wp-orchestrator'); ?></h2>
            <table class="widefat fixed striped" style="max-width:900px;">
                <tbody>
                    <tr><th><?php echo esc_html__('Server Status', 'wp-orchestrator'); ?></th><td><?php echo $this->renderDashicon(((string) ($corePayload['server_status'] ?? '') === 'online') ? 'ok' : 'warn'); ?> <?php echo esc_html((string) ($corePayload['server_status'] ?? '-')); ?></td></tr>
                    <tr><th><?php echo esc_html__('WordPress Status', 'wp-orchestrator'); ?></th><td><?php echo $this->renderDashicon(((string) ($corePayload['wordpress_status'] ?? '') === 'ok') ? 'ok' : 'warn'); ?> <?php echo esc_html((string) ($corePayload['wordpress_status'] ?? '-')); ?></td></tr>
                    <tr><th><?php echo esc_html__('WordPress Version', 'wp-orchestrator'); ?></th><td><?php echo esc_html((string) ($corePayload['wp_version'] ?? '-')); ?></td></tr>
                    <tr><th><?php echo esc_html__('Plugin Status', 'wp-orchestrator'); ?></th><td><?php echo $this->renderDashicon((string) ($corePayload['plugin_status'] ?? '') === 'ok' ? 'ok' : 'warn'); ?> <?php echo esc_html((string) ($corePayload['plugin_status'] ?? '-')); ?></td></tr>
                    <tr><th><?php echo esc_html__('Update Verfuegbar (Plugins)', 'wp-orchestrator'); ?></th><td><?php $u = isset($corePayload['updates_available_plugins']) && is_int($corePayload['updates_available_plugins']) ? (int) $corePayload['updates_available_plugins'] : null; echo $this->renderDashicon($u === null ? 'unknown' : ($u > 0 ? 'warn' : 'ok')); ?> <?php echo esc_html($u !== null ? (string) $u : '-'); ?></td></tr>
                    <tr><th><?php echo esc_html__('Plugins Total/Active/Inactive', 'wp-orchestrator'); ?></th><td><?php echo esc_html((string) ($corePayload['plugins_total'] ?? '-') . ' / ' . (string) ($corePayload['plugins_active'] ?? '-') . ' / ' . (string) ($corePayload['plugins_inactive'] ?? '-')); ?></td></tr>
                    <tr><th><?php echo esc_html__('Theme Status', 'wp-orchestrator'); ?></th><td><?php echo $this->renderDashicon((string) ($corePayload['theme_status'] ?? '') === 'ok' ? 'ok' : 'warn'); ?> <?php echo esc_html((string) ($corePayload['theme_status'] ?? '-')); ?></td></tr>
                    <tr><th><?php echo esc_html__('Aktives Theme', 'wp-orchestrator'); ?></th><td><?php echo esc_html((string) ($corePayload['theme_active'] ?? '-')); ?></td></tr>
                    <tr><th><?php echo esc_html__('Themes Total', 'wp-orchestrator'); ?></th><td><?php echo esc_html((string) ($corePayload['themes_total'] ?? '-')); ?></td></tr>
                    <tr><th><?php echo esc_html__('Update Verfuegbar (Themes)', 'wp-orchestrator'); ?></th><td><?php $ut = isset($corePayload['updates_available_themes']) && is_int($corePayload['updates_available_themes']) ? (int) $corePayload['updates_available_themes'] : null; echo $this->renderDashicon($ut === null ? 'unknown' : ($ut > 0 ? 'warn' : 'ok')); ?> <?php echo esc_html($ut !== null ? (string) $ut : '-'); ?></td></tr>
                    <tr><th><?php echo esc_html__('Update Verfuegbar (Gesamt)', 'wp-orchestrator'); ?></th><td><?php $ug = isset($corePayload['updates_available_total']) && is_int($corePayload['updates_available_total']) ? (int) $corePayload['updates_available_total'] : null; echo $this->renderDashicon($ug === null ? 'unknown' : ($ug > 0 ? 'warn' : 'ok')); ?> <?php echo esc_html($ug !== null ? (string) $ug : '-'); ?></td></tr>
                    <tr><th><?php echo esc_html__('Update Quelle', 'wp-orchestrator'); ?></th><td><?php echo esc_html((string) ($corePayload['updates_source'] ?? '-')); ?></td></tr>
                    <?php if ($needsWpAiHint && !$wpAiAvailable) : ?>
                        <tr>
                            <th><?php echo esc_html__('Hinweis', 'wp-orchestrator'); ?></th>
                            <td><?php echo esc_html__('Die native WP REST API dieser Instanz liefert keine Update-Metadaten. Installiere und aktiviere wpAI auf der Zielinstanz, um Update-Verfuegbarkeit zu erhalten.', 'wp-orchestrator'); ?></td>
                        </tr>
                    <?php elseif ($needsWpAiHint && $wpAiAvailable) : ?>
                        <tr>
                            <th><?php echo esc_html__('Hinweis', 'wp-orchestrator'); ?></th>
                            <td><?php echo esc_html__('wpAI ist erkannt, aber Update-Daten sind nicht abrufbar. Bitte pruefe, ob auf der Zielinstanz die wpAI-Route /wp-json/wpai/v1/updates/preflight verfuegbar ist.', 'wp-orchestrator'); ?></td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>

            <h2 style="margin-top:24px;"><?php echo esc_html__('Actions', 'wp-orchestrator'); ?></h2>
            <div style="display:grid;grid-template-columns:repeat(auto-fit,minmax(240px,1fr));gap:12px;max-width:980px;">
                <?php $this->renderTargetActionForms($target, $lastJobId); ?>
            </div>

            <h2 style="margin-top:24px;"><?php echo esc_html__('Debug Probe', 'wp-orchestrator'); ?></h2>
            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="margin:0 0 12px;">
                <input type="hidden" name="action" value="wpo_run_target_debug_probe">
                <input type="hidden" name="target_id" value="<?php echo esc_attr((string) $targetId); ?>">
                <?php wp_nonce_field('wpo_run_target_debug_probe_' . (string) $targetId); ?>
                <?php submit_button(__('Run Debug Probe now', 'wp-orchestrator'), 'secondary', 'submit', false); ?>
            </form>

            <?php if (is_array($debugProbe)) : ?>
                <table class="widefat fixed striped">
                    <thead>
                        <tr>
                            <th><?php echo esc_html__('Endpoint', 'wp-orchestrator'); ?></th>
                            <th><?php echo esc_html__('Method', 'wp-orchestrator'); ?></th>
                            <th><?php echo esc_html__('HTTP', 'wp-orchestrator'); ?></th>
                            <th><?php echo esc_html__('Error Class', 'wp-orchestrator'); ?></th>
                            <th><?php echo esc_html__('Response JSON', 'wp-orchestrator'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach (($debugProbe['probes'] ?? []) as $probe) : ?>
                            <tr>
                                <td><code><?php echo esc_html((string) ($probe['endpoint'] ?? '')); ?></code></td>
                                <td><?php echo esc_html((string) ($probe['method'] ?? 'GET')); ?></td>
                                <td><?php echo esc_html((string) ($probe['status_code'] ?? '0')); ?></td>
                                <td><?php echo esc_html((string) ($probe['error_class'] ?? '')); ?></td>
                                <td><pre style="white-space:pre-wrap;max-height:220px;overflow:auto;"><?php echo esc_html((string) ($probe['json_pretty'] ?? '')); ?></pre></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>

            <h2 style="margin-top:24px;"><?php echo esc_html__('Action History (last 20 for this target)', 'wp-orchestrator'); ?></h2>
            <table class="widefat fixed striped">
                <thead>
                    <tr>
                        <th><?php echo esc_html__('ID', 'wp-orchestrator'); ?></th>
                        <th><?php echo esc_html__('Type', 'wp-orchestrator'); ?></th>
                        <th><?php echo esc_html__('Status', 'wp-orchestrator'); ?></th>
                        <th><?php echo esc_html__('Started', 'wp-orchestrator'); ?></th>
                        <th><?php echo esc_html__('Finished', 'wp-orchestrator'); ?></th>
                        <th><?php echo esc_html__('Action UUID', 'wp-orchestrator'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php if ($recentActions === []) : ?>
                        <tr><td colspan="6"><?php echo esc_html__('Noch keine Aktionen vorhanden.', 'wp-orchestrator'); ?></td></tr>
                    <?php else : ?>
                        <?php foreach ($recentActions as $action) : ?>
                            <tr>
                                <td><?php echo esc_html((string) $action['id']); ?></td>
                                <td><?php echo esc_html((string) $action['action_type']); ?></td>
                                <td><?php echo esc_html((string) $action['status']); ?></td>
                                <td><?php echo esc_html((string) ($action['started_at'] ?? '-')); ?></td>
                                <td><?php echo esc_html((string) ($action['finished_at'] ?? '-')); ?></td>
                                <td><code><?php echo esc_html((string) $action['action_uuid']); ?></code></td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
        <?php
    }

    /**
     * @param array<string, mixed> $target
     */
    private function renderTargetActionForms(array $target, string $lastJobId): void
    {
        $targetId = (int) ($target['id'] ?? 0);
        $isWpAi = (int) ($target['wpai_available'] ?? 0) === 1;

        ?>
        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="padding:10px;border:1px solid #ddd;background:#fff;">
            <input type="hidden" name="action" value="wpo_test_target">
            <input type="hidden" name="target_id" value="<?php echo esc_attr((string) $targetId); ?>">
            <?php wp_nonce_field('wpo_test_target_' . (string) $targetId); ?>
            <?php submit_button(__('Test Connection', 'wp-orchestrator'), 'secondary', 'submit', false); ?>
        </form>

        <?php if (!$isWpAi) : ?>
            <div style="padding:10px;border:1px solid #ddd;background:#fff;">
                <?php echo esc_html__('wpAI ist fuer diese Instanz nicht erkannt. Fuehre zunaechst Health Checks aus.', 'wp-orchestrator'); ?>
            </div>
            <?php
            return;
        endif;
        ?>

        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="padding:10px;border:1px solid #ddd;background:#fff;">
            <input type="hidden" name="action" value="wpo_wpai_preflight">
            <input type="hidden" name="target_id" value="<?php echo esc_attr((string) $targetId); ?>">
            <?php wp_nonce_field('wpo_wpai_preflight_' . (string) $targetId); ?>
            <?php submit_button(__('wpAI Preflight', 'wp-orchestrator'), 'secondary', 'submit', false); ?>
        </form>

        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="padding:10px;border:1px solid #ddd;background:#fff;">
            <input type="hidden" name="action" value="wpo_wpai_apply">
            <input type="hidden" name="target_id" value="<?php echo esc_attr((string) $targetId); ?>">
            <?php wp_nonce_field('wpo_wpai_apply_' . (string) $targetId); ?>
            <?php submit_button(__('wpAI Apply', 'wp-orchestrator'), 'primary', 'submit', false); ?>
        </form>

        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="padding:10px;border:1px solid #ddd;background:#fff;">
            <input type="hidden" name="action" value="wpo_wpai_enqueue_job">
            <input type="hidden" name="target_id" value="<?php echo esc_attr((string) $targetId); ?>">
            <?php wp_nonce_field('wpo_wpai_enqueue_job_' . (string) $targetId); ?>
            <?php submit_button(__('wpAI Enqueue Job', 'wp-orchestrator'), 'secondary', 'submit', false); ?>
        </form>

        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="padding:10px;border:1px solid #ddd;background:#fff;">
            <input type="hidden" name="action" value="wpo_wpai_poll_job">
            <input type="hidden" name="target_id" value="<?php echo esc_attr((string) $targetId); ?>">
            <?php wp_nonce_field('wpo_wpai_poll_job_' . (string) $targetId); ?>
            <input type="text" name="job_id" value="<?php echo esc_attr($lastJobId); ?>" placeholder="job id" style="width:100%;margin-bottom:8px;">
            <?php submit_button(__('wpAI Poll Job', 'wp-orchestrator'), 'secondary', 'submit', false); ?>
        </form>

        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="padding:10px;border:1px solid #ddd;background:#fff;">
            <input type="hidden" name="action" value="wpo_wpai_changeset_apply">
            <input type="hidden" name="target_id" value="<?php echo esc_attr((string) $targetId); ?>">
            <?php wp_nonce_field('wpo_wpai_changeset_apply_' . (string) $targetId); ?>
            <textarea name="changeset_json" rows="5" style="width:100%;margin-bottom:8px;">{"changes":[]}</textarea>
            <?php submit_button(__('wpAI Changeset Apply', 'wp-orchestrator'), 'secondary', 'submit', false); ?>
        </form>
        <?php
    }

    private function renderNotices(): void
    {
        $message = isset($_GET['wpo_message']) ? sanitize_text_field((string) $_GET['wpo_message']) : '';
        $error = isset($_GET['wpo_error']) ? sanitize_text_field((string) $_GET['wpo_error']) : '';

        if ($message !== '') {
            echo '<div class="notice notice-success"><p>' . esc_html($message) . '</p></div>';
        }

        if ($error !== '') {
            echo '<div class="notice notice-error"><p>' . esc_html($error) . '</p></div>';
        }
    }

    public function handleSaveTarget(): void
    {
        $this->guard->assertAccess();
        $this->guard->assertNonce('wpo_save_target');

        $name = isset($_POST['name']) ? sanitize_text_field((string) $_POST['name']) : '';
        $baseUrl = isset($_POST['base_url']) ? esc_url_raw((string) $_POST['base_url']) : '';
        $authUser = isset($_POST['auth_user']) ? sanitize_text_field((string) $_POST['auth_user']) : '';
        $authSecret = isset($_POST['auth_secret']) ? (string) $_POST['auth_secret'] : '';
        $tags = isset($_POST['tags']) ? sanitize_text_field((string) $_POST['tags']) : '';

        if ($name === '' || $authUser === '' || $authSecret === '' || !wp_http_validate_url($baseUrl)) {
            $this->redirectWith('wpo_error', __('Bitte alle Pflichtfelder korrekt ausfuellen.', 'wp-orchestrator'));
        }

        try {
            $this->targets->create($name, $baseUrl, $authUser, $authSecret, $tags);
            $this->logger->info('Target created.', ['base_url' => $baseUrl, 'auth_secret' => $authSecret]);
            $this->redirectWith('wpo_message', __('Zielinstanz wurde gespeichert.', 'wp-orchestrator'));
        } catch (\RuntimeException $e) {
            $this->logger->error('Target create failed.', ['base_url' => $baseUrl, 'error' => $e->getMessage()]);
            $this->redirectWith('wpo_error', __('Zielinstanz konnte nicht gespeichert werden.', 'wp-orchestrator'));
        }
    }

    public function handleTestTarget(): void
    {
        $this->guard->assertAccess();

        $targetId = isset($_POST['target_id']) ? (int) $_POST['target_id'] : 0;
        $this->guard->assertNonce('wpo_test_target_' . $targetId);

        if ($targetId < 1) {
            $this->redirectWith('wpo_error', __('Ungueltige Zielinstanz.', 'wp-orchestrator'));
        }

        $target = $this->targets->findById($targetId);

        if ($target === null) {
            $this->redirectWith('wpo_error', __('Zielinstanz nicht gefunden.', 'wp-orchestrator'));
        }

        try {
            $result = $this->connectionTester->test($target);
            $status = $result['ok'] ? 'active' : (string) $result['status'];
            $lastSeen = $result['ok'] ? current_time('mysql') : null;

            $this->targets->updateStatus($targetId, $status, $lastSeen);

            $this->logger->info('Connection test executed.', [
                'target_id' => $targetId,
                'status' => $status,
                'http_code' => $result['http_code'],
            ]);

            if ($result['ok']) {
                $this->redirectWith('wpo_message', __('Verbindungstest erfolgreich.', 'wp-orchestrator'), $targetId);
            }

            $this->redirectWith('wpo_error', (string) $result['message'], $targetId);
        } catch (\RuntimeException $e) {
            $this->logger->error('Connection test failed.', [
                'target_id' => $targetId,
                'error' => $e->getMessage(),
            ]);
            $this->redirectWith('wpo_error', __('Verbindungstest fehlgeschlagen.', 'wp-orchestrator'), $targetId);
        }
    }

    public function handleSaveAlertSettings(): void
    {
        $this->guard->assertAccess();
        $this->guard->assertNonce('wpo_save_alert_settings');

        $emails = isset($_POST['wpo_alert_email_recipients']) ? sanitize_text_field((string) $_POST['wpo_alert_email_recipients']) : '';
        $threshold = isset($_POST['wpo_alert_score_threshold']) ? (int) $_POST['wpo_alert_score_threshold'] : 60;
        $telegramEnabled = isset($_POST['wpo_telegram_enabled']) ? '1' : '0';
        $telegramToken = isset($_POST['wpo_telegram_bot_token']) ? sanitize_text_field((string) $_POST['wpo_telegram_bot_token']) : '';
        $telegramChatId = isset($_POST['wpo_telegram_chat_id']) ? sanitize_text_field((string) $_POST['wpo_telegram_chat_id']) : '';

        if ($threshold < 0) {
            $threshold = 0;
        }
        if ($threshold > 100) {
            $threshold = 100;
        }

        update_option('wpo_alert_email_recipients', $emails, false);
        update_option('wpo_alert_score_threshold', $threshold, false);
        update_option('wpo_telegram_enabled', $telegramEnabled, false);
        update_option('wpo_telegram_bot_token', $telegramToken, false);
        update_option('wpo_telegram_chat_id', $telegramChatId, false);

        $this->redirectWith('wpo_message', __('Alerting-Einstellungen gespeichert.', 'wp-orchestrator'));
    }

    public function handleRunChecksNow(): void
    {
        $this->guard->assertAccess();
        $this->guard->assertNonce('wpo_run_checks_now');

        try {
            if (function_exists('wpo_plugin')) {
                wpo_plugin()->runScheduledChecks();
            } else {
                do_action('wpo_run_health_checks');
            }

            $this->redirectWith('wpo_message', __('Health Checks wurden gestartet.', 'wp-orchestrator'));
        } catch (\Throwable $e) {
            $this->logger->error('Manual health check run failed.', ['error' => $e->getMessage()]);
            $this->redirectWith('wpo_error', __('Health Checks konnten nicht gestartet werden.', 'wp-orchestrator'));
        }
    }

    public function handleWpAiPreflight(): void
    {
        $this->guard->assertAccess();

        $targetId = isset($_POST['target_id']) ? (int) $_POST['target_id'] : 0;
        $this->guard->assertNonce('wpo_wpai_preflight_' . $targetId);

        $target = $this->targets->findById($targetId);
        if ($target === null) {
            $this->redirectWith('wpo_error', __('Zielinstanz nicht gefunden.', 'wp-orchestrator'));
        }

        if ((int) ($target['wpai_available'] ?? 0) !== 1) {
            $this->redirectWith('wpo_error', __('wpAI ist fuer diese Instanz nicht erkannt.', 'wp-orchestrator'), $targetId);
        }

        $service = $this->buildWpAiUpdateService();
        $actorId = (int) get_current_user_id();
        $result = $service->runPreflight($targetId, $target, $actorId);

        if ($result['ok']) {
            $this->redirectWith('wpo_message', __('wpAI Preflight erfolgreich.', 'wp-orchestrator'), $targetId);
        }

        $this->redirectWith('wpo_error', __('wpAI Preflight fehlgeschlagen.', 'wp-orchestrator') . ' ' . (string) $result['message'], $targetId);
    }

    public function handleWpAiApply(): void
    {
        $this->guard->assertAccess();

        $targetId = isset($_POST['target_id']) ? (int) $_POST['target_id'] : 0;
        $this->guard->assertNonce('wpo_wpai_apply_' . $targetId);

        $target = $this->targets->findById($targetId);
        if ($target === null) {
            $this->redirectWith('wpo_error', __('Zielinstanz nicht gefunden.', 'wp-orchestrator'));
        }

        if ((int) ($target['wpai_available'] ?? 0) !== 1) {
            $this->redirectWith('wpo_error', __('wpAI ist fuer diese Instanz nicht erkannt.', 'wp-orchestrator'), $targetId);
        }

        $service = $this->buildWpAiUpdateService();
        $actorId = (int) get_current_user_id();
        $result = $service->runApply($targetId, $target, $actorId);

        if ($result['ok']) {
            $this->redirectWith('wpo_message', __('wpAI Apply erfolgreich.', 'wp-orchestrator'), $targetId);
        }

        $this->redirectWith('wpo_error', __('wpAI Apply fehlgeschlagen.', 'wp-orchestrator') . ' ' . (string) $result['message'], $targetId);
    }

    public function handleWpAiEnqueueJob(): void
    {
        $this->guard->assertAccess();

        $targetId = isset($_POST['target_id']) ? (int) $_POST['target_id'] : 0;
        $this->guard->assertNonce('wpo_wpai_enqueue_job_' . $targetId);

        $target = $this->targets->findById($targetId);
        if ($target === null) {
            $this->redirectWith('wpo_error', __('Zielinstanz nicht gefunden.', 'wp-orchestrator'));
        }

        if ((int) ($target['wpai_available'] ?? 0) !== 1) {
            $this->redirectWith('wpo_error', __('wpAI ist fuer diese Instanz nicht erkannt.', 'wp-orchestrator'), $targetId);
        }

        $service = $this->buildWpAiJobService();
        $actorId = (int) get_current_user_id();
        $result = $service->enqueue($targetId, $target, $actorId);

        if ($result['ok']) {
            $this->redirectWith('wpo_message', __('wpAI Job enqueued.', 'wp-orchestrator') . ' Job ID: ' . (string) $result['job_id'], $targetId);
        }

        $this->redirectWith('wpo_error', __('wpAI Job enqueue fehlgeschlagen.', 'wp-orchestrator') . ' ' . (string) $result['message'], $targetId);
    }

    public function handleWpAiPollJob(): void
    {
        $this->guard->assertAccess();

        $targetId = isset($_POST['target_id']) ? (int) $_POST['target_id'] : 0;
        $this->guard->assertNonce('wpo_wpai_poll_job_' . $targetId);

        $target = $this->targets->findById($targetId);
        if ($target === null) {
            $this->redirectWith('wpo_error', __('Zielinstanz nicht gefunden.', 'wp-orchestrator'));
        }

        if ((int) ($target['wpai_available'] ?? 0) !== 1) {
            $this->redirectWith('wpo_error', __('wpAI ist fuer diese Instanz nicht erkannt.', 'wp-orchestrator'), $targetId);
        }

        $jobId = isset($_POST['job_id']) ? sanitize_text_field((string) $_POST['job_id']) : '';
        $service = $this->buildWpAiJobService();
        $actorId = (int) get_current_user_id();
        $result = $service->pollStatus($targetId, $target, $actorId, $jobId);

        if ($result['ok']) {
            $this->redirectWith('wpo_message', __('wpAI Job Status:', 'wp-orchestrator') . ' ' . (string) $result['status'] . ' (Job ID: ' . (string) $result['job_id'] . ')', $targetId);
        }

        $this->redirectWith('wpo_error', __('wpAI Job status fehlgeschlagen.', 'wp-orchestrator') . ' ' . (string) $result['message'], $targetId);
    }

    public function handleWpAiChangesetApply(): void
    {
        $this->guard->assertAccess();

        $targetId = isset($_POST['target_id']) ? (int) $_POST['target_id'] : 0;
        $this->guard->assertNonce('wpo_wpai_changeset_apply_' . $targetId);

        $target = $this->targets->findById($targetId);
        if ($target === null) {
            $this->redirectWith('wpo_error', __('Zielinstanz nicht gefunden.', 'wp-orchestrator'));
        }

        if ((int) ($target['wpai_available'] ?? 0) !== 1) {
            $this->redirectWith('wpo_error', __('wpAI ist fuer diese Instanz nicht erkannt.', 'wp-orchestrator'), $targetId);
        }

        $rawChangeset = isset($_POST['changeset_json']) ? wp_unslash((string) $_POST['changeset_json']) : '';
        $decoded = json_decode($rawChangeset, true);

        if (!is_array($decoded)) {
            $this->redirectWith('wpo_error', __('Ungueltiges Changeset JSON.', 'wp-orchestrator'), $targetId);
        }

        $service = $this->buildWpAiChangesetService();
        $actorId = (int) get_current_user_id();
        $result = $service->apply($targetId, $target, $actorId, $decoded);

        if ($result['ok']) {
            $this->redirectWith('wpo_message', __('wpAI Changeset Apply erfolgreich.', 'wp-orchestrator'), $targetId);
        }

        $this->redirectWith('wpo_error', __('wpAI Changeset Apply fehlgeschlagen.', 'wp-orchestrator') . ' ' . (string) $result['message'], $targetId);
    }

    public function handleRunTargetDebugProbe(): void
    {
        $this->guard->assertAccess();

        $targetId = isset($_POST['target_id']) ? (int) $_POST['target_id'] : 0;
        $this->guard->assertNonce('wpo_run_target_debug_probe_' . $targetId);

        $target = $this->targets->findById($targetId);
        if ($target === null) {
            $this->redirectWith('wpo_error', __('Zielinstanz nicht gefunden.', 'wp-orchestrator'));
        }

        try {
            $probe = $this->runTargetDebugProbe($target);
            set_transient($this->debugProbeKey($targetId), $probe, 30 * MINUTE_IN_SECONDS);
            $this->redirectWith('wpo_message', __('Debug Probe abgeschlossen.', 'wp-orchestrator'), $targetId);
        } catch (\Throwable $e) {
            $this->logger->error('Debug probe failed.', ['target_id' => $targetId, 'error' => $e->getMessage()]);
            $this->redirectWith('wpo_error', __('Debug Probe fehlgeschlagen.', 'wp-orchestrator') . ' ' . $e->getMessage(), $targetId);
        }
    }

    private function buildWpAiUpdateService(): WpAiUpdateService
    {
        return new WpAiUpdateService(
            new TransportClient(),
            $this->healthChecks,
            new ActionRepository(),
            new AuditRepository(),
            $this->logger
        );
    }

    private function buildWpAiJobService(): WpAiJobService
    {
        return new WpAiJobService(
            new TransportClient(),
            $this->healthChecks,
            new ActionRepository(),
            new AuditRepository(),
            $this->logger
        );
    }

    private function buildWpAiChangesetService(): WpAiChangesetService
    {
        return new WpAiChangesetService(
            new TransportClient(),
            $this->healthChecks,
            new ActionRepository(),
            new AuditRepository(),
            $this->logger
        );
    }

    private function baseUrl(): string
    {
        return admin_url('tools.php?page=' . self::MENU_SLUG);
    }

    private function detailUrl(int $targetId): string
    {
        return add_query_arg('target_id', $targetId, $this->baseUrl());
    }

    /**
     * @param array<string, mixed> $target
     * @return array<string, mixed>
     */
    private function runTargetDebugProbe(array $target): array
    {
        $transport = new TransportClient();
        $baseUrl = untrailingslashit((string) ($target['base_url'] ?? ''));
        $authUser = (string) ($target['auth_user'] ?? '');
        $authSecret = (string) ($target['auth_secret'] ?? '');

        $definitions = [
            ['method' => 'GET', 'endpoint' => '/wp-json/wp/v2/plugins?context=edit&per_page=100', 'body' => null],
            ['method' => 'GET', 'endpoint' => '/wp-json/wp/v2/plugins?status=upgrade&per_page=100', 'body' => null],
            ['method' => 'GET', 'endpoint' => '/wp-json/wp/v2/themes?context=edit&per_page=100', 'body' => null],
            ['method' => 'GET', 'endpoint' => '/wp-json/wp/v2/themes?status=upgrade&per_page=100', 'body' => null],
            ['method' => 'POST', 'endpoint' => '/wp-json/wpai/v1/updates/preflight', 'body' => ['source' => 'wp-orchestrator-debug']],
        ];

        $probes = [];
        foreach ($definitions as $definition) {
            $response = $transport->requestJson(
                method: (string) $definition['method'],
                url: $baseUrl . (string) $definition['endpoint'],
                authUser: $authUser,
                authSecret: $authSecret,
                timeout: 20,
                maxAttempts: 1,
                body: is_array($definition['body']) ? $definition['body'] : null
            );

            $probes[] = [
                'method' => (string) $definition['method'],
                'endpoint' => (string) $definition['endpoint'],
                'status_code' => (int) $response['status_code'],
                'error_class' => (string) $response['error_class'],
                'error_message' => (string) $response['error_message'],
                'json_pretty' => wp_json_encode($response['json'], JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES),
            ];
        }

        return [
            'target_id' => (int) ($target['id'] ?? 0),
            'ran_at' => current_time('mysql'),
            'probes' => $probes,
        ];
    }

    private function debugProbeKey(int $targetId): string
    {
        return 'wpo_debug_probe_' . $targetId;
    }

    /**
     * @param array<string, mixed>|false $debugProbe
     */
    private function probeStatusCode(array|false $debugProbe, string $endpoint, string $method): ?int
    {
        if (!is_array($debugProbe) || !isset($debugProbe['probes']) || !is_array($debugProbe['probes'])) {
            return null;
        }

        foreach ($debugProbe['probes'] as $probe) {
            if (!is_array($probe)) {
                continue;
            }
            if ((string) ($probe['endpoint'] ?? '') !== $endpoint) {
                continue;
            }
            if (strtoupper((string) ($probe['method'] ?? 'GET')) !== strtoupper($method)) {
                continue;
            }
            return isset($probe['status_code']) ? (int) $probe['status_code'] : null;
        }

        return null;
    }

    private function renderDashicon(string $state): string
    {
        $map = [
            'ok' => ['yes-alt', '#46b450'],
            'warn' => ['warning', '#dba617'],
            'fail' => ['dismiss', '#d63638'],
            'unknown' => ['minus', '#8c8f94'],
        ];

        [$icon, $color] = $map[$state] ?? $map['unknown'];

        return sprintf(
            '<span class="dashicons dashicons-%s" style="color:%s;vertical-align:text-bottom;"></span>',
            esc_attr($icon),
            esc_attr($color)
        );
    }

    private function formatRelativeTime(?string $mysqlDateTime): string
    {
        if ($mysqlDateTime === null || trim($mysqlDateTime) === '') {
            return '-';
        }

        $timestamp = strtotime($mysqlDateTime);
        if ($timestamp === false) {
            return '-';
        }

        $now = current_time('timestamp');
        $delta = max(0, $now - $timestamp);

        if ($delta < HOUR_IN_SECONDS) {
            $minutes = max(1, (int) floor($delta / MINUTE_IN_SECONDS));
            return sprintf('vor %d Minuten', $minutes);
        }

        if ($delta < DAY_IN_SECONDS) {
            $hours = max(1, (int) floor($delta / HOUR_IN_SECONDS));
            return sprintf('vor %d Stunden', $hours);
        }

        $days = max(1, (int) floor($delta / DAY_IN_SECONDS));
        return sprintf('vor %d Tagen', $days);
    }

    private function redirectWith(string $key, string $message, ?int $targetId = null): void
    {
        $url = add_query_arg($key, $message, $targetId !== null && $targetId > 0 ? $this->detailUrl($targetId) : $this->baseUrl());
        wp_safe_redirect($url);
        exit;
    }
}
