<?php

declare(strict_types=1);

namespace WpOrchestrator\Support;

final class SecretBox
{
    private const CIPHER = 'aes-256-gcm';

    public function encrypt(string $plainText): string
    {
        if (function_exists('sodium_crypto_secretbox')) {
            $key = sodium_crypto_generichash($this->keyMaterial(), '', SODIUM_CRYPTO_SECRETBOX_KEYBYTES);
            $nonce = random_bytes(SODIUM_CRYPTO_SECRETBOX_NONCEBYTES);
            $cipher = sodium_crypto_secretbox($plainText, $nonce, $key);

            return 'sodium:' . base64_encode($nonce . $cipher);
        }

        $key = hash('sha256', $this->keyMaterial(), true);
        $iv = random_bytes(12);
        $tag = '';

        $cipher = openssl_encrypt($plainText, self::CIPHER, $key, OPENSSL_RAW_DATA, $iv, $tag);

        if ($cipher === false) {
            throw new \RuntimeException('Unable to encrypt secret.');
        }

        return 'openssl:' . base64_encode($iv . $tag . $cipher);
    }

    public function decrypt(string $encoded): string
    {
        if (str_starts_with($encoded, 'sodium:')) {
            $payload = base64_decode(substr($encoded, 7), true);

            if ($payload === false) {
                throw new \RuntimeException('Invalid secret payload.');
            }

            $nonce = substr($payload, 0, SODIUM_CRYPTO_SECRETBOX_NONCEBYTES);
            $cipher = substr($payload, SODIUM_CRYPTO_SECRETBOX_NONCEBYTES);
            $key = sodium_crypto_generichash($this->keyMaterial(), '', SODIUM_CRYPTO_SECRETBOX_KEYBYTES);
            $plain = sodium_crypto_secretbox_open($cipher, $nonce, $key);

            if ($plain === false) {
                throw new \RuntimeException('Unable to decrypt secret.');
            }

            return $plain;
        }

        if (str_starts_with($encoded, 'openssl:')) {
            $payload = base64_decode(substr($encoded, 8), true);

            if ($payload === false || strlen($payload) < 29) {
                throw new \RuntimeException('Invalid secret payload.');
            }

            $iv = substr($payload, 0, 12);
            $tag = substr($payload, 12, 16);
            $cipher = substr($payload, 28);
            $key = hash('sha256', $this->keyMaterial(), true);

            $plain = openssl_decrypt($cipher, self::CIPHER, $key, OPENSSL_RAW_DATA, $iv, $tag);

            if ($plain === false) {
                throw new \RuntimeException('Unable to decrypt secret.');
            }

            return $plain;
        }

        throw new \RuntimeException('Unknown secret format.');
    }

    private function keyMaterial(): string
    {
        return wp_salt('auth') . '|' . ABSPATH;
    }
}
