<?php

declare(strict_types=1);

namespace WpOrchestrator;

use WpOrchestrator\Admin\AdminPages;
use WpOrchestrator\Infrastructure\Database\Migrator;
use WpOrchestrator\Modules\Alerting\AlertDispatcher;
use WpOrchestrator\Modules\Audit\AuditRepository;
use WpOrchestrator\Modules\Capabilities\CapabilityResolver;
use WpOrchestrator\Modules\Monitoring\HealthCheckRepository;
use WpOrchestrator\Modules\Monitoring\HealthCheckService;
use WpOrchestrator\Modules\Targets\ConnectionTester;
use WpOrchestrator\Modules\Targets\TargetRepository;
use WpOrchestrator\Modules\Transport\TransportClient;
use WpOrchestrator\Modules\WpAi\WpAiSyncService;
use WpOrchestrator\Security\AccessGuard;
use WpOrchestrator\Support\Logger;

final class Plugin
{
    private Migrator $migrator;

    public function __construct()
    {
        $this->migrator = new Migrator();
    }

    public function activate(): void
    {
        add_filter('cron_schedules', [$this->migrator, 'registerCronInterval']);
        $this->migrator->migrate();
        $this->ensureScheduler();
    }

    public function deactivate(): void
    {
        wp_clear_scheduled_hook(Migrator::CRON_HOOK);
    }

    public function boot(): void
    {
        add_filter('cron_schedules', [$this->migrator, 'registerCronInterval']);
        add_action('init', [$this, 'ensureScheduler']);
        add_action(Migrator::CRON_HOOK, [$this, 'runScheduledChecks']);
        add_action('admin_init', [$this, 'maybeMigrate']);

        if (!is_admin()) {
            return;
        }

        $logger = new Logger();
        $guard = new AccessGuard();
        $targets = new TargetRepository();

        (new AdminPages($guard, $targets, $logger))->register();
    }

    public function maybeMigrate(): void
    {
        $version = get_option(Migrator::SCHEMA_OPTION, '');

        if ($version === Migrator::SCHEMA_VERSION) {
            return;
        }

        $this->migrator->migrate();
    }

    public function ensureScheduler(): void
    {
        if (!wp_next_scheduled(Migrator::CRON_HOOK)) {
            wp_schedule_event(time() + 60, Migrator::CRON_INTERVAL, Migrator::CRON_HOOK);
        }
    }

    public function runScheduledChecks(): void
    {
        $lockKey = 'wpo_cron_health_lock';

        if (get_transient($lockKey)) {
            return;
        }

        set_transient($lockKey, 1, 4 * MINUTE_IN_SECONDS);

        try {
            $logger = new Logger();
            $transport = new TransportClient();
            $checks = new HealthCheckRepository();
            $service = new HealthCheckService(
                new TargetRepository(),
                new ConnectionTester($transport),
                $checks,
                $logger,
                new AlertDispatcher($logger),
                new CapabilityResolver(),
                new WpAiSyncService($transport, $checks, new AuditRepository(), $logger),
                $transport
            );
            $service->runAll();
        } finally {
            delete_transient($lockKey);
        }
    }
}
