<?php

declare(strict_types=1);

namespace WpOrchestrator\Modules\WpAi;

use WpOrchestrator\Modules\Actions\ActionRepository;
use WpOrchestrator\Modules\Audit\AuditRepository;
use WpOrchestrator\Modules\Monitoring\HealthCheckRepository;
use WpOrchestrator\Modules\Transport\TransportClient;
use WpOrchestrator\Support\Logger;

final class WpAiUpdateService
{
    public function __construct(
        private readonly TransportClient $transport,
        private readonly HealthCheckRepository $checks,
        private readonly ActionRepository $actions,
        private readonly AuditRepository $audit,
        private readonly Logger $logger
    ) {
    }

    /**
     * @param array<string, mixed> $target
     * @return array{ok: bool, status: string, message: string, action_uuid: string}
     */
    public function runPreflight(int $targetId, array $target, int $actorId): array
    {
        return $this->executeUpdateEndpoint(
            targetId: $targetId,
            target: $target,
            actorId: $actorId,
            actionType: 'wpai_updates_preflight',
            endpoint: '/wp-json/wpai/v1/updates/preflight'
        );
    }

    /**
     * @param array<string, mixed> $target
     * @return array{ok: bool, status: string, message: string, action_uuid: string}
     */
    public function runApply(int $targetId, array $target, int $actorId): array
    {
        $preflight = $this->runPreflight($targetId, $target, $actorId);

        if (!$preflight['ok']) {
            return [
                'ok' => false,
                'status' => 'preflight_failed',
                'message' => __('Apply aborted: preflight failed.', 'wp-orchestrator'),
                'action_uuid' => $preflight['action_uuid'],
            ];
        }

        return $this->executeUpdateEndpoint(
            targetId: $targetId,
            target: $target,
            actorId: $actorId,
            actionType: 'wpai_updates_apply',
            endpoint: '/wp-json/wpai/v1/updates/apply'
        );
    }

    /**
     * @param array<string, mixed> $target
     * @return array{ok: bool, status: string, message: string, action_uuid: string}
     */
    private function executeUpdateEndpoint(
        int $targetId,
        array $target,
        int $actorId,
        string $actionType,
        string $endpoint
    ): array {
        $baseUrl = untrailingslashit((string) ($target['base_url'] ?? ''));
        $authUser = (string) ($target['auth_user'] ?? '');
        $authSecret = (string) ($target['auth_secret'] ?? '');

        $actionUuid = $this->actions->start($targetId, $actionType, ['endpoint' => $endpoint]);

        $response = $this->transport->requestJson(
            method: 'POST',
            url: $baseUrl . $endpoint,
            authUser: $authUser,
            authSecret: $authSecret,
            timeout: 20,
            maxAttempts: 2,
            body: [
                'source' => 'wp-orchestrator',
                'action_uuid' => $actionUuid,
            ]
        );

        $ok = (bool) $response['ok'];
        $status = $ok ? 'success' : (string) $response['error_class'];
        $message = $ok
            ? __('wpAI update action executed.', 'wp-orchestrator')
            : (string) $response['error_message'];

        $payload = [
            'status' => $status,
            'message' => $message,
            'http_code' => (int) $response['status_code'],
            'endpoint' => $endpoint,
            'response' => $response['json'],
            'action_uuid' => $actionUuid,
        ];

        $this->actions->finish($actionUuid, $ok ? 'success' : 'failed', $payload);

        $this->checks->add(
            $targetId,
            $actionType,
            [
                'status' => $status,
                'message' => $message,
                'http_code' => (int) $response['status_code'],
                'checked_at' => current_time('mysql'),
                'response' => $response['json'],
                'action_uuid' => $actionUuid,
            ],
            $ok ? 100 : 0
        );

        $this->audit->addLocalAudit(
            $targetId,
            $actorId,
            'local:' . $actionUuid,
            $ok ? 'success' : 'failed',
            [
                'action_type' => $actionType,
                'status' => $status,
                'http_code' => (int) $response['status_code'],
                'endpoint' => $endpoint,
            ]
        );

        $this->logger->info('wpAI update action executed.', [
            'target_id' => $targetId,
            'action_type' => $actionType,
            'action_uuid' => $actionUuid,
            'status' => $status,
            'http_code' => $response['status_code'],
        ]);

        return [
            'ok' => $ok,
            'status' => $status,
            'message' => $message,
            'action_uuid' => $actionUuid,
        ];
    }
}
