<?php

declare(strict_types=1);

namespace WpOrchestrator\Modules\WpAi;

use WpOrchestrator\Modules\Actions\ActionRepository;
use WpOrchestrator\Modules\Audit\AuditRepository;
use WpOrchestrator\Modules\Monitoring\HealthCheckRepository;
use WpOrchestrator\Modules\Transport\TransportClient;
use WpOrchestrator\Support\Logger;

final class WpAiJobService
{
    public function __construct(
        private readonly TransportClient $transport,
        private readonly HealthCheckRepository $checks,
        private readonly ActionRepository $actions,
        private readonly AuditRepository $audit,
        private readonly Logger $logger
    ) {
    }

    /**
     * @param array<string, mixed> $target
     * @return array{ok: bool, status: string, message: string, action_uuid: string, job_id: string}
     */
    public function enqueue(int $targetId, array $target, int $actorId): array
    {
        $baseUrl = untrailingslashit((string) ($target['base_url'] ?? ''));
        $authUser = (string) ($target['auth_user'] ?? '');
        $authSecret = (string) ($target['auth_secret'] ?? '');

        $actionUuid = $this->actions->start($targetId, 'wpai_jobs_enqueue', ['endpoint' => '/wp-json/wpai/v1/jobs/enqueue']);

        $response = $this->transport->requestJson(
            method: 'POST',
            url: $baseUrl . '/wp-json/wpai/v1/jobs/enqueue',
            authUser: $authUser,
            authSecret: $authSecret,
            timeout: 20,
            maxAttempts: 2,
            body: [
                'type' => 'orchestrator_health_probe',
                'payload' => [
                    'source' => 'wp-orchestrator',
                    'requested_at' => current_time('mysql'),
                ],
                'source' => 'wp-orchestrator',
                'action_uuid' => $actionUuid,
            ]
        );

        $jobId = $this->extractJobId($response['json']);
        $ok = (bool) $response['ok'] && $jobId !== '';
        $status = $ok ? 'queued' : ((bool) $response['ok'] ? 'job_id_missing' : (string) $response['error_class']);
        $message = $ok
            ? __('wpAI job queued.', 'wp-orchestrator')
            : ((string) $response['error_message'] !== '' ? (string) $response['error_message'] : __('Unable to enqueue wpAI job.', 'wp-orchestrator'));

        $payload = [
            'status' => $status,
            'message' => $message,
            'http_code' => (int) $response['status_code'],
            'response' => $response['json'],
            'action_uuid' => $actionUuid,
            'job_id' => $jobId,
        ];

        $this->actions->finish($actionUuid, $ok ? 'success' : 'failed', $payload);
        $this->checks->add($targetId, 'wpai_jobs_enqueue', $payload, $ok ? 100 : 0);

        $this->audit->addLocalAudit(
            $targetId,
            $actorId,
            'local:' . $actionUuid,
            $ok ? 'success' : 'failed',
            [
                'action_type' => 'wpai_jobs_enqueue',
                'status' => $status,
                'http_code' => (int) $response['status_code'],
                'job_id' => $jobId,
            ]
        );

        if ($jobId !== '') {
            $this->setLastJobId($targetId, $jobId);
        }

        $this->logger->info('wpAI job enqueue executed.', [
            'target_id' => $targetId,
            'action_uuid' => $actionUuid,
            'status' => $status,
            'job_id' => $jobId,
        ]);

        return [
            'ok' => $ok,
            'status' => $status,
            'message' => $message,
            'action_uuid' => $actionUuid,
            'job_id' => $jobId,
        ];
    }

    /**
     * @param array<string, mixed> $target
     * @return array{ok: bool, status: string, message: string, action_uuid: string, job_id: string}
     */
    public function pollStatus(int $targetId, array $target, int $actorId, string $jobId): array
    {
        $jobId = trim($jobId);
        if ($jobId === '') {
            return [
                'ok' => false,
                'status' => 'invalid_job_id',
                'message' => __('Missing job ID.', 'wp-orchestrator'),
                'action_uuid' => '',
                'job_id' => '',
            ];
        }

        $baseUrl = untrailingslashit((string) ($target['base_url'] ?? ''));
        $authUser = (string) ($target['auth_user'] ?? '');
        $authSecret = (string) ($target['auth_secret'] ?? '');

        $actionUuid = $this->actions->start($targetId, 'wpai_jobs_status', [
            'endpoint' => '/wp-json/wpai/v1/jobs/{id}',
            'job_id' => $jobId,
        ]);

        $response = $this->transport->requestJson(
            method: 'GET',
            url: $baseUrl . '/wp-json/wpai/v1/jobs/' . rawurlencode($jobId),
            authUser: $authUser,
            authSecret: $authSecret,
            timeout: 15,
            maxAttempts: 2
        );

        $remoteStatus = $this->extractJobStatus($response['json']);
        $ok = (bool) $response['ok'];
        $status = $ok ? ($remoteStatus !== '' ? $remoteStatus : 'ok') : (string) $response['error_class'];
        $message = $ok
            ? __('wpAI job status fetched.', 'wp-orchestrator')
            : ((string) $response['error_message'] !== '' ? (string) $response['error_message'] : __('Unable to fetch wpAI job status.', 'wp-orchestrator'));

        $score = $this->scoreFromJobStatus($ok, $remoteStatus);

        $payload = [
            'status' => $status,
            'message' => $message,
            'http_code' => (int) $response['status_code'],
            'response' => $response['json'],
            'action_uuid' => $actionUuid,
            'job_id' => $jobId,
        ];

        $this->actions->finish($actionUuid, $ok ? 'success' : 'failed', $payload);
        $this->checks->add($targetId, 'wpai_jobs_status', $payload, $score);

        $this->audit->addLocalAudit(
            $targetId,
            $actorId,
            'local:' . $actionUuid,
            $ok ? 'success' : 'failed',
            [
                'action_type' => 'wpai_jobs_status',
                'status' => $status,
                'http_code' => (int) $response['status_code'],
                'job_id' => $jobId,
            ]
        );

        $this->setLastJobId($targetId, $jobId);

        $this->logger->info('wpAI job status polled.', [
            'target_id' => $targetId,
            'action_uuid' => $actionUuid,
            'job_id' => $jobId,
            'status' => $status,
        ]);

        return [
            'ok' => $ok,
            'status' => $status,
            'message' => $message,
            'action_uuid' => $actionUuid,
            'job_id' => $jobId,
        ];
    }

    public function getLastJobId(int $targetId): string
    {
        return (string) get_option($this->lastJobOptionKey($targetId), '');
    }

    private function setLastJobId(int $targetId, string $jobId): void
    {
        update_option($this->lastJobOptionKey($targetId), $jobId, false);
    }

    private function lastJobOptionKey(int $targetId): string
    {
        return 'wpo_wpai_last_job_' . $targetId;
    }

    /**
     * @param array<string, mixed>|null $json
     */
    private function extractJobId(?array $json): string
    {
        if (!is_array($json)) {
            return '';
        }

        foreach (['id', 'job_id'] as $key) {
            if (isset($json[$key]) && (string) $json[$key] !== '') {
                return (string) $json[$key];
            }
        }

        if (isset($json['job']) && is_array($json['job'])) {
            foreach (['id', 'job_id'] as $key) {
                if (isset($json['job'][$key]) && (string) $json['job'][$key] !== '') {
                    return (string) $json['job'][$key];
                }
            }
        }

        return '';
    }

    /**
     * @param array<string, mixed>|null $json
     */
    private function extractJobStatus(?array $json): string
    {
        if (!is_array($json)) {
            return '';
        }

        foreach (['status', 'state'] as $key) {
            if (isset($json[$key]) && (string) $json[$key] !== '') {
                return sanitize_key((string) $json[$key]);
            }
        }

        if (isset($json['job']) && is_array($json['job'])) {
            foreach (['status', 'state'] as $key) {
                if (isset($json['job'][$key]) && (string) $json['job'][$key] !== '') {
                    return sanitize_key((string) $json['job'][$key]);
                }
            }
        }

        return '';
    }

    private function scoreFromJobStatus(bool $ok, string $jobStatus): int
    {
        if (!$ok) {
            return 0;
        }

        return match (sanitize_key($jobStatus)) {
            'completed', 'success', 'done' => 100,
            'running', 'queued', 'pending' => 70,
            'failed', 'error', 'cancelled' => 0,
            default => 60,
        };
    }
}
