<?php

declare(strict_types=1);

namespace WpOrchestrator\Modules\Monitoring;

use WpOrchestrator\Modules\Alerting\AlertDispatcher;
use WpOrchestrator\Modules\Capabilities\CapabilityResolver;
use WpOrchestrator\Modules\Targets\ConnectionTester;
use WpOrchestrator\Modules\Targets\TargetRepository;
use WpOrchestrator\Modules\Transport\TransportClient;
use WpOrchestrator\Modules\WpAi\WpAiSyncService;
use WpOrchestrator\Support\Logger;

final class HealthCheckService
{
    public function __construct(
        private readonly TargetRepository $targets,
        private readonly ConnectionTester $connectionTester,
        private readonly HealthCheckRepository $checks,
        private readonly Logger $logger,
        private readonly AlertDispatcher $alerts,
        private readonly CapabilityResolver $capabilityResolver,
        private readonly WpAiSyncService $wpAiSync,
        private readonly TransportClient $transport
    ) {
    }

    public function runAll(): void
    {
        $targets = $this->targets->allMonitorable();

        foreach ($targets as $targetRow) {
            $targetId = (int) ($targetRow['id'] ?? 0);

            if ($targetId < 1) {
                continue;
            }

            $this->runForTargetId($targetId);
        }
    }

    public function runForTargetId(int $targetId): void
    {
        $target = $this->targets->findById($targetId);

        if ($target === null) {
            return;
        }

        $connection = $this->connectionTester->test($target);

        if (!$connection['ok']) {
            $result = [
                'reachable' => false,
                'rest_ok' => false,
                'status' => (string) $connection['status'],
                'message' => (string) $connection['message'],
                'http_code' => (int) $connection['http_code'],
                'checked_at' => current_time('mysql'),
            ];

            $score = 0;
            $this->checks->add($targetId, 'core', $result, $score);
            $this->targets->updateStatus($targetId, (string) $connection['status'], null);
            $this->alerts->dispatchIfNeeded($target, $result, $score);

            $this->logger->error('Health check failed.', [
                'target_id' => $targetId,
                'status' => $connection['status'],
                'http_code' => $connection['http_code'],
            ]);

            return;
        }

        $meta = $this->extractMeta($target, $connection);
        $runtime = $this->fetchRuntimeStatus($target);
        $theme = $this->fetchThemeStatus($target);
        $updateSummary = $this->resolveUpdateSummary($target, $runtime, $theme);
        $pluginUpdates = $runtime['updates_available'];
        $themeUpdates = $theme['theme_updates_available'];

        if (is_int($updateSummary['total']) && ($pluginUpdates === null || $pluginUpdates === 0) && $updateSummary['source'] !== 'wp_v2_plugins_themes') {
            // If WP core REST does not expose update metadata reliably, use wpAI preflight as authoritative fallback.
            $pluginUpdates = (int) $updateSummary['total'];
            if ($themeUpdates === null) {
                $themeUpdates = 0;
            }
        }
        $score = $this->calculateScore($meta, $runtime, $theme);

        $result = [
            'reachable' => true,
            'rest_ok' => true,
            'status' => $score >= 60 ? 'active' : 'degraded',
            'message' => (string) $connection['message'],
            'http_code' => (int) $connection['http_code'],
            'checked_at' => current_time('mysql'),
            'server_status' => 'online',
            'wordpress_status' => $meta['wp_version'] !== null ? 'ok' : 'unknown',
            'plugin_status' => $runtime['plugin_status'],
            'updates_available_plugins' => $pluginUpdates,
            'plugins_total' => $runtime['plugins_total'],
            'plugins_active' => $runtime['plugins_active'],
            'plugins_inactive' => $runtime['plugins_inactive'],
            'theme_status' => $theme['theme_status'],
            'theme_active' => $theme['theme_active'],
            'themes_total' => $theme['themes_total'],
            'updates_available_themes' => $themeUpdates,
            'updates_available_total' => $updateSummary['total'],
            'updates_source' => $updateSummary['source'],
            'site_name' => $meta['site_name'],
            'site_url' => $meta['site_url'],
            'wp_version' => $meta['wp_version'],
            'php_version' => $meta['php_version'],
            'namespaces' => $meta['namespaces'],
        ];

        $status = (string) $result['status'];
        $lastSeenAt = current_time('mysql');
        $resolved = $this->capabilityResolver->resolveFromNamespaces($meta['namespaces']);

        $this->checks->add($targetId, 'core', $result, $score);
        $this->targets->updateStatus($targetId, $status, $lastSeenAt);
        $this->targets->updateCapabilities(
            $targetId,
            (bool) $resolved['wpai_available'],
            is_array($resolved['capabilities']) ? $resolved['capabilities'] : []
        );

        if ((bool) $resolved['wpai_available']) {
            $this->wpAiSync->sync($targetId, $target);
        }

        $this->alerts->dispatchIfNeeded($target, $result, $score);

        $this->logger->info('Health check completed.', [
            'target_id' => $targetId,
            'status' => $status,
            'score' => $score,
            'http_code' => $connection['http_code'],
            'wpai_available' => (bool) $resolved['wpai_available'],
        ]);
    }

    /**
     * @param array<string, mixed> $target
     * @return array{site_name: string, site_url: string, wp_version: string|null, php_version: string|null, namespaces: array<int, string>}
     */
    private function extractMeta(array $target, array $connection): array
    {
        $baseUrl = untrailingslashit((string) ($target['base_url'] ?? ''));
        $data = isset($connection['payload']) && is_array($connection['payload']) ? $connection['payload'] : null;

        if ($data === null) {
            return [
                'site_name' => '',
                'site_url' => $baseUrl,
                'wp_version' => null,
                'php_version' => null,
                'namespaces' => [],
            ];
        }

        $generator = isset($data['generator']) ? (string) $data['generator'] : '';
        $wpVersion = null;

        if ($generator !== '' && preg_match('/\?v=([0-9.]+)/', $generator, $matches) === 1) {
            $wpVersion = isset($matches[1]) ? (string) $matches[1] : null;
        }

        return [
            'site_name' => isset($data['name']) ? (string) $data['name'] : '',
            'site_url' => isset($data['url']) ? (string) $data['url'] : $baseUrl,
            'wp_version' => $wpVersion,
            'php_version' => null,
            'namespaces' => isset($data['namespaces']) && is_array($data['namespaces']) ? array_values(array_map('strval', $data['namespaces'])) : [],
        ];
    }

    /**
     * @param array<string, mixed> $target
     * @return array{plugin_status: string, updates_available: int|null, plugins_total: int, plugins_active: int, plugins_inactive: int}
     */
    private function fetchRuntimeStatus(array $target): array
    {
        $baseUrl = untrailingslashit((string) ($target['base_url'] ?? ''));
        $authUser = (string) ($target['auth_user'] ?? '');
        $authSecret = (string) ($target['auth_secret'] ?? '');

        $response = $this->transport->getJson($baseUrl . '/wp-json/wp/v2/plugins?context=edit&per_page=100', $authUser, $authSecret, 15, 2);

        if (!$response['ok'] || !is_array($response['json'])) {
            return [
                'plugin_status' => (string) $response['error_class'] ?: 'unavailable',
                'updates_available' => null,
                'plugins_total' => 0,
                'plugins_active' => 0,
                'plugins_inactive' => 0,
            ];
        }

        $plugins = $response['json'];
        if (!array_is_list($plugins)) {
            return [
                'plugin_status' => 'unexpected_payload',
                'updates_available' => null,
                'plugins_total' => 0,
                'plugins_active' => 0,
                'plugins_inactive' => 0,
            ];
        }
        $total = 0;
        $active = 0;
        $inactive = 0;
        $updates = 0;
        $sawUpdateSignal = false;

        foreach ($plugins as $plugin) {
            if (!is_array($plugin)) {
                continue;
            }

            $total++;
            $status = isset($plugin['status']) ? (string) $plugin['status'] : '';
            if ($status === 'active') {
                $active++;
            } else {
                $inactive++;
            }

            if ($this->pluginHasUpdateSignal($plugin)) {
                $sawUpdateSignal = true;
            }

            if ($this->hasAvailablePluginUpdate($plugin)) {
                $updates++;
            }
        }

        if (!$sawUpdateSignal) {
            $upgradeCount = $this->countUpgradableItems($baseUrl . '/wp-json/wp/v2/plugins?context=edit&status=upgrade&per_page=100', $authUser, $authSecret);
            if ($upgradeCount === null) {
                $upgradeCount = $this->countUpgradableItems($baseUrl . '/wp-json/wp/v2/plugins?status=upgrade&per_page=100', $authUser, $authSecret);
            }

            if ($upgradeCount !== null) {
                return [
                    'plugin_status' => 'ok_upgrade_filter',
                    'updates_available' => $upgradeCount,
                    'plugins_total' => $total,
                    'plugins_active' => $active,
                    'plugins_inactive' => $inactive,
                ];
            }
        }

        return [
            'plugin_status' => $sawUpdateSignal ? 'ok' : 'ok_no_update_signal',
            'updates_available' => $sawUpdateSignal ? $updates : null,
            'plugins_total' => $total,
            'plugins_active' => $active,
            'plugins_inactive' => $inactive,
        ];
    }

    /**
     * @param array<string, mixed> $target
     * @return array{theme_status: string, theme_active: string, themes_total: int, theme_updates_available: int|null}
     */
    private function fetchThemeStatus(array $target): array
    {
        $baseUrl = untrailingslashit((string) ($target['base_url'] ?? ''));
        $authUser = (string) ($target['auth_user'] ?? '');
        $authSecret = (string) ($target['auth_secret'] ?? '');

        $response = $this->transport->getJson($baseUrl . '/wp-json/wp/v2/themes?context=edit&per_page=100', $authUser, $authSecret, 15, 2);

        if (!$response['ok'] || !is_array($response['json'])) {
            return [
                'theme_status' => (string) $response['error_class'] ?: 'unavailable',
                'theme_active' => '',
                'themes_total' => 0,
                'theme_updates_available' => null,
            ];
        }

        $themes = $response['json'];
        if (!array_is_list($themes)) {
            return [
                'theme_status' => 'unexpected_payload',
                'theme_active' => '',
                'themes_total' => 0,
                'theme_updates_available' => null,
            ];
        }

        $total = 0;
        $activeTheme = '';
        $updates = 0;
        $sawUpdateSignal = false;

        foreach ($themes as $theme) {
            if (!is_array($theme)) {
                continue;
            }

            $total++;
            $status = isset($theme['status']) ? (string) $theme['status'] : '';
            if ($status === 'active') {
                if (isset($theme['stylesheet']) && (string) $theme['stylesheet'] !== '') {
                    $activeTheme = (string) $theme['stylesheet'];
                } elseif (isset($theme['name']) && is_array($theme['name']) && isset($theme['name']['raw'])) {
                    $activeTheme = (string) $theme['name']['raw'];
                } elseif (isset($theme['name']) && is_string($theme['name'])) {
                    $activeTheme = (string) $theme['name'];
                } else {
                    $activeTheme = 'active';
                }
            }

            if ($this->themeHasUpdateSignal($theme)) {
                $sawUpdateSignal = true;
            }

            if ($this->hasAvailableThemeUpdate($theme)) {
                $updates++;
            }
        }

        if (!$sawUpdateSignal) {
            $upgradeCount = $this->countUpgradableItems($baseUrl . '/wp-json/wp/v2/themes?context=edit&status=upgrade&per_page=100', $authUser, $authSecret);
            if ($upgradeCount === null) {
                $upgradeCount = $this->countUpgradableItems($baseUrl . '/wp-json/wp/v2/themes?status=upgrade&per_page=100', $authUser, $authSecret);
            }

            if ($upgradeCount !== null) {
                return [
                    'theme_status' => 'ok_upgrade_filter',
                    'theme_active' => $activeTheme,
                    'themes_total' => $total,
                    'theme_updates_available' => $upgradeCount,
                ];
            }
        }

        return [
            'theme_status' => $sawUpdateSignal ? 'ok' : 'ok_no_update_signal',
            'theme_active' => $activeTheme,
            'themes_total' => $total,
            'theme_updates_available' => $sawUpdateSignal ? $updates : null,
        ];
    }

    /**
     * @param array{site_name: string, site_url: string, wp_version: string|null, php_version: string|null, namespaces: array<int, string>} $meta
     * @param array{plugin_status: string, updates_available: int|null, plugins_total: int, plugins_active: int, plugins_inactive: int} $runtime
     * @param array{theme_status: string, theme_active: string, themes_total: int, theme_updates_available: int|null} $theme
     */
    private function calculateScore(array $meta, array $runtime, array $theme): int
    {
        $score = 50;

        if ($meta['wp_version'] !== null && $meta['wp_version'] !== '') {
            $score += 20;
        }

        if (in_array('wp/v2', $meta['namespaces'], true)) {
            $score += 20;
        }

        if ($meta['site_name'] !== '' && $meta['site_url'] !== '') {
            $score += 10;
        }

        if ($runtime['plugin_status'] === 'ok') {
            $score += 10;
        }

        if (is_int($runtime['updates_available']) && $runtime['updates_available'] > 0) {
            $score -= min(20, $runtime['updates_available'] * 2);
        }

        if ($theme['theme_status'] === 'ok') {
            $score += 5;
        }

        if (is_int($theme['theme_updates_available']) && $theme['theme_updates_available'] > 0) {
            $score -= min(10, $theme['theme_updates_available'] * 2);
        }

        return max(0, min(100, $score));
    }

    /**
     * @return array{total: int|null, source: string}
     */
    private function resolveUpdateSummary(array $target, array $runtime, array $theme): array
    {
        $pluginUpdates = $runtime['updates_available'];
        $themeUpdates = $theme['theme_updates_available'];

        if (is_int($pluginUpdates) || is_int($themeUpdates)) {
            return [
                'total' => (int) ($pluginUpdates ?? 0) + (int) ($themeUpdates ?? 0),
                'source' => 'wp_v2_plugins_themes',
            ];
        }

        if ((int) ($target['wpai_available'] ?? 0) === 1) {
            $fallback = $this->fetchUpdateSummaryFromWpAi($target);
            if (is_int($fallback)) {
                return [
                    'total' => $fallback,
                    'source' => 'wpai_updates_preflight',
                ];
            }
        }

        return [
            'total' => null,
            'source' => 'unknown',
        ];
    }

    /**
     * @return int|null
     */
    private function fetchUpdateSummaryFromWpAi(array $target): ?int
    {
        $baseUrl = untrailingslashit((string) ($target['base_url'] ?? ''));
        $authUser = (string) ($target['auth_user'] ?? '');
        $authSecret = (string) ($target['auth_secret'] ?? '');

        $response = $this->transport->requestJson(
            method: 'POST',
            url: $baseUrl . '/wp-json/wpai/v1/updates/preflight',
            authUser: $authUser,
            authSecret: $authSecret,
            timeout: 15,
            maxAttempts: 1,
            body: ['source' => 'wp-orchestrator-health']
        );

        if (!$response['ok'] || !is_array($response['json'])) {
            return null;
        }

        return $this->countAvailableUpdates($response['json']);
    }

    private function hasAvailableUpdate(mixed $updateField): bool
    {
        if (is_string($updateField)) {
            $value = strtolower(trim($updateField));
            return in_array($value, ['available', 'upgrade', 'update_available', 'yes', 'true'], true);
        }

        if (is_bool($updateField)) {
            return $updateField;
        }

        if (is_array($updateField)) {
            if ($updateField === []) {
                return false;
            }

            foreach (['available', 'update_available', 'new_version', 'version'] as $key) {
                if (!array_key_exists($key, $updateField)) {
                    continue;
                }

                $value = $updateField[$key];
                if (is_bool($value) && $value) {
                    return true;
                }
                if (is_string($value) && trim($value) !== '' && strtolower($value) !== 'false') {
                    return true;
                }
            }

            return !empty($updateField);
        }

        return false;
    }

    /**
     * @param array<string, mixed> $plugin
     */
    private function hasAvailablePluginUpdate(array $plugin): bool
    {
        // Common WP REST shape: `update` can be string/bool/object.
        if ($this->hasAvailableUpdate($plugin['update'] ?? null)) {
            return true;
        }

        if (isset($plugin['needs_update']) && is_bool($plugin['needs_update']) && $plugin['needs_update']) {
            return true;
        }

        if (isset($plugin['update_version']) && is_string($plugin['update_version']) && trim($plugin['update_version']) !== '') {
            return true;
        }

        $current = isset($plugin['version']) ? (string) $plugin['version'] : '';
        $new = '';
        foreach (['new_version', 'version_latest', 'latest_version'] as $key) {
            if (isset($plugin[$key]) && is_string($plugin[$key]) && trim($plugin[$key]) !== '') {
                $new = (string) $plugin[$key];
                break;
            }
        }

        if ($current !== '' && $new !== '' && version_compare($new, $current, '>')) {
            return true;
        }

        return false;
    }

    /**
     * @param array<string, mixed> $plugin
     */
    private function pluginHasUpdateSignal(array $plugin): bool
    {
        foreach (['update', 'needs_update', 'new_version', 'latest_version', 'version_latest', 'update_version'] as $key) {
            if (array_key_exists($key, $plugin)) {
                return true;
            }
        }

        return false;
    }

    /**
     * @param array<string, mixed> $theme
     */
    private function hasAvailableThemeUpdate(array $theme): bool
    {
        if ($this->hasAvailableUpdate($theme['update'] ?? null)) {
            return true;
        }

        if (isset($theme['needs_update']) && is_bool($theme['needs_update']) && $theme['needs_update']) {
            return true;
        }

        $current = isset($theme['version']) ? (string) $theme['version'] : '';
        $new = '';
        foreach (['new_version', 'version_latest', 'latest_version'] as $key) {
            if (isset($theme[$key]) && is_string($theme[$key]) && trim($theme[$key]) !== '') {
                $new = (string) $theme[$key];
                break;
            }
        }

        if ($current !== '' && $new !== '' && version_compare($new, $current, '>')) {
            return true;
        }

        return false;
    }

    /**
     * @param array<string, mixed> $theme
     */
    private function themeHasUpdateSignal(array $theme): bool
    {
        foreach (['update', 'needs_update', 'new_version', 'latest_version', 'version_latest', 'update_version'] as $key) {
            if (array_key_exists($key, $theme)) {
                return true;
            }
        }

        return false;
    }

    private function countAvailableUpdates(array $payload): int
    {
        // First, try explicit counter-style fields.
        $explicit = $this->extractExplicitUpdateCount($payload);
        if ($explicit !== null) {
            return max(0, $explicit);
        }

        // Then, try plugin/theme containers that often hold update objects.
        $count = 0;
        foreach (['plugins', 'themes'] as $bucket) {
            if (!isset($payload[$bucket]) || !is_array($payload[$bucket])) {
                continue;
            }

            foreach ($payload[$bucket] as $item) {
                if (is_array($item) && $this->hasAvailableUpdate($item)) {
                    $count++;
                }
            }
        }

        // Fallback: generic deep scan on update-related keys.
        if ($count === 0) {
            $stack = [$payload];
            while ($stack !== []) {
                $current = array_pop($stack);
                if (!is_array($current)) {
                    continue;
                }

                foreach ($current as $key => $value) {
                    if (is_array($value)) {
                        $stack[] = $value;
                    }

                    $normalizedKey = strtolower((string) $key);
                    if (!str_contains($normalizedKey, 'update')) {
                        continue;
                    }

                    if ($this->hasAvailableUpdate($value)) {
                        $count++;
                    }
                }
            }
        }

        return max(0, $count);
    }

    private function countUpgradableItems(string $url, string $authUser, string $authSecret): ?int
    {
        $response = $this->transport->getJson($url, $authUser, $authSecret, 15, 1);

        if (!$response['ok'] || !is_array($response['json'])) {
            return null;
        }

        $json = $response['json'];

        if (array_is_list($json)) {
            return count($json);
        }

        foreach (['items', 'plugins', 'themes', 'data'] as $key) {
            if (isset($json[$key]) && is_array($json[$key]) && array_is_list($json[$key])) {
                return count($json[$key]);
            }
        }

        return null;
    }

    private function extractExplicitUpdateCount(array $payload): ?int
    {
        $keys = [
            'updates_available_total',
            'updates_total',
            'total_updates',
            'update_count',
            'available_updates',
        ];

        foreach ($keys as $key) {
            if (isset($payload[$key]) && is_numeric($payload[$key])) {
                return (int) $payload[$key];
            }
        }

        // Common nested form: summary/update totals under data/result.
        foreach (['data', 'result', 'summary'] as $parent) {
            if (!isset($payload[$parent]) || !is_array($payload[$parent])) {
                continue;
            }

            $nested = $this->extractExplicitUpdateCount($payload[$parent]);
            if ($nested !== null) {
                return $nested;
            }
        }

        return null;
    }
}
